/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.parser;

import org.xml.sax.Attributes;
/**
 * Default implementation for Attributes
 * 
 * @see org.xml.sax.Attributes
 */
public class AttributesImpl implements Attributes {
	// The number of elements or the size of the vector.
	int elementCount;
	// Elements
	Attribute[] attributes;	
	// The amount by which the capacity of the vector is increased.
	int capacityIncrement;
	// Different attribute Types
	public static final String CDATA		= "CDATA"; //$NON-NLS-1$
	//public static final String ID			= "ID"; //$NON-NLS-1$
	//public static final String IDREF		= "IDREF"; //$NON-NLS-1$
	//public static final String IDREFS		= "IDREFS"; //$NON-NLS-1$
	//public static final String NMTOKEN		= "NMTOKEN"; //$NON-NLS-1$
	//public static final String NMTOKENS		= "NMTOKENS"; //$NON-NLS-1$
	//public static final String ENTITY		= "ENTITY"; //$NON-NLS-1$
	//public static final String ENTITIES		= "ENTITIES"; //$NON-NLS-1$
	//public static final String NOTATION		= "NOTATION"; //$NON-NLS-1$
	public static final AttributesImpl EmptyList = new AttributesImpl(0);
	class Attribute {
		NSDeclaration nsDeclaration;
		String prefix;
		String qName;
		String localName;
		String type;
		String value;
		private String uri;
		/**
		 * Constructor for Attribute
		 */
		Attribute(NSDeclaration nsDeclaration, String prefix, String qName, String localName, String type, String value) {
			super();
			this.nsDeclaration = nsDeclaration;
			this.prefix = prefix;
			this.qName = qName;
			this.localName = localName;
			this.type = type;
			this.value = value;
			this.uri = null;
		}
		final String getUri() {
			if (uri != null) return uri;
			this.uri = (nsDeclaration == null)?"":nsDeclaration.getUriOnAttributeFor(prefix);
			return uri;
		}
	}
/**
 * Constructor for AttributesImpl
 */
public AttributesImpl() {
	this(20);
}
AttributesImpl(int capacityIncrement) {
	super();
	this.capacityIncrement = capacityIncrement;
	this.elementCount = 0;
	this.attributes = null;
}
/**
 * Return the number of attributes in the list.
 *
 * <p>Once you know the number of attributes, you can iterate
 * through the list.</p>
 *
 * @return The number of attributes in the list.
 * @see #getURI(int)
 * @see #getLocalName(int)
 * @see #getQName(int)
 * @see #getType(int)
 * @see #getValue(int)
 */
public int getLength() {
	return elementCount;
}
/**
 * Look up an attribute's Namespace URI by index.
 *
 * @param index The attribute index (zero-based).
 * @return The Namespace URI, or the empty string if none
 *         is available, or null if the index is out of
 *         range.
 * @see #getLength
 */
public String getURI(int index) {
	try {
		Attribute att = attributes[index];
		if (att.nsDeclaration == null) return "";
		return att.nsDeclaration.getUriOnAttributeFor(att.prefix);
	} catch (IndexOutOfBoundsException e) {
		return null;
	}
}
/**
 * Look up an attribute's local name by index.
 *
 * @param index The attribute index (zero-based).
 * @return The local name, or the empty string if Namespace
 *         processing is not being performed, or null
 *         if the index is out of range.
 * @see #getLength
 */
public String getLocalName(int index) {
	try {
		return attributes[index].localName;
	} catch (IndexOutOfBoundsException e) {
		return null;
	}
}
/**
 * Look up an attribute's XML 1.0 qualified name by index.
 *
 * @param index The attribute index (zero-based).
 * @return The XML 1.0 qualified name, or the empty string
 *         if none is available, or null if the index
 *         is out of range.
 * @see #getLength
 */
public String getQName(int index) {
	try {
		return attributes[index].qName;
	} catch (IndexOutOfBoundsException e) {
		return null;
	}
}
/**
 * Look up an attribute's type by index.
 *
 * <p>The attribute type is one of the strings "CDATA", "ID",
 * "IDREF", "IDREFS", "NMTOKEN", "NMTOKENS", "ENTITY", "ENTITIES",
 * or "NOTATION" (always in upper case).</p>
 *
 * <p>If the parser has not read a declaration for the attribute,
 * or if the parser does not report attribute types, then it must
 * return the value "CDATA" as stated in the XML 1.0 Recommentation
 * (clause 3.3.3, "Attribute-Value Normalization").</p>
 *
 * <p>For an enumerated attribute that is not a notation, the
 * parser will report the type as "NMTOKEN".</p>
 *
 * @param index The attribute index (zero-based).
 * @return The attribute's type as a string, or null if the
 *         index is out of range.
 * @see #getLength
 */
public String getType(int index) {
	try {
		return attributes[index].type;
	} catch (IndexOutOfBoundsException e) {
		return null;
	}
}
/**
 * Look up an attribute's value by index.
 *
 * <p>If the attribute value is a list of tokens (IDREFS,
 * ENTITIES, or NMTOKENS), the tokens will be concatenated
 * into a single string with each token separated by a
 * single space.</p>
 *
 * @param index The attribute index (zero-based).
 * @return The attribute's value as a string, or null if the
 *         index is out of range.
 * @see #getLength
 */
public String getValue(int index) {
	try {
		return attributes[index].value;
	} catch (IndexOutOfBoundsException e) {
		return null;
	}
}

/**
 * Look up the index of an attribute by Namespace name.
 *
 * @param uri The Namespace URI, or the empty string if
 *        the name has no Namespace URI.
 * @param localName The attribute's local name.
 * @return The index of the attribute, or -1 if it does not
 *         appear in the list.
 */
public int getIndex(String uri, String localPart) {
	try {
		Attribute att;
		for (int i = 0; i < elementCount; i++) {
			att = attributes[i];
			if ((att.getUri().equals(uri)) &&
				(att.localName.equals(localPart))) return i;
		}
		return -1;
	} catch (NullPointerException e) {
		// ==> uri and/or localName are null ==> Namespaces are not available
		return -1;
	}
}
/**
 * Look up the index of an attribute by XML 1.0 qualified name.
 *
 * @param qName The qualified (prefixed) name.
 * @return The index of the attribute, or -1 if it does not
 *         appear in the list.
 */
public int getIndex(String qName) {
	for (int i = 0; i < elementCount; i++) {
		if (attributes[i].localName.equals(qName)) return i;
	}
	return -1;
}
/**
 * Look up an attribute's type by Namespace name.
 *
 * <p>See {@link #getType(int) getType(int)} for a description
 * of the possible types.</p>
 *
 * @param uri The Namespace URI, or the empty String if the
 *        name has no Namespace URI.
 * @param localName The local name of the attribute.
 * @return The attribute type as a string, or null if the
 *         attribute is not in the list or if Namespace
 *         processing is not being performed.
 */
public String getType(String uri, String localName) {
	try {
		Attribute att;
		for (int i = 0; i < elementCount; i++) {
			att = attributes[i];
			if ((att.getUri().equals(uri)) &&
				(att.localName.equals(localName))) return att.type;
		}
		return null;
	} catch (NullPointerException e) {
		// ==> uri and/or localName are null ==> Namespaces are not available
		return null;
	}
}
/**
 * Look up an attribute's type by XML 1.0 qualified name.
 *
 * <p>See {@link #getType(int) getType(int)} for a description
 * of the possible types.</p>
 *
 * @param qName The XML 1.0 qualified name.
 * @return The attribute type as a string, or null if the
 *         attribute is not in the list or if qualified names
 *         are not available.
 */
public String getType(String qName) {
	try {
		Attribute att;
		for (int i = 0; i < elementCount; i++) {
			att = attributes[i];
			if (att.localName.equals(qName)) return att.type;
		}
		return null;
	} catch (NullPointerException e) {
		return null;
	}
}
/**
 * Look up an attribute's value by Namespace name.
 *
 * <p>See {@link #getValue(int) getValue(int)} for a description
 * of the possible values.</p>
 *
 * @param uri The Namespace URI, or the empty String if the
 *        name has no Namespace URI.
 * @param localName The local name of the attribute.
 * @return The attribute value as a string, or null if the
 *         attribute is not in the list.
 */
public String getValue(String uri, String localName) {
	try {
		Attribute att;
		for (int i = 0; i < elementCount; i++) {
			att = attributes[i];
			if ((att.getUri().equals(uri)) &&
				(att.localName.equals(localName))) return att.value;
		}
		return null;
	} catch (NullPointerException e) {
		return null;
	}
}
/**
 * Look up an attribute's value by XML 1.0 qualified name.
 *
 * <p>See {@link #getValue(int) getValue(int)} for a description
 * of the possible values.</p>
 *
 * @param qName The XML 1.0 qualified name.
 * @return The attribute value as a string, or null if the
 *         attribute is not in the list or if qualified names
 *         are not available.
 */
public String getValue(String qName) {
	try {
		Attribute att;
		for (int i = 0; i < elementCount; i++) {
			att = attributes[i];
			if (att.localName.equals(qName)) return att.value;
		}
		return null;
	} catch (NullPointerException e) {
		return null;
	}
}
/**
 * If uri and/or localName are null
 * 	==> No namespace currently available
 */
public void addAttribute(NSDeclaration nsDeclaration, String prefix, String qName, String localName, String type, String value) {
	if ((elementCount == 0) ||
		(elementCount == attributes.length)) grow();
	attributes[elementCount++] =
		new Attribute(nsDeclaration, prefix, qName, localName, type, value);
}
private void grow() {
	Attribute[] newAtts = new Attribute[elementCount+capacityIncrement];
	if (attributes != null) {
		System.arraycopy(attributes, 0, newAtts, 0, elementCount);
	}
	attributes = newAtts;
}
}

