/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ercp.xml.parser;
 
import java.io.IOException;
import java.io.InputStream;
import java.util.Hashtable;

import org.eclipse.ercp.xml.io.IOSpecificCall;
import org.eclipse.ercp.xml.io.PropertyFileReader;



/**
 * String Externalization for MLRF
 */
public class EXmlMsg {
	Hashtable strings;
	String prefix;
	static EXmlMsg Default = null;
	
	static final String PropertiesExtension = ".properties";

/**
 * 
 */
public static EXmlMsg getDefault() {
	if (Default != null) return Default;
	Default = new EXmlMsg(EXmlMsg.class, "StringsExternalization", "Err_");	 //$NON-NLS-1$ //$NON-NLS-2$
	return Default;
}
/**
 * Constructor
 */
public EXmlMsg(Class clazz, String resourceName, String prefix) {
	super();
	this.strings = readStrings(clazz, resourceName);
	this.prefix = prefix;
}
protected Hashtable readStrings(Class clazz, String resourceName) {
	String language = IOSpecificCall.getDefault().getSystemProperty("user.language"); //$NON-NLS-1$
	String region = IOSpecificCall.getDefault().getSystemProperty("user.region"); //$NON-NLS-1$
	String variant = IOSpecificCall.getDefault().getSystemProperty("user.variant"); //$NON-NLS-1$
	
	language = (language != null)?language:"en"; //$NON-NLS-1$
	region = (region != null)?region:"US"; //$NON-NLS-1$
	variant = (variant != null)?variant:""; //$NON-NLS-1$
	
	return readStrings(clazz, resourceName, language, region, variant);
}
protected Hashtable readStrings(Class clazz, String resourceName, String language, String region, String variant) {
	String filename;
	InputStream is = null;
	Hashtable strings = new Hashtable();
	try {
		if ((language != null) && (language.length() != 0)) {
			if ((region != null) && (region.length() != 0)) {
				if ((variant != null) && (variant.length() != 0)) {
					filename = resourceName+'_'+language+'_'+region+'_'+variant+PropertiesExtension; //$NON-NLS-1$
					is = clazz.getResourceAsStream(filename);
					if (is != null) {
						strings = new PropertyFileReader().read(is, strings);
						is.close();
					}
				}
				filename = resourceName+'_'+language+'_'+region+PropertiesExtension; //$NON-NLS-1$
				is = clazz.getResourceAsStream(filename);
				if (is != null) {
					strings = new PropertyFileReader().read(is, strings);
					is.close();
				}
			}
			filename = resourceName+'_'+language+PropertiesExtension; //$NON-NLS-1$
			is = clazz.getResourceAsStream(filename);
			if (is != null) {
				strings = new PropertyFileReader().read(is, strings);
				is.close();
			}
		}
		filename = resourceName+PropertiesExtension; //$NON-NLS-1$
		is = clazz.getResourceAsStream(filename);
		if (is != null) {
			strings = new PropertyFileReader().read(is, strings);
			is.close();
		}
	} catch (IOException e) {
		// NOP
	}
	return strings;		
}
String getAlphaKey(int key) {
	return prefix+Integer.toString(key);
}
/**
 * Retrieves a message which has no arguments.
 */
public String getString(int key) {
	return getString(key, new Object[] {});
}
/**
 * Retrieves a message which takes 1 argument.
 */
public String getString (int key, Object arg) {
	return getString(key, new Object[] {arg});
}
/**
 * Retrieves a message which takes 1 integer argument.
 */
public String getString (int key, int arg) {
	return getString(key, new Object[] {Integer.toString(arg)});
}
/**
 * Retrieves a message which takes 1 character argument.
 */
public String getString (int key, char arg) {
	return getString(key, new Object[] {String.valueOf(arg)});
}
/**
 * Retrieves a message which takes several arguments.
 */
public String getString (int key, Object[] args) {
	String alphaKey = getAlphaKey(key);
	String format = (String)(strings.get(alphaKey));
	if (format == null)  format = alphaKey;

	return format(format, args);
}
/**
 * Generates a formatted text string given a source string
 * containing "argument markers" of the form "{argNum}"
 * where each argNum must be in the range 0..9. The result
 * is generated by inserting the toString of each argument
 * into the position indicated in the string.
 * <p>
 * To insert the "{" character into the output, use a single
 * backslash character to escape it (i.e. "\{"). The "}"
 * character does not need to be escaped.
 *
 * @version		initial
 *
 * @param		format String
 *					the format to use when printing.
 * @param		args Object[]
 *					the arguments to use.
 * @return		String
 *					the formatted message.
 */
static private String format (String format, Object[] args) {
	StringBuffer answer = new StringBuffer();
	String[] argStrings = new String[args.length];
	for (int i = 0; i < args.length; ++i) {
		if (args[i] == null)
			argStrings[i] = "<"+Default.getString(Null)+">";
		else
			argStrings[i] = args[i].toString();
	}
	int lastI = 0;
	for (int i = format.indexOf('{', 0); i >= 0; i = format.indexOf('{', lastI)) {
		if (i != 0 && format.charAt(i-1) == '\\') {
			// It's escaped, just print and loop.
			if (i != 1)
				answer.append(format.substring(lastI,i-1));
			answer.append('{');
			lastI = i+1;
		} else {
			// It's a format character.
			if (i > format.length()-3) {
				// Bad format, just print and loop.
				answer.append(format.substring(lastI, format.length()));
				lastI = format.length();
			} else {
				int argnum = (byte) Character.digit(format.charAt(i+1), 10);
				if (argnum < 0 || format.charAt(i+2) != '}') {
					// Bad format, just print and loop.
					answer.append(format.substring(lastI, i+1));
					lastI = i+1;
				} else {
					// Got a good one!
					answer.append(format.substring(lastI, i));
					if (argnum >= argStrings.length)
						answer.append("<"+Default.getString(MissingArgument)+">"); //$NON-NLS-1$
					else
						answer.append(argStrings[argnum]);
					lastI = i + 3;
				}
			}
		}
	}
	if (lastI < format.length())
		answer.append(format.substring(lastI, format.length()));
	return answer.toString();
}
public final static int
					//Default_Error					= 0,	// Error at [{0}, {1}]: {2}
					//Default_Fatal_Error				= 1,	// Fatal Error at [{0}, {1}]: {2}
					//Default_Warning					= 2,	// Warning at [{0}, {1}]: {2}
					Parsing_yet						= 3,	// Parsing yet
					Null							= 4,	// null
					MissingArgument					= 5,	// Missing Argument
					
					ERROR_ATTRIBUT_VALUE_EXPECTED	= 10,	//Attribut Value expected
					//ERROR_ATTRIBUT_NAME_EXPECTED	= 11,	//Attribut name expected
					ERROR_TAG_EXPECTED				= 12,	//Tag expected
					ERROR_TAG_NAME_EXPECTED			= 13,	//Tag name expected
					ERROR_WRONG_TAG_HEADER			= 14,	//Wrong tag header
					ERROR_GT_EXPECTED				= 15,	//'>' expected
					ERROR_LT_EXPECTED				= 16,	//'<' expected
					ERROR_TAG_COMMENT_EXPECTED		= 17,	//XML comment expected
					ERROR_WRONG_PI					= 18,	//Wrong Processing Instruction
					ERROR_EQUAL_EXPECTED			= 19,	//'=' expected
					ERROR_END_OF_TAG_EXPECTED		= 20,	//End of tag expected
					ERROR_END_QUOTE_EXPECTED		= 21,	//'\"' expected
					//ERROR_ILLEGAL_CHARACTER			= 22,	//Illegal character
					ERROR_QUESTIONMARK_EXPECTED		= 23,	//'?' expected
					WARNING_UNKNOWN_CHARACTER_DEF 	= 24,	//Unknown character reference
					WARNING_END_TAG_EXPECTED		= 25,	//End tag expected
					WARNING_UNSUPPORTED_ENCODING	= 26,	//Unsupported encoding
					//ERROR_UNMANAGED_STATE			= 27,	//Unknown state
					//CharacterStream_or_ByteStream_should_be_set_1 = 28,	// CharacterStream or ByteStream should be set
					//W_Ending_Tag_Expected 			= 29,	// </{0}> expected!
					WRONG_DOCUMENT_ERR				= 30,	//Wrong Document
					INUSE_ATTRIBUTE_ERR				= 31,	//Attribute used in another element
					NOT_FOUND_ERR					= 32,	//Node not found
					EndingTagExpected				= 33,	//</{0}>
					NOT_SUPPORTED					= 34,	//Not supported
					DOC_TYPE_CREATED_YET			= 35,	//document type created by another document
					INSERTING_NODE					= 36,	//inserting node
					REPLACING_NODE					= 37,	//replacing node
					REMOVING_NODE					= 38,	//removing node
					APPENDING_NODE					= 39,	//appending node
					COALESCING_NOT_SUPPORTED		= 40,	//Coalescing not supported in eXML
					VALIDATION_NOT_SUPPORTED		= 41,	//XML validation not supported in eXML
					EXPAND_ENTITY_NOT_SUPPORTED		= 42,	//Expand Entity not supported in eXML
					COMMENTS_NOT_SUPPORTED			= 43,	//Comments always ignored in eXML
					WS_IGNORED_NOT_SUPPORTED		= 44,	//Whitespaces are never ignored in eXML
					CANNOT_BE_NULL					= 45,	//{0} cannot be null
					//PARSING_YET						= 46,	//Parsing yet!
					MALFORMED_UTF8_1				= 47,	//Malformed UTF-8 character: 0x{0}
					MALFORMED_UTF8_2				= 48,	//Malformed UTF-8 character: 0x{0} 0x{1}
					MALFORMED_UTF8_3				= 49,	//Malformed UTF-8 character: 0x{0} 0x{1} 0x{2}
					MALFORMED_UTF16BE				= 50,	//Malformed UTF-16BE character: 0x{0}
					MALFORMED_UTF16LE				= 51,	//Malformed UTF-16LE character: 0x{0}
					CS_OR_BS_SET_ERROR				= 52,	//CharacterStream or ByteStream should be set
					SYSTEMID_SET_ERROR				= 53,	//SystemId error
					WARNING_PREFIX_UNKNOWN			= 54,	// Prefix unknown: {0}
					WARNING_START_TAG_EXPECTED		= 55,	// Start tag is missing <{0}>
					WRONG_ENDING_DOC				= 56,	// XML document structures must start and end within the same entity
					ERROR_CDATA						= 57,	// Wrong CDATA declaration
					ERROR_DOCTYPE					= 58,	// Wrong DOCTYPE declaration
					HIERARCHY_REQUEST_ERR			= 59;	// Node hierarchy error
}
