/*******************************************************************************
 * Copyright (c) 2005 Nokia.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Nokia - eSWT Demo application
 *******************************************************************************/
package org.eclipse.ercp.swt.samples.eswtdemo.Calendar;


import org.eclipse.ercp.swt.samples.eswtdemo.Messages;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;

import java.util.*;



public class CalendarMonthWidget  extends Canvas implements PaintListener, KeyListener, FocusListener, DisposeListener, MouseListener {
	
	Date selecteddate;
	Composite parent;
	
	String firstRow[] = {Messages.getString("CalendarMonthWidget.week"),Messages.getString("CalendarMonthWidget.sunday"),Messages.getString("CalendarMonthWidget.monday"),Messages.getString("CalendarMonthWidget.tuesday"),Messages.getString("CalendarMonthWidget.wednesday"),Messages.getString("CalendarMonthWidget.thursday"),Messages.getString("CalendarMonthWidget.friday"),Messages.getString("CalendarMonthWidget.saturday")}; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$ //$NON-NLS-8$
	
	// precomputed images to speed up the drawing of the widget
	Image 		backbufferImage;
	Image 		pictureOfTheMonth;
	Image 		pictureWeekSelected;
	Image 		pictureWeekSelected_nonFocusing;
	Image 		pictureDayOfWeekSelected;
	Image 		pictureDayOfWeekSelected_nonFocusing;
	Image 		pictureDayOfMonthSelected;
	Image 		pictureDayOfMonthSelected_nonFocusing;
	
	Color foregroundColor;
	
	GC	  		backbufferGC;
	GC	  		pictureOfTheMonthGC;
	
	Rectangle 	area;
	
	// size of the cell of the calendar
	int 		cellwidth;
	int			cellheight;

	// listener will get the selection events
	CalendarMonthWidgetListener listener;
	
	// style of cells
	static int 	NONE									=-1;
	static int 	PICTUREWEEKSELECTED						=0;
	static int 	PICTUREWEEKSELECTED_NONFOCUSING			=1;
	static int	PICTUREDAYOFWEEKSELECTED				=2;
	static int 	PICTUREDAYOFWEEKSELECTED_NONFOCUSING	=3;
	static int	PICTUREDAYOFMONTHSELECTED				=4;
	static int	PICTUREDAYOFMONTHSELECTED_NONFOCUSING	=5;	
	
	// meeting database
	CalendarDataBase calendardatabase;
	
	CalendarMonthWidget(Composite parent, CalendarDataBase calendardatabase) {
		super(parent, SWT.NO_BACKGROUND);
		this.calendardatabase = calendardatabase;
		selecteddate = new Date(); // init to now
	    foregroundColor = new Color(getDisplay(), 80, 180, 80);
		this.parent = parent;
		addPaintListener(this);
		addKeyListener(this);
		addMouseListener(this);
		computeArea();
		buildDoubleBufferingGC();
		redrawAllMonth();
		setSelectedDate(selecteddate);
		addFocusListener(this);
		
		// Add a disposeListener to dispose the ressources when the parent are dispose.
		parent.addDisposeListener(this);
	}
	
	public void addCalendarMonthWidgetListener(CalendarMonthWidgetListener l) {
	    listener = l;
	}
	
	public void paintControl(PaintEvent e) {
	    // draw backbuffer image
		e.gc.drawImage(backbufferImage,0,0);
	}

	public Date getSelecteddate() {
		return selecteddate;
	}
	
	private void drawCell(String text, int posx, int posy, GC gc, Rectangle area, int style, boolean meetingthisday) {
		
	    // compute text extent to center the text into the cell
	    Point te = gc.textExtent(text);
		Point delta = new Point((cellwidth-te.x)/2,(cellheight-te.y)/2);

		if (style == PICTUREWEEKSELECTED){
		    if (isFocusControl()) {
		        gc.drawImage(pictureWeekSelected,area.x+(posx*cellwidth),area.y+(posy*cellheight));
		    } else {
		        gc.drawImage(pictureWeekSelected_nonFocusing,area.x+(posx*cellwidth),area.y+(posy*cellheight));
		    }
		}
		else if (style == PICTUREDAYOFWEEKSELECTED){
		    if (isFocusControl()) {
		        gc.drawImage(pictureDayOfWeekSelected,area.x+(posx*cellwidth),area.y+(posy*cellheight));
		    } else {
		        gc.drawImage(pictureDayOfWeekSelected_nonFocusing,area.x+(posx*cellwidth),area.y+(posy*cellheight));
		    }
		} 
		else if (style == PICTUREDAYOFMONTHSELECTED){
		    if (isFocusControl()) {
		        gc.drawImage(pictureDayOfMonthSelected,area.x+(posx*cellwidth),area.y+(posy*cellheight));
		    } else {
		        gc.drawImage(pictureDayOfMonthSelected_nonFocusing,area.x+(posx*cellwidth),area.y+(posy*cellheight));
		    }		        
		} 
		
		// text is in green if we have a meeting this day
		if (meetingthisday) {
		    gc.setForeground(foregroundColor);
		}		
		else {
//		    gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_LIST_SELECTION_TEXT ));
			gc.setForeground(new Color(getDisplay(), 0, 0, 0));
		}
		
		// draw the text.
		gc.drawText(text,area.x+(posx*cellwidth)+delta.x,area.y+(posy*cellheight)+delta.y,SWT.DRAW_TRANSPARENT);
		
	}

	private void drawCellBackground(GC gc, int style) {
	    
	    // draw the background with the corresponding system color
	    if ((style==PICTUREWEEKSELECTED) ||
	        (style==PICTUREDAYOFWEEKSELECTED) ||
	        (style==PICTUREDAYOFMONTHSELECTED)
	        )    {
	        gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_LIST_SELECTION ));
	    }
	    else if ((style==PICTUREWEEKSELECTED_NONFOCUSING) ||
		        (style==PICTUREDAYOFWEEKSELECTED_NONFOCUSING) ||
		        (style==PICTUREDAYOFMONTHSELECTED_NONFOCUSING)
		        ) {
	        gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND ));
	    }
	    Rectangle cellArea = gc.getClipping();
	    gc.fillRectangle(cellArea);
	    
		Point topleft = new Point(0,0);
		Point bottomright = new Point(cellArea.width-1,cellArea.height-1);

		gc.setLineStyle(1);
	    gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW ));
		gc.drawLine(bottomright.x,topleft.y,bottomright.x,bottomright.y);
		gc.drawLine(topleft.x,bottomright.y,bottomright.x,bottomright.y);
	    gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_LIGHT_SHADOW));
	    gc.drawLine(topleft.x,topleft.y,topleft.x,topleft.y+cellheight-2);
		gc.drawLine(topleft.x,topleft.y,topleft.x+cellwidth-2,topleft.y);

	    // draw a focus rectangle with the system color
		if ((style==PICTUREDAYOFMONTHSELECTED) ||
		        (style==PICTUREDAYOFMONTHSELECTED_NONFOCUSING) 
		        )  
	    {
		    gc.setLineStyle(1);
//		    gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_BORDER ));
		    gc.setForeground(new Color(getDisplay(), 0, 255, 255));
			gc.drawRectangle(topleft.x,topleft.y,bottomright.x-1,bottomright.y-1);
		}

		
	}

	
	public void setSelectedDate(Date date) {

		GregorianCalendar g= new GregorianCalendar();
		g.setFirstDayOfWeek(Calendar.SUNDAY);
		
		// get month and year of the new date
		g.setTime(date);
		int selectedMonth = g.get(Calendar.MONTH);
		int selectedYear = g.get(Calendar.YEAR);
		
		// get month and year of the current seleted date
		g.setTime(selecteddate);
		int currentMonth = g.get(Calendar.MONTH);
		int currentYear = g.get(Calendar.YEAR);
		
		// check if the month has changed
		selecteddate = date;
		if ((currentYear!=selectedYear) || (currentMonth!=selectedMonth)) {
			// redraw all month
			redrawAllMonth();
		}
		
		// erase selection, drawing the month's picture
		// this is faster than drawing 3 little pictures
	    backbufferGC.drawImage(pictureOfTheMonth,0,0);
	    
	    // draw the newly selected date
	    drawDaySelection(selecteddate);

		// notify listener
	    if (listener!=null) {
		    listener.newDateSelected(date);
		}
	    redraw();
	}

	
	/**
	 * Key handling
	 */
	public void keyPressed(KeyEvent e) {
		GregorianCalendar c=new GregorianCalendar();
		c.setFirstDayOfWeek(Calendar.SUNDAY);
		c.setTime(selecteddate);
		if (e.keyCode==SWT.ARROW_RIGHT) {
			c.add(Calendar.DAY_OF_YEAR,1);
			Date date = c.getTime();
			setSelectedDate(date);
		}
		else if (e.keyCode==SWT.ARROW_LEFT) {
			c.add(Calendar.DAY_OF_YEAR,-1);
			Date date = c.getTime();
			setSelectedDate(date);
		} 
		else if (e.keyCode==SWT.ARROW_DOWN) {
			c.add(Calendar.WEEK_OF_MONTH,1);
			Date date = c.getTime();
			setSelectedDate(date);
		} 
		else if (e.keyCode==SWT.ARROW_UP) {
			c.add(Calendar.WEEK_OF_MONTH,-1);
			Date date = c.getTime();
			setSelectedDate(date);
		} 
	}
	
	public void keyReleased(KeyEvent e) {
		
	}


	/**
	 * mouse handling
	 */
	public void mouseDown(MouseEvent e) {
		int cellX = e.x / cellwidth;
		int cellY = e.y / cellheight;
				
		// hit test cells
		GregorianCalendar g = new GregorianCalendar();
		g.setFirstDayOfWeek(Calendar.SUNDAY);
		g.setTime(selecteddate);

		for (int i=1;i<=g.getActualMaximum(Calendar.DAY_OF_MONTH);i++) {
			g.setTime(selecteddate);
			g.set(Calendar.DAY_OF_MONTH,i);
			if (cellX == g.get(Calendar.DAY_OF_WEEK) && 
			    cellY == g.get(Calendar.WEEK_OF_MONTH)) {
				setSelectedDate(g.getTime());
				break;
			}
		}
	}
	
	public void mouseUp(MouseEvent e) {}
	public void mouseDoubleClick(MouseEvent e) {}

	
	/**
	 * build double buffering images,
	 * also build background cell images
	 *
	 */
	private void buildDoubleBufferingGC() {
			if (backbufferImage!=null)
				backbufferImage.dispose();
			if (backbufferGC!=null)
				backbufferGC.dispose();
			backbufferImage = new Image(parent.getDisplay(),parent.getClientArea());
			backbufferGC = new GC(backbufferImage);
			
			pictureOfTheMonth = new Image(parent.getDisplay(),parent.getClientArea());
			pictureOfTheMonthGC = new GC(pictureOfTheMonth);
			
			GC gc;
			
			pictureWeekSelected = new Image(parent.getDisplay(),cellwidth,cellheight);
			gc= new GC(pictureWeekSelected);
			drawCellBackground(gc,PICTUREWEEKSELECTED);
			gc.dispose();
			
			pictureWeekSelected_nonFocusing = new Image(parent.getDisplay(),cellwidth,cellheight);
			gc = new GC(pictureWeekSelected_nonFocusing);
			drawCellBackground(gc,PICTUREWEEKSELECTED_NONFOCUSING);
			gc.dispose();

			pictureDayOfWeekSelected = new Image(parent.getDisplay(),cellwidth,cellheight);
			gc = new GC(pictureDayOfWeekSelected);
			drawCellBackground(gc,PICTUREDAYOFWEEKSELECTED);
			gc.dispose();
			
			pictureDayOfWeekSelected_nonFocusing = new Image(parent.getDisplay(),cellwidth,cellheight);
			gc = new GC(pictureDayOfWeekSelected_nonFocusing);
			drawCellBackground(gc,PICTUREDAYOFWEEKSELECTED_NONFOCUSING);
			gc.dispose();			

			pictureDayOfMonthSelected = new Image(parent.getDisplay(),cellwidth,cellheight);
			gc = new GC(pictureDayOfMonthSelected);
			drawCellBackground(gc,PICTUREDAYOFMONTHSELECTED);	
			gc.dispose();		

			pictureDayOfMonthSelected_nonFocusing = new Image(parent.getDisplay(),cellwidth,cellheight);
			gc = new GC(pictureDayOfMonthSelected_nonFocusing);
			drawCellBackground(gc,PICTUREDAYOFMONTHSELECTED_NONFOCUSING);	
			gc.dispose();		

	}
	
	
	/**
	 *  Draw a slected day of the month
	 */
	private void drawDaySelection(Date date) {
		// draw unselected cells
		GregorianCalendar selectedDate = new GregorianCalendar();
		selectedDate.setFirstDayOfWeek(Calendar.SUNDAY);
		selectedDate.setTime(date);
		GregorianCalendar g = new GregorianCalendar();
		g.setFirstDayOfWeek(Calendar.SUNDAY);
		g.setTime(date);
		
		// draw selected weeks of the month
		g.setTime(date); // we play with weeks, so we can jump from one month to another. therefore we need to reset here 
		Integer integer = new Integer(g.get(Calendar.WEEK_OF_YEAR));
		int weekofmonth=g.get(Calendar.WEEK_OF_MONTH);
		drawCell(integer.toString(),0,weekofmonth,backbufferGC,area,PICTUREWEEKSELECTED,false);
		
		// draw selected day of the week
		int dayofweek = g.get(Calendar.DAY_OF_WEEK);
		drawCell(firstRow[dayofweek],dayofweek,0,backbufferGC,area,PICTUREDAYOFWEEKSELECTED,false);

		// draw selected day
		drawCell((new Integer(g.get(Calendar.DAY_OF_MONTH))).toString(),dayofweek,weekofmonth,backbufferGC,area,PICTUREDAYOFMONTHSELECTED,calendardatabase.meetingThisDay(selecteddate));
		
		
	}
	
	
	/**
	 * This update the whole calendar. draw into a backbuffer image.
	 * one column for the week number
	 * one row for the day of the month
	 * metting day in green.
	 * 
	 * use system color to fit device's skin
	 */
	public void redrawAllMonth() {

		// clean first
		pictureOfTheMonthGC.setBackground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		pictureOfTheMonthGC.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
		pictureOfTheMonthGC.fillRectangle(area);
		

		if (parent.isFocusControl()) {
		    pictureOfTheMonthGC.setLineWidth(2);
		}
		else {
		    pictureOfTheMonthGC.setLineWidth(1);    
		}
		pictureOfTheMonthGC.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND ));

				
		// horizontal line
		pictureOfTheMonthGC.setLineWidth(1);
		for (int i=1;i<7;i++)	{
		    pictureOfTheMonthGC.drawLine(area.x+cellwidth,area.y+(i*cellheight),area.x+area.width,area.y+(i*cellheight));
			}

		// vertical line
		pictureOfTheMonthGC.drawLine(area.x+cellwidth,area.y+cellheight,area.x+cellwidth,area.y+area.height);

		
		// draw texts
		// find best font for the size we need
		for (int i=0; i<8;i++) {
			drawCell(firstRow[i],i,0,pictureOfTheMonthGC,area,NONE,false);
		}

		GregorianCalendar selectedDate = new GregorianCalendar();
		selectedDate.setFirstDayOfWeek(Calendar.SUNDAY);
		selectedDate.setTime(selecteddate);
		GregorianCalendar g = new GregorianCalendar();
		g.setFirstDayOfWeek(Calendar.SUNDAY);
		
		// draw weeks of the month
		// TODO optisation, using calendar is very slow
		g.setTime(selecteddate);
		for (int i=1;i<=g.getActualMaximum(Calendar.WEEK_OF_MONTH);i++) {
			g.setTime(selecteddate); // we play with weeks, so we can jump from one month to another. therefore we need to reset here 
			g.set(Calendar.WEEK_OF_MONTH,i);
			Integer integer = new Integer(g.get(Calendar.WEEK_OF_YEAR));
			drawCell(integer.toString(),0,i,pictureOfTheMonthGC,area,NONE,false);
		}
		
		// fill the days
		// TODO optisation, using calendar is very slow
		g.setTime(selecteddate); 
		for (int i=1;i<=g.getActualMaximum(Calendar.DAY_OF_MONTH);i++) {
			g.setTime(selecteddate);
			g.set(Calendar.DAY_OF_MONTH,i);
			int celltodrawx = g.get(Calendar.DAY_OF_WEEK); 
			int celltodrawy = g.get(Calendar.WEEK_OF_MONTH);
			Integer integer = new Integer(i);
			drawCell(integer.toString(),celltodrawx,celltodrawy,pictureOfTheMonthGC,area,NONE,calendardatabase.meetingThisDay(g.getTime()));
		}
		
		drawDaySelection(selecteddate);
	}
	
	private void computeArea() {

		area = parent.getClientArea();
		
		// we need 8 column
		// and 7 row
		area.width=area.width-area.width%8;
		area.height=area.height-area.height%7;
		
		// now we need square cells in the table, so take the min one
		//int minsize=Math.min(area.width/8,area.height/7);
		cellwidth = area.width/8;
		cellheight = area.height/7;
		
		// recompute the area
		area.width = cellwidth * 8;
		area.height = cellheight * 7;	
	}
	
	public void focusGained(FocusEvent e) {
	    // redraw selection
	    drawDaySelection(selecteddate);
	}

	public void focusLost(FocusEvent e) {
	    // redraw selection
	    drawDaySelection(selecteddate);
	}


	void releaseRessource() {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.DisposeListener#widgetDisposed(org.eclipse.swt.events.DisposeEvent)
	 */
	public void widgetDisposed(DisposeEvent arg0) {
		if (backbufferImage != null) {
			backbufferImage.dispose();
		}
		if (pictureOfTheMonth != null) {
			pictureOfTheMonth.dispose();
		}
		if (pictureWeekSelected != null) {
			pictureWeekSelected.dispose();
		}
		if (pictureWeekSelected_nonFocusing != null) {
			pictureWeekSelected_nonFocusing.dispose();
		}
		if (pictureDayOfWeekSelected != null) {
			pictureDayOfWeekSelected.dispose();
		}
		if (pictureDayOfWeekSelected_nonFocusing != null) {
			pictureDayOfWeekSelected_nonFocusing.dispose();
		}
		if (pictureDayOfMonthSelected != null) {
			pictureDayOfMonthSelected.dispose();
		}
		if (pictureDayOfMonthSelected_nonFocusing != null) {
			pictureDayOfMonthSelected_nonFocusing.dispose();
		}

		if (backbufferGC != null) {
			backbufferGC.dispose();
		}
		if (pictureOfTheMonthGC != null) {
			pictureOfTheMonthGC.dispose();
		}
		
		if (foregroundColor != null) {
		    foregroundColor.dispose();
		}
	}	
}
