package org.eclipse.jface.preference;

import java.io.IOException;
import java.util.Iterator;

import org.eclipse.ercp.swt.mobile.Command;
import org.eclipse.ercp.swt.mobile.MultiPageDialog;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.util.Assert;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.SWT;


/**
* A preference dialog is a presentation of preference pages. Each
* page is represented by a node that may be selected from within the dialog.
* When a node is selected, the corresponding page is shown.
*/
public class PreferenceDialog extends MultiPageDialog 
	implements IPreferencePageContainer {

	protected PreferenceManager preferenceManager;
	private Command okCommand = null;
	private Command cancelCommand = null;
	private boolean bDialogClose;

	//The id of the root node to display
	private static String lastPreferenceId = null;
	
	/**
	 * Preference store, initially <code>null</code> meaning none.
	 * 
	 * @see #setPreferenceStore
	 */
	private IPreferenceStore preferenceStore;

	/**
	 * Construct PreferenceDialog
	 * 
	 * @param parentShell
	 */
	public PreferenceDialog(Shell parentShell) {
		super(parentShell, SWT.SYSTEM_MODAL);
	}
	
	/**
	 * Construct PreferenceDialog and set its PreferenceManager
	 * 
	 * @param parentShell
	 * @param manager
	 */
	public PreferenceDialog(Shell parentShell, PreferenceManager manager) {
		super(parentShell, SWT.SYSTEM_MODAL);
		preferenceManager = manager;
	}

	/**
	 * Override checkSubclass so that Mobile Extensions impl does
	 * not complain about sublcassing MultiPageDialog
	 */
	protected void checkSubclass() {
	}
	
	/**
	 * Create all the pages and select one if directed
	 * 
	 * @param parent
	 * @return control parentDialog
	 */
	protected Control createContents(final PreferenceDialog parent) {
	
	    Composite parentDialog = null;

		Iterator iterator = preferenceManager.
			getElements(PreferenceManager.PRE_ORDER).iterator();
		while (iterator.hasNext()) {
			IPreferenceNode next = (IPreferenceNode) iterator.next();
			parentDialog = createPage(next);
		}
		return parentDialog;
	}
	
	protected Composite createSubPages(final IPreferenceNode node) {
		Composite control = createPage(node);
		
	    IPreferenceNode[] subnodes = node.getSubNodes();
	    if (subnodes != null) {
	    	for (int i = 0; i < subnodes.length; i++) {
	    		control = createSubPages(subnodes[i]);
			}
		}
		return control;
	}	
	
	protected Composite createPage(final IPreferenceNode node) {
		Composite control = null;
		
		PreferencePage page = (PreferencePage)node.getPage();
		if (page == null) {
			node.createPage();
			page = (PreferencePage)node.getPage();
		}	
		if (page != null) {
			page.setContainer(this);
			Composite mpdPage = createPage(page.getTitle(), page.getImage());
			createPageControl(page, mpdPage);
			control = mpdPage.getParent();
		}	
		return control;
	}

	
	/**
	 * Create the page control for the supplied page.
	 * 
	 * @param page - the preference page to be shown
	 * @param parent - the composite to parent the page
	 * 
	 * @since 3.1
	 */
	protected void createPageControl(PreferencePage page, Composite parent) {
		page.createControl(parent);
	}
	
	
	/** 
	 * Override MultiPageDialog open in order to get pages created
	 */
	public void open() {
		setText(JFaceResources.getString("PreferenceDialog.title")); //$NON-NLS-1$
		Control c = createContents(this);
		createCommands(c);
		
		/* if neither cancel nor OK has been pressed then dialog has
		 * been closed by closing the dialog shell. This should be
		 * treated as an "ok".
		 */
		bDialogClose = true;
		super.open();
		if (bDialogClose)
			cancelPressed();
	}

	/**
	 *  Override MultiPageDialog close in order to clean up commands
	 */
	public void close() {

		if (okCommand != null) {
            okCommand.dispose();
            okCommand = null;
        }
		if (cancelCommand != null) {
			cancelCommand.dispose();
			cancelCommand = null;
        }
		
		// close the MPD only if not already closed
		if (bDialogClose != true)
			super.close();
	}
 
	private void createCommands(Control c){
		if(okCommand == null) {
			okCommand = new Command(c, Command.OK, 1);
			okCommand.setText(JFaceResources.getString("ok")); //$NON-NLS-1$
			okCommand.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent event) {
					bDialogClose = false;
					okPressed();
				}
					public void widgetDefaultSelected(SelectionEvent event) {
					bDialogClose = false;
					okPressed();
				}
			});
		}
		if (cancelCommand == null) {
			cancelCommand = new Command(c, Command.CANCEL, 1);
			cancelCommand.setText(JFaceResources.getString("cancel")); //$NON-NLS-1$
			cancelCommand.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent event) {
					bDialogClose = false;
					cancelPressed();
				}
	
				public void widgetDefaultSelected(SelectionEvent event) {
					bDialogClose = false;
					cancelPressed();
				}
			});
		}
	}
	
	/**
	 * The preference dialog implementation of this <code>Dialog</code>
	 * framework method sends <code>performOk</code> to all Nodes of the
	 * preference dialog, then calls <code>handleSave</code> on this dialog to
	 * save any state, and then calls <code>close</code> to close this dialog.
	 */
	protected void okPressed() {
	
		SafeRunnable.run(new SafeRunnable() {
			
			private boolean errorOccurred;

			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.core.runtime.ISafeRunnable#run()
			 */
			public void run() {
				errorOccurred = false;
				boolean hasFailedOK = false;
				try {
					// Notify all the pages and give them a chance to abort
					Iterator nodes = preferenceManager.
						getElements(PreferenceManager.PRE_ORDER).iterator();
					while (nodes.hasNext()) {
						PreferenceNode next = (PreferenceNode) nodes.next();
						PreferencePage page= (PreferencePage) next.getPage();
						if (page != null ) {
							if (!page.performOk()){
								hasFailedOK = true;
								return;
							}
							//Since all pages are browsed to save besides current open pages, it's good for Jface design. However, it's not suitable to eJface design, 
							//because only current open page needs to be saved. The fix is to set current pages as null after sucessful saving.  
							next.setPage(null);
						}
					}
				} catch (Exception e) {
					errorOccurred = true;
					handleException(e);
				} finally {
					//If no page says to ignore ok and no errors then
					//Give subclasses the choice to save the state of the
				    //preference pages.
					if (!hasFailedOK && !errorOccurred)
						handleSave();
					
					//If no page says to ignore ok or dialog closed then 
					//close preference pages
					if (!hasFailedOK || bDialogClose)
						close();
					else
						//reset flag to true since dialog has not been closed yet
						bDialogClose = true;
				}
			}
		});
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.Dialog#cancelPressed()
	 */
	protected void cancelPressed() {
		
		// Inform all pages that we are cancelling
		SafeRunnable.run(new SafeRunnable() {
			public void run() {
				Iterator nodes = preferenceManager.
					getElements(PreferenceManager.PRE_ORDER).iterator();
				while (nodes.hasNext()) {
					PreferenceNode next = (PreferenceNode) nodes.next();
					if( next == null) 
						return;
					PreferencePage page= (PreferencePage) next.getPage();
					if (page != null) {
						if (!page.performCancel())
							return;
						next.setPage(null);
					}
				}
				close();
			}
		});
	}
	
	/**
	 * Sets the preference store for this preference dialog.
	 * 
	 * @param store
	 *            the preference store
	 * @see #getPreferenceStore
	 */
	public void setPreferenceStore(IPreferenceStore store) {
		Assert.isNotNull(store);
		preferenceStore = store;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.preference.IPreferencePageContainer#getPreferenceStore()
	 */
	public IPreferenceStore getPreferenceStore() {
		return preferenceStore;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.preference.IPreferencePageContainer#updateButtons()
	 */
	public void updateButtons() {
		System.out.println("PreferenceDialog: updateButtons not impl"); //$NON-NLS-1$
//		okButton.setEnabled(isCurrentPageValid());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.preference.IPreferencePageContainer#updateMessage()
	 */
	public void updateMessage() {
		System.out.println("PreferenceDialog: updateMessage not impl"); //$NON-NLS-1$
/*		String message = null;
		String errorMessage = null;
		if(currentPage != null){
			message = currentPage.getMessage();
			errorMessage = currentPage.getErrorMessage();
		}
		int messageType = IMessageProvider.NONE;
		if (message != null && currentPage instanceof IMessageProvider)
			messageType = ((IMessageProvider) currentPage).getMessageType();

		if (errorMessage == null){
			if (showingError) {
				// we were previously showing an error
				showingError = false;
			}
		}
		else {
			message = errorMessage;
			messageType = IMessageProvider.ERROR;
			if (!showingError) {
				// we were not previously showing an error
				showingError = true;
			}
		}  
		messageArea.updateText(message,messageType);
*/
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.preference.IPreferencePageContainer#updateTitle()
	 */
	public void updateTitle() {
		System.out.println("PreferenceDialog: updateTitle not impl"); //$NON-NLS-1$
/*		if(currentPage == null)
			return;
		messageArea.showTitle(currentPage.getTitle(), currentPage.getImage());
*/
	}

	/**
	 * Save the values specified in the pages.
	 * <p>
	 * The default implementation of this framework method saves all pages of
	 * type <code>PreferencePage</code> (if their store needs saving and is a
	 * <code>PreferenceStore</code>).
	 * </p>
	 * <p>
	 * Subclasses may override.
	 * </p>
	 */
	protected void handleSave() {
		Iterator nodes = preferenceManager.getElements(PreferenceManager.PRE_ORDER).iterator();
		while (nodes.hasNext()) {
			IPreferenceNode node = (IPreferenceNode) nodes.next();
			IPreferencePage page = node.getPage();
			if (page instanceof PreferencePage) {
				// Save now in case tbe workbench does not shutdown cleanly
				IPreferenceStore store = ((PreferencePage) page).getPreferenceStore();
				if (store != null && store.needsSaving()
						&& store instanceof IPersistentPreferenceStore) {
					try {
						((IPersistentPreferenceStore) store).save();
					} catch (IOException e) {
						MessageBox mb = new MessageBox((Shell) null, SWT.ICON_ERROR);
						mb.setText(JFaceResources.getString("PreferenceDialog.saveErrorTitle")); //$NON-NLS-1$
						mb.setMessage(JFaceResources.format("PreferenceDialog.saveErrorMessage", new Object[] { page.getTitle(), e.getMessage() })); //$NON-NLS-1$
						mb.open();
					}
				}
			}
		}
	}
	
	/**
	 * Sets the name of the selected item preference. Public equivalent to
	 * <code>setSelectedNodePreference</code>.
	 * 
	 * @param pageId
	 *            The identifier for the page
	 * @since 3.0
	 */
	public void setSelectedNode(String pageId) {
		lastPreferenceId = pageId;
	}

}
