/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jface.preference;

import java.util.ArrayList;
import java.util.StringTokenizer;
import java.util.List;

import org.eclipse.jface.util.Assert;


/**
 * A preference manager maintains a hierarchy of preference nodes and
 * associated preference pages.
 */
public class PreferenceManager {

	/**
     * Pre-order traversal means visit the root first,
     * then the children.
     */
    public static final int PRE_ORDER = 0;

    /**
     * Post-order means visit the children, and then the root.
     */
    public static final int POST_ORDER = 1;

    /**
     * The root node.
     * Note that the root node is a special internal node
     * that is used to collect together all the nodes that
     * have no parent; it is not given out to clients.
     */
    PreferenceNode root = new PreferenceNode(""); //$NON-NLS-1$

    /**
     * The path separator character.
     */
    String separator = "."; //$NON-NLS-1$

    /**
     * Returns the root node.
     * Note that the root node is a special internal node
     * that is used to collect together all the nodes that
     * have no parent; it is not given out to clients.
     *
     * @return the root node
     */
    protected IPreferenceNode getRoot() {
        return root;
    }
    
	
    /**
     * Adds the given preference node as a subnode of the
     * root.
     *
     * @param node the node to add, which must implement 
     *   <code>IPreferenceNode</code>
     */
    public void addToRoot(IPreferenceNode node) {
        Assert.isNotNull(node);
        root.add(node);
    }
    
    /**
     * Adds the given preference node as a subnode of the
     * node at the given path.
     *
     * @param path the path
     * @param node the node to add
     * @return <code>true</code> if the add was successful,
     *  and <code>false</code> if there is no contribution at
     *  the given path
     */
    public boolean addTo(String path, IPreferenceNode node) {
        IPreferenceNode target = find(path);
        if (target == null)
            return false;
        target.add(node);
        return true;
    }
  
    /**
     * Removes the given prefreence node if it is managed by
     * this contribution manager.
     *
     * @param node the node to remove
     * @return <code>true</code> if the node was removed,
     *  and <code>false</code> otherwise
     */
    public boolean remove(IPreferenceNode node) {
        Assert.isNotNull(node);

        return root.remove(node);
    }

    /**
     * Removes all contribution nodes known to this manager.
     */
    public void removeAll() {
        root = new PreferenceNode(""); //$NON-NLS-1$
    }

    /**
     * Recursively enumerates all nodes at or below the given node
     * and adds them to the given list in the given order.
     * 
     * @param node the starting node
     * @param sequence a read-write list of preference nodes
     *  (element type: <code>IPreferenceNode</code>)
     *  in the given order
     * @param order the traversal order, one of 
     *	<code>PRE_ORDER</code> and <code>POST_ORDER</code>
     */
    protected void buildSequence(IPreferenceNode node, List sequence, int order) {
        if (order == PRE_ORDER)
            sequence.add(node);
        IPreferenceNode[] subnodes = node.getSubNodes();
        for (int i = 0; i < subnodes.length; i++) {
            buildSequence(subnodes[i], sequence, order);
        }
        if (order == POST_ORDER)
            sequence.add(node);
    }
    
    /**
     * Returns all preference nodes managed by this
     * manager.
     *
     * @param order the traversal order, one of 
     *	<code>PRE_ORDER</code> and <code>POST_ORDER</code>
     * @return a list of preference nodes
     *  (element type: <code>IPreferenceNode</code>)
     *  in the given order
     */
    public List getElements(int order) {
        Assert.isTrue(order == PRE_ORDER || order == POST_ORDER,
                "invalid traversal order");//$NON-NLS-1$
        ArrayList sequence = new ArrayList();
        IPreferenceNode[] subnodes = getRoot().getSubNodes();
        for (int i = 0; i < subnodes.length; i++)
            buildSequence(subnodes[i], sequence, order);
        return sequence;
    }

    /**
     * Returns root preference nodes managed by this
     * manager.
     *
     *  @return a list of preference nodes
     *  (element type: <code>IPreferenceNode</code>)
     */
    public List getRootElements() {
        
        ArrayList sequence = new ArrayList();
        IPreferenceNode[] subnodes = getRoot().getSubNodes();
        for (int i = 0; i < subnodes.length; i++)
        	sequence.add(subnodes[i]);
        return sequence;
    }

    /**
     * Finds and returns the contribution node at the given path.
     *
     * @param path the path
     * @return the node, or <code>null</code> if none
     */
    public IPreferenceNode find(String path) {
       return find(path,root);
    }
    
    /**
     * Finds and returns the preference node directly
     * below the top at the given path.
     *
     * @param path the path
     * @return the node, or <code>null</code> if none
     * 
     * @since 3.1
     */
    protected IPreferenceNode find(String path,IPreferenceNode top){
    	 Assert.isNotNull(path);
         StringTokenizer stok = new StringTokenizer(path, separator);
         IPreferenceNode node = top;
         while (stok.hasMoreTokens()) {
             String id = stok.nextToken();
             node = node.findSubNode(id);
             if (node == null)
                 return null;
         }
         if (node == top)
             return null;
         return node;
    }

}
