/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial implementation
 *******************************************************************************/
package org.eclipse.ercp.eswt.sample;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.log.LogService;

/**
 * A BundleActivator that sends a log message to
 * the standard LogService when started and stopped.
 * <p>
 * If the LogService is not available then the message
 * is written to <code>System.out</code> instead.
 */
public class SampleBundleActivator extends Object implements BundleActivator {

	/** The context under which this bundle executes. */
	private BundleContext bc;

	/** The name of the LogService */
	static final String logServiceName = LogService.class.getName();

	/** Default constructor */
	public SampleBundleActivator() {
		super();
	}

	/**
	 * Starts the bundle.
	 *
	 * @param bc the BundleContext under which this bundle will execute
	 */
	public void start(BundleContext bc) throws Exception {
		this.bc = bc;

		String message = getBundleName() + Msg.getString("Hello"); //$NON-NLS-1$

		if (!log(LogService.LOG_INFO, message, null)) {
			print(LogService.LOG_INFO, message, null);
		}
		
		Thread t = new Thread(new Test());
		t.start();
	}

	/**
	 * Stops the bundle.
	 *
	 * @param bc the BundleContext under which this bundle executes
	 */
	public void stop(BundleContext bc) throws Exception {
		String message = getBundleName() + Msg.getString("Bye"); //$NON-NLS-1$

		if (!log(LogService.LOG_INFO, message, null)) {
			print(LogService.LOG_INFO, message, null);
		}

		this.bc = null;
	}

	/**
	 * Records an event in the LogService.
	 *
	 * @param severity the integer severity code
	 * @param message a java.lang.String containing the message to be recorded in the log
	 * @param e the java.lang.Throwable which triggered the event being recorded
	 * 
	 * @return <code>true</code> if the message could be logged, <code>false</code> otherwise
	 */
	protected boolean log(int severity, String message, Throwable e) {
	
		// Get the reference to the LogService
		ServiceReference logRef = bc.getServiceReference(logServiceName);
		if (logRef != null) {
			// Get the LogService object
			LogService log = (LogService) bc.getService(logRef);
			if (log != null) {
				try {
					// Write to the LogService
					log.log(severity, message, e);
					return true;
				} finally {
					// Release the reference to the LogService
					bc.ungetService(logRef);
				}
			}
		}
		return false;
	}

	/**
	 * Prints the message to System.out or System.err if the severity is an Error.
	 *
	 * @param severity the integer severity code
	 * @param message a java.lang.String containing the message to be recorded in the log
	 * @param e the java.lang.Throwable which triggered the event being recorded
	 */
	protected void print(int severity, String message, Throwable e) {

		switch (severity) {
			case LogService.LOG_ERROR :
				System.err.println(message);
				break;
			case LogService.LOG_WARNING :
				System.out.println(message);
				break;
			case LogService.LOG_INFO :
				System.out.println(message);
				break;
			default :
		}

		if (e != null) {
			e.printStackTrace(System.err);
		}
	}

	/**
	 * Returns the name of this bundle as defined in the bundle's
	 * Manifest file or <code>null</code> if the bundle name is not
	 * defined or the bundle is not started or does not have the
	 * AdminPermission when run in a secure OSGi framework.
	 */
	public String getBundleName() {
		try {
			return (String) bc.getBundle().getHeaders().get("Bundle-Name"); //$NON-NLS-1$
		} catch (NullPointerException e) {
		} catch (IllegalStateException e) {
			// The bundle has been stopped
		} catch (SecurityException e) {
			// The bundle does not have the permission to read the headers
		}
		return null;
	}
}