//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2021 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.common.svg;

import static org.eclipse.escet.common.java.Maps.map;
import static org.eclipse.escet.common.java.Sets.set;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

/** Scalable Vector Graphics (SVG) name related utility methods. */
public class SvgNameUtils {
    /** XML 'NameStartChar' regular expression. See {@link #isValidSvgName}. */
    private static final String NAME_START_CHAR_RE = "[:A-Z_a-z]";

    /** XML 'NameChar' regular expression. See {@link #isValidSvgName}. */
    private static final String NAME_CHAR_RE = "[:A-Z_a-z\\-.0-9]";

    /** XML 'Name' regular expression. See {@link #isValidSvgName}. */
    private static final Pattern NAME_PATTERN = Pattern.compile(fmt("%s(%s)*", NAME_START_CHAR_RE, NAME_CHAR_RE));

    /** Constructor for the {@link SvgNameUtils} class. */
    private SvgNameUtils() {
        // Static class.
    }

    /**
     * Mapping from SVG element names to the attributes that are allowed to be defined on them. Both SVG presentation
     * attributes and CSS style attributes are included.
     *
     * <p>
     * This information can be used to prevent accidentally setting attributes on the wrong elements, setting attributes
     * that have no effect on that particular element, typos in attribute names, etc.
     * </p>
     *
     * <p>
     * This information is taken from the SVG specification, 'SVG 1.1 (Second Edition) - 16 August 2011', appendices L,
     * M, and N.
     * </p>
     *
     * <p>
     * The SVG 1.1 standard <strong>allows</strong> all CSS attributes/properties on a lot of elements. It does specify
     * that a lot of them only have <strong>effect</strong> on certain elements. That information we use here. However,
     * according to the standard, the 'visibility' property has no effect on 'g' elements. We do use that. It works in
     * Batik and Firefox (and probably other libraries as well). So, we allow 'visibility' on all elements that also
     * support 'display'.
     * </p>
     *
     * <p>
     * For SVG 1.2, this will need to be updated, see:
     * <ul>
     * <li><a href="http://www.w3.org/TR/SVG12/">http://www.w3.org/TR/SVG12/</a></li>
     * <li><a href=
     * "http://www.w3.org/TR/SVGTiny12/elementTable.html">http://www.w3.org/TR/SVGTiny12/elementTable.html</a></li>
     * <li><a href=
     * "http://www.w3.org/TR/SVGTiny12/attributeTable.html">http://www.w3.org/TR/SVGTiny12/attributeTable.html</a></li>
     * <li><a href=
     * "http://www.w3.org/TR/SVGMobile12/attributeTable.html">http://www.w3.org/TR/SVGMobile12/attributeTable.html</a></li>
     * <li><a href=
     * "http://www.w3.org/Graphics/SVG/1.2/rng/Full-1.2/">http://www.w3.org/Graphics/SVG/1.2/rng/Full-1.2/</a></li>
     * <li><a href="http://www.w3.org/TR/2004/WD-SVG12-20041027/">http://www.w3.org/TR/2004/WD-SVG12-20041027/</a></li>
     * <li><a href=
     * "http://dev.w3.org/SVG/modules/integration/master/attribute_table.html">http://dev.w3.org/SVG/modules/integration/master/attribute_table.html</a></li>
     * </ul>
     * </p>
     *
     * @see <a href="http://www.w3.org/TR/SVG11/eltindex.html">Appendix L: Element Index</a>
     * @see <a href="http://www.w3.org/TR/SVG11/attindex.html">Appendix M: Attribute Index</a>
     * @see <a href="http://www.w3.org/TR/SVG11/propidx.html">Appendix N: Property Index</a>
     * @see #getDefinedAttrs
     */
    private static final Map<String, Set<String>> SVG_ELEM_ATTR_MAP;

    static {
        // Initialization.
        Map<String, Set<String>> mapping = map();
        Set<String> attrs;

        // a
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("target");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("a", Collections.unmodifiableSet(attrs));

        // altGlyph
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("dx");
        attrs.add("dy");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("format");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("glyphRef");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("rotate");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("altGlyph", Collections.unmodifiableSet(attrs));

        // altGlyphDef
        attrs = set();
        attrs.add("id");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("altGlyphDef", Collections.unmodifiableSet(attrs));

        // altGlyphItem
        attrs = set();
        attrs.add("id");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("altGlyphItem", Collections.unmodifiableSet(attrs));

        // animate
        attrs = set();
        attrs.add("accumulate");
        attrs.add("additive");
        attrs.add("alignment-baseline");
        attrs.add("attributeName");
        attrs.add("attributeType");
        attrs.add("baseline-shift");
        attrs.add("begin");
        attrs.add("by");
        attrs.add("calcMode");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("dur");
        attrs.add("enable-background");
        attrs.add("end");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("from");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("keySplines");
        attrs.add("keyTimes");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("max");
        attrs.add("min");
        attrs.add("onbegin");
        attrs.add("onend");
        attrs.add("onload");
        attrs.add("onrepeat");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("repeatCount");
        attrs.add("repeatDur");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("restart");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("to");
        attrs.add("unicode-bidi");
        attrs.add("values");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("animate", Collections.unmodifiableSet(attrs));

        // animateColor
        attrs = set();
        attrs.add("accumulate");
        attrs.add("additive");
        attrs.add("alignment-baseline");
        attrs.add("attributeName");
        attrs.add("attributeType");
        attrs.add("baseline-shift");
        attrs.add("begin");
        attrs.add("by");
        attrs.add("calcMode");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("dur");
        attrs.add("enable-background");
        attrs.add("end");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("from");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("keySplines");
        attrs.add("keyTimes");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("max");
        attrs.add("min");
        attrs.add("onbegin");
        attrs.add("onend");
        attrs.add("onload");
        attrs.add("onrepeat");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("repeatCount");
        attrs.add("repeatDur");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("restart");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("to");
        attrs.add("unicode-bidi");
        attrs.add("values");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("animateColor", Collections.unmodifiableSet(attrs));

        // animateMotion
        attrs = set();
        attrs.add("accumulate");
        attrs.add("additive");
        attrs.add("begin");
        attrs.add("by");
        attrs.add("calcMode");
        attrs.add("dur");
        attrs.add("end");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("from");
        attrs.add("id");
        attrs.add("keyPoints");
        attrs.add("keySplines");
        attrs.add("keyTimes");
        attrs.add("max");
        attrs.add("min");
        attrs.add("onbegin");
        attrs.add("onend");
        attrs.add("onload");
        attrs.add("onrepeat");
        attrs.add("origin");
        attrs.add("path");
        attrs.add("repeatCount");
        attrs.add("repeatDur");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("restart");
        attrs.add("rotate");
        attrs.add("systemLanguage");
        attrs.add("to");
        attrs.add("values");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("animateMotion", Collections.unmodifiableSet(attrs));

        // animateTransform
        attrs = set();
        attrs.add("accumulate");
        attrs.add("additive");
        attrs.add("attributeName");
        attrs.add("attributeType");
        attrs.add("begin");
        attrs.add("by");
        attrs.add("calcMode");
        attrs.add("dur");
        attrs.add("end");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("from");
        attrs.add("id");
        attrs.add("keySplines");
        attrs.add("keyTimes");
        attrs.add("max");
        attrs.add("min");
        attrs.add("onbegin");
        attrs.add("onend");
        attrs.add("onload");
        attrs.add("onrepeat");
        attrs.add("repeatCount");
        attrs.add("repeatDur");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("restart");
        attrs.add("systemLanguage");
        attrs.add("to");
        attrs.add("type");
        attrs.add("values");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("animateTransform", Collections.unmodifiableSet(attrs));

        // circle
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("cx");
        attrs.add("cy");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("r");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("circle", Collections.unmodifiableSet(attrs));

        // clipPath
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("clipPathUnits");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("clipPath", Collections.unmodifiableSet(attrs));

        // color-profile
        attrs = set();
        attrs.add("id");
        attrs.add("local");
        attrs.add("name");
        attrs.add("rendering-intent");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("color-profile", Collections.unmodifiableSet(attrs));

        // cursor
        attrs = set();
        attrs.add("externalResourcesRequired");
        attrs.add("id");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("systemLanguage");
        attrs.add("x");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("cursor", Collections.unmodifiableSet(attrs));

        // defs
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("defs", Collections.unmodifiableSet(attrs));

        // desc
        attrs = set();
        attrs.add("class");
        attrs.add("id");
        attrs.add("style");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("desc", Collections.unmodifiableSet(attrs));

        // ellipse
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("cx");
        attrs.add("cy");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("rx");
        attrs.add("ry");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("ellipse", Collections.unmodifiableSet(attrs));

        // feBlend
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("in2");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("mode");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feBlend", Collections.unmodifiableSet(attrs));

        // feColorMatrix
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("type");
        attrs.add("unicode-bidi");
        attrs.add("values");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feColorMatrix", Collections.unmodifiableSet(attrs));

        // feComponentTransfer
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feComponentTransfer", Collections.unmodifiableSet(attrs));

        // feComposite
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("in2");
        attrs.add("k1");
        attrs.add("k2");
        attrs.add("k3");
        attrs.add("k4");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("operator");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feComposite", Collections.unmodifiableSet(attrs));

        // feConvolveMatrix
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("bias");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("divisor");
        attrs.add("dominant-baseline");
        attrs.add("edgeMode");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kernelMatrix");
        attrs.add("kernelUnitLength");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("order");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("preserveAlpha");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("targetX");
        attrs.add("targetY");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feConvolveMatrix", Collections.unmodifiableSet(attrs));

        // feDiffuseLighting
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("diffuseConstant");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kernelUnitLength");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("surfaceScale");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feDiffuseLighting", Collections.unmodifiableSet(attrs));

        // feDisplacementMap
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("in2");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("scale");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xChannelSelector");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        attrs.add("yChannelSelector");
        mapping.put("feDisplacementMap", Collections.unmodifiableSet(attrs));

        // feDistantLight
        attrs = set();
        attrs.add("azimuth");
        attrs.add("elevation");
        attrs.add("id");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("feDistantLight", Collections.unmodifiableSet(attrs));

        // feFlood
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feFlood", Collections.unmodifiableSet(attrs));

        // feFuncA
        attrs = set();
        attrs.add("amplitude");
        attrs.add("exponent");
        attrs.add("id");
        attrs.add("intercept");
        attrs.add("offset");
        attrs.add("slope");
        attrs.add("tableValues");
        attrs.add("type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("feFuncA", Collections.unmodifiableSet(attrs));

        // feFuncB
        attrs = set();
        attrs.add("amplitude");
        attrs.add("exponent");
        attrs.add("id");
        attrs.add("intercept");
        attrs.add("offset");
        attrs.add("slope");
        attrs.add("tableValues");
        attrs.add("type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("feFuncB", Collections.unmodifiableSet(attrs));

        // feFuncG
        attrs = set();
        attrs.add("amplitude");
        attrs.add("exponent");
        attrs.add("id");
        attrs.add("intercept");
        attrs.add("offset");
        attrs.add("slope");
        attrs.add("tableValues");
        attrs.add("type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("feFuncG", Collections.unmodifiableSet(attrs));

        // feFuncR
        attrs = set();
        attrs.add("amplitude");
        attrs.add("exponent");
        attrs.add("id");
        attrs.add("intercept");
        attrs.add("offset");
        attrs.add("slope");
        attrs.add("tableValues");
        attrs.add("type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("feFuncR", Collections.unmodifiableSet(attrs));

        // feGaussianBlur
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stdDeviation");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feGaussianBlur", Collections.unmodifiableSet(attrs));

        // feImage
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("preserveAspectRatio");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feImage", Collections.unmodifiableSet(attrs));

        // feMerge
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feMerge", Collections.unmodifiableSet(attrs));

        // feMergeNode
        attrs = set();
        attrs.add("id");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("feMergeNode", Collections.unmodifiableSet(attrs));

        // feMorphology
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("operator");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("radius");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feMorphology", Collections.unmodifiableSet(attrs));

        // feOffset
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("dx");
        attrs.add("dy");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feOffset", Collections.unmodifiableSet(attrs));

        // fePointLight
        attrs = set();
        attrs.add("id");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        attrs.add("z");
        mapping.put("fePointLight", Collections.unmodifiableSet(attrs));

        // feSpecularLighting
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kernelUnitLength");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("specularConstant");
        attrs.add("specularExponent");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("surfaceScale");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feSpecularLighting", Collections.unmodifiableSet(attrs));

        // feSpotLight
        attrs = set();
        attrs.add("id");
        attrs.add("limitingConeAngle");
        attrs.add("pointsAtX");
        attrs.add("pointsAtY");
        attrs.add("pointsAtZ");
        attrs.add("specularExponent");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        attrs.add("z");
        mapping.put("feSpotLight", Collections.unmodifiableSet(attrs));

        // feTile
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("in");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feTile", Collections.unmodifiableSet(attrs));

        // feTurbulence
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseFrequency");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("numOctaves");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("result");
        attrs.add("seed");
        attrs.add("shape-rendering");
        attrs.add("stitchTiles");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("type");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("feTurbulence", Collections.unmodifiableSet(attrs));

        // filter
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("filterRes");
        attrs.add("filterUnits");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("primitiveUnits");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("filter", Collections.unmodifiableSet(attrs));

        // font
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("horiz-adv-x");
        attrs.add("horiz-origin-x");
        attrs.add("horiz-origin-y");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("vert-adv-y");
        attrs.add("vert-origin-x");
        attrs.add("vert-origin-y");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("font", Collections.unmodifiableSet(attrs));

        // font-face
        attrs = set();
        attrs.add("accent-height");
        attrs.add("alphabetic");
        attrs.add("ascent");
        attrs.add("bbox");
        attrs.add("cap-height");
        attrs.add("descent");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("hanging");
        attrs.add("id");
        attrs.add("ideographic");
        attrs.add("mathematical");
        attrs.add("overline-position");
        attrs.add("overline-thickness");
        attrs.add("panose-1");
        attrs.add("slope");
        attrs.add("stemh");
        attrs.add("stemv");
        attrs.add("strikethrough-position");
        attrs.add("strikethrough-thickness");
        attrs.add("underline-position");
        attrs.add("underline-thickness");
        attrs.add("unicode-range");
        attrs.add("units-per-em");
        attrs.add("v-alphabetic");
        attrs.add("v-hanging");
        attrs.add("v-ideographic");
        attrs.add("v-mathematical");
        attrs.add("widths");
        attrs.add("x-height");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("font-face", Collections.unmodifiableSet(attrs));

        // font-face-format
        attrs = set();
        attrs.add("id");
        attrs.add("string");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("font-face-format", Collections.unmodifiableSet(attrs));

        // font-face-name
        attrs = set();
        attrs.add("id");
        attrs.add("name");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("font-face-name", Collections.unmodifiableSet(attrs));

        // font-face-src
        attrs = set();
        attrs.add("id");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("font-face-src", Collections.unmodifiableSet(attrs));

        // font-face-uri
        attrs = set();
        attrs.add("id");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("font-face-uri", Collections.unmodifiableSet(attrs));

        // foreignObject
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("foreignObject", Collections.unmodifiableSet(attrs));

        // g
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("g", Collections.unmodifiableSet(attrs));

        // glyph
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("arabic-form");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("d");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-name");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("horiz-adv-x");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("lang");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("orientation");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode");
        attrs.add("unicode-bidi");
        attrs.add("vert-adv-y");
        attrs.add("vert-origin-x");
        attrs.add("vert-origin-y");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("glyph", Collections.unmodifiableSet(attrs));

        // glyphRef
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("dx");
        attrs.add("dy");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("format");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("glyphRef");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("glyphRef", Collections.unmodifiableSet(attrs));

        // hkern
        attrs = set();
        attrs.add("g1");
        attrs.add("g2");
        attrs.add("id");
        attrs.add("k");
        attrs.add("u1");
        attrs.add("u2");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("hkern", Collections.unmodifiableSet(attrs));

        // image
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("preserveAspectRatio");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("image", Collections.unmodifiableSet(attrs));

        // line
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x1");
        attrs.add("x2");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y1");
        attrs.add("y2");
        mapping.put("line", Collections.unmodifiableSet(attrs));

        // linearGradient
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("gradientTransform");
        attrs.add("gradientUnits");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("shape-rendering");
        attrs.add("spreadMethod");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x1");
        attrs.add("x2");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y1");
        attrs.add("y2");
        mapping.put("linearGradient", Collections.unmodifiableSet(attrs));

        // marker
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("markerHeight");
        attrs.add("markerUnits");
        attrs.add("markerWidth");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("orient");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("preserveAspectRatio");
        attrs.add("refX");
        attrs.add("refY");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("viewBox");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("marker", Collections.unmodifiableSet(attrs));

        // mask
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("maskContentUnits");
        attrs.add("maskUnits");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("mask", Collections.unmodifiableSet(attrs));

        // metadata
        attrs = set();
        attrs.add("id");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("metadata", Collections.unmodifiableSet(attrs));

        // missing-glyph
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("d");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("horiz-adv-x");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("vert-adv-y");
        attrs.add("vert-origin-x");
        attrs.add("vert-origin-y");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("missing-glyph", Collections.unmodifiableSet(attrs));

        // mpath
        attrs = set();
        attrs.add("externalResourcesRequired");
        attrs.add("id");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("mpath", Collections.unmodifiableSet(attrs));

        // path
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("d");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pathLength");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("path", Collections.unmodifiableSet(attrs));

        // pattern
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("patternContentUnits");
        attrs.add("patternTransform");
        attrs.add("patternUnits");
        attrs.add("pointer-events");
        attrs.add("preserveAspectRatio");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("viewBox");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("pattern", Collections.unmodifiableSet(attrs));

        // polygon
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("points");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("polygon", Collections.unmodifiableSet(attrs));

        // polyline
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("points");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("polyline", Collections.unmodifiableSet(attrs));

        // radialGradient
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("cx");
        attrs.add("cy");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("fx");
        attrs.add("fy");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("gradientTransform");
        attrs.add("gradientUnits");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("r");
        attrs.add("shape-rendering");
        attrs.add("spreadMethod");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("radialGradient", Collections.unmodifiableSet(attrs));

        // rect
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("rx");
        attrs.add("ry");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("rect", Collections.unmodifiableSet(attrs));

        // script
        attrs = set();
        attrs.add("externalResourcesRequired");
        attrs.add("id");
        attrs.add("type");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("script", Collections.unmodifiableSet(attrs));

        // set
        attrs = set();
        attrs.add("attributeName");
        attrs.add("attributeType");
        attrs.add("begin");
        attrs.add("dur");
        attrs.add("end");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("id");
        attrs.add("max");
        attrs.add("min");
        attrs.add("onbegin");
        attrs.add("onend");
        attrs.add("onload");
        attrs.add("onrepeat");
        attrs.add("repeatCount");
        attrs.add("repeatDur");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("restart");
        attrs.add("systemLanguage");
        attrs.add("to");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("set", Collections.unmodifiableSet(attrs));

        // stop
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("offset");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("stop", Collections.unmodifiableSet(attrs));

        // style
        attrs = set();
        attrs.add("id");
        attrs.add("media");
        attrs.add("title");
        attrs.add("type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("style", Collections.unmodifiableSet(attrs));

        // svg
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseProfile");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("contentScriptType");
        attrs.add("contentStyleType");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onabort");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onerror");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("onresize");
        attrs.add("onscroll");
        attrs.add("onunload");
        attrs.add("onzoom");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("preserveAspectRatio");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("version");
        attrs.add("viewBox");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        attrs.add("zoomAndPan");
        mapping.put("svg", Collections.unmodifiableSet(attrs));

        // switch
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("switch", Collections.unmodifiableSet(attrs));

        // symbol
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("preserveAspectRatio");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("unicode-bidi");
        attrs.add("viewBox");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("symbol", Collections.unmodifiableSet(attrs));

        // text
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("dx");
        attrs.add("dy");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("lengthAdjust");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("rotate");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("textLength");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("text", Collections.unmodifiableSet(attrs));

        // textPath
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("lengthAdjust");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("method");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("spacing");
        attrs.add("startOffset");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("textLength");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("textPath", Collections.unmodifiableSet(attrs));

        // title
        attrs = set();
        attrs.add("class");
        attrs.add("id");
        attrs.add("style");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("title", Collections.unmodifiableSet(attrs));

        // tref
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("dx");
        attrs.add("dy");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("lengthAdjust");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("rotate");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("textLength");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("tref", Collections.unmodifiableSet(attrs));

        // tspan
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("dx");
        attrs.add("dy");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("lengthAdjust");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("rotate");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("textLength");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("tspan", Collections.unmodifiableSet(attrs));

        // use
        attrs = set();
        attrs.add("alignment-baseline");
        attrs.add("baseline-shift");
        attrs.add("class");
        attrs.add("clip");
        attrs.add("clip-path");
        attrs.add("clip-rule");
        attrs.add("color");
        attrs.add("color-interpolation");
        attrs.add("color-interpolation-filters");
        attrs.add("color-profile");
        attrs.add("color-rendering");
        attrs.add("cursor");
        attrs.add("direction");
        attrs.add("display");
        attrs.add("dominant-baseline");
        attrs.add("enable-background");
        attrs.add("externalResourcesRequired");
        attrs.add("fill");
        attrs.add("fill-opacity");
        attrs.add("fill-rule");
        attrs.add("filter");
        attrs.add("flood-color");
        attrs.add("flood-opacity");
        attrs.add("font-family");
        attrs.add("font-size");
        attrs.add("font-size-adjust");
        attrs.add("font-stretch");
        attrs.add("font-style");
        attrs.add("font-variant");
        attrs.add("font-weight");
        attrs.add("glyph-orientation-horizontal");
        attrs.add("glyph-orientation-vertical");
        attrs.add("height");
        attrs.add("id");
        attrs.add("image-rendering");
        attrs.add("kerning");
        attrs.add("letter-spacing");
        attrs.add("lighting-color");
        attrs.add("marker-end");
        attrs.add("marker-mid");
        attrs.add("marker-start");
        attrs.add("mask");
        attrs.add("onactivate");
        attrs.add("onclick");
        attrs.add("onfocusin");
        attrs.add("onfocusout");
        attrs.add("onload");
        attrs.add("onmousedown");
        attrs.add("onmousemove");
        attrs.add("onmouseout");
        attrs.add("onmouseover");
        attrs.add("onmouseup");
        attrs.add("opacity");
        attrs.add("overflow");
        attrs.add("pointer-events");
        attrs.add("requiredExtensions");
        attrs.add("requiredFeatures");
        attrs.add("shape-rendering");
        attrs.add("stop-color");
        attrs.add("stop-opacity");
        attrs.add("stroke");
        attrs.add("stroke-dasharray");
        attrs.add("stroke-dashoffset");
        attrs.add("stroke-linecap");
        attrs.add("stroke-linejoin");
        attrs.add("stroke-miterlimit");
        attrs.add("stroke-opacity");
        attrs.add("stroke-width");
        attrs.add("style");
        attrs.add("systemLanguage");
        attrs.add("text-anchor");
        attrs.add("text-decoration");
        attrs.add("text-rendering");
        attrs.add("transform");
        attrs.add("unicode-bidi");
        attrs.add("visibility");
        attrs.add("width");
        attrs.add("word-spacing");
        attrs.add("writing-mode");
        attrs.add("x");
        attrs.add("xlink:actuate");
        attrs.add("xlink:arcrole");
        attrs.add("xlink:href");
        attrs.add("xlink:role");
        attrs.add("xlink:show");
        attrs.add("xlink:title");
        attrs.add("xlink:type");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("y");
        mapping.put("use", Collections.unmodifiableSet(attrs));

        // view
        attrs = set();
        attrs.add("externalResourcesRequired");
        attrs.add("id");
        attrs.add("preserveAspectRatio");
        attrs.add("viewBox");
        attrs.add("viewTarget");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        attrs.add("zoomAndPan");
        mapping.put("view", Collections.unmodifiableSet(attrs));

        // vkern
        attrs = set();
        attrs.add("g1");
        attrs.add("g2");
        attrs.add("id");
        attrs.add("k");
        attrs.add("u1");
        attrs.add("u2");
        attrs.add("xml:base");
        attrs.add("xml:lang");
        attrs.add("xml:space");
        mapping.put("vkern", Collections.unmodifiableSet(attrs));

        // Store mapping in static variable.
        SVG_ELEM_ATTR_MAP = Collections.unmodifiableMap(mapping);
    }

    /**
     * Returns the attributes that are defined in the SVG 1.1 standard as being allowed for elements with the given
     * name. Both SVG presentation attributes and CSS style attributes are returned. If no element with the given name
     * is specified in the standard, {@code null} is returned.
     *
     * <p>
     * The SVG 1.1 standard <strong>allows</strong> all CSS attributes/properties on a lot of elements. It does specify
     * that a lot of them only have <strong>effect</strong> on certain elements. That information we use here. However,
     * according to the standard, the 'visibility' property has no effect on 'g' elements. We do use that. It works in
     * Batik and Firefox (and probably other libraries as well). So, we allow 'visibility' on all elements that also
     * support 'display'.
     * </p>
     *
     * @param elemName The name of the SVG element. Note that this is not the 'id' of the element, but the name (for
     *     instance: 'rect', 'g').
     * @return The allowed attributes, or {@code null}.
     */
    public static Set<String> getDefinedAttrs(String elemName) {
        return SVG_ELEM_ATTR_MAP.get(elemName);
    }

    /**
     * Checks an SVG name for validity.
     *
     * <p>
     * The SVG standard builds on the XML standard, which defines the valid syntax for names ('Name' symbol). The
     * <a href="http://www.w3.org/TR/REC-xml/#NT-Name">XML standard</a> defines the syntax of the 'Name' symbol (using
     * EBNF notation) as the following (all non-ASCII characters are omitted):
     *
     * <pre>
     *  [4]  NameStartChar ::= ":" | [A-Z] | "_" | [a-z]
     *  [4a] NameChar      ::= NameStartChar | "-" | "." | [0-9]
     *  [5]  Name          ::= NameStartChar (NameChar)*
     * </pre>
     * </p>
     *
     * @param name The name to check.
     * @return {@code true} if the name is valid, {@code false} otherwise.
     */
    public static boolean isValidSvgName(String name) {
        return NAME_PATTERN.matcher(name).matches();
    }

    /**
     * Checks an SVG name prefix for validity. Valid prefixes are: valid SVG names and empty prefixes.
     *
     * @param prefix The prefix to check.
     * @return {@code true} if the prefix is valid, {@code false} otherwise.
     */
    public static boolean isValidSvgPrefixName(String prefix) {
        return prefix.isEmpty() || isValidSvgName(prefix);
    }

    /**
     * Checks an SVG name postfix for validity. Valid postfixes are: postfixes that are valid SVG names after prefixing
     * them with {@code "_"}, as well as empty postfixes.
     *
     * @param postfix The postfix to check.
     * @return {@code true} if the postfix is valid, {@code false} otherwise.
     */
    public static boolean isValidSvgPostfixName(String postfix) {
        return postfix.isEmpty() || isValidSvgName("_" + postfix);
    }
}
