//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2022 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.codegen;

import java.util.List;

import org.eclipse.escet.cif.codegen.assignments.Destination;
import org.eclipse.escet.common.box.CodeBox;
import org.eclipse.escet.common.box.MemoryCodeBox;
import org.eclipse.escet.common.java.Assert;

/** Generated code and data value from the expression code generator. */
public class ExprCode {
    /** Empty code place holder, to avoid creating empty code boxes that are never filled. */
    private static final CodeBox EMPTY_CODE = new MemoryCodeBox();

    /**
     * Generated code, to be executed before using the data value. Is {@code null} if no code has been generated so far.
     */
    private CodeBox code = null;

    /** Whether the generated code has been queried from the object. */
    private boolean codeRetrieved = false;

    /** Destination to store the given data value. */
    private Destination destination = null;

    /** Data value generated by the expression code generator. */
    private DataValue value = null;

    /** Constructor of the {@link ExprCode} class. */
    public ExprCode() {
    }

    /** Ensure a code box exists for adding generated code. */
    private void prepareAddCode() {
        Assert.check(!codeRetrieved);
        if (code == null) {
            code = new MemoryCodeBox();
        }
    }

    /**
     * Test whether the object contains generated code.
     *
     * @return {@code true} if it contains code, {@code false} otherwise.
     */
    public boolean hasCode() {
        return code != null && !code.isEmpty();
    }

    /** Append an empty line to the generated code. */
    public void add() {
        prepareAddCode();
        code.add();
    }

    /**
     * Append a line to the generated code.
     *
     * @param line Line to append.
     */
    public void add(String line) {
        prepareAddCode();
        code.add(line);
    }

    /**
     * Append a parameterized line to the generated code.
     *
     * @param line Template line.
     * @param args Arguments of the template.
     */
    public void add(String line, Object[] args) {
        prepareAddCode();
        code.add(line, args);
    }

    /**
     * Append many lines to the generated code.
     *
     * @param lines Lines to append.
     */
    public void add(List<String> lines) {
        prepareAddCode();
        code.add(lines);
    }

    /**
     * Append many lines to the generated code.
     *
     * @param lines Lines to append.
     */
    public void add(String[] lines) {
        prepareAddCode();
        code.add(lines);
    }

    /**
     * Append code to the generated code.
     *
     * @param box Code to append.
     */
    public void add(CodeBox box) {
        prepareAddCode();
        code.add(box);
    }

    /**
     * Append previous code from the expression code generator to the generated code.
     *
     * @param exprCode Previous code generation result to append.
     */
    public void add(ExprCode exprCode) {
        prepareAddCode();
        CodeBox rawCode = exprCode.getRawCode();
        if (rawCode != null && !rawCode.isEmpty()) {
            code.add(rawCode);
        }
    }

    /**
     * Retrieve the generated code from the result if it exists. Can be done only one time.
     *
     * <p>
     * This method is mainly intended for internal use, the {@link #getCode} method is easier to use.
     * </p>
     *
     * @return The code of the result if available, else {@code null}.
     */
    public CodeBox getRawCode() {
        Assert.check(!codeRetrieved);

        if (destination != null) {
            Assert.check(value != null);
            if (code == null) {
                prepareAddCode();
            }
            destination.getTypeInfo().storeValue(code, value, destination);
            destination = null;
            value = null;
        }
        codeRetrieved = true;
        return code;
    }

    /**
     * Retrieve the generated code from the result. Can be done only one time.
     *
     * @return The code of the result.
     */
    public CodeBox getCode() {
        CodeBox rawCode = getRawCode();
        return (rawCode == null) ? EMPTY_CODE : rawCode;
    }

    /**
     * Set the desired storage destination of the data value.
     *
     * @param dest Desired storage destination of the value.
     */
    public void setDestination(Destination dest) {
        Assert.check(!codeRetrieved);
        destination = dest;
    }

    /**
     * Set the data value of the expression code generator result. Can be used only one time.
     *
     * @param value Value to set.
     */
    public void setDataValue(DataValue value) {
        Assert.check(this.value == null);
        this.value = value;
    }

    /**
     * Test whether the expression code generation result has a data value.
     *
     * <p>
     * Main intended use is for contract verification.
     * </p>
     *
     * @return {@code true} if the object has a data value, else {@code false}.
     */
    public boolean hasDataValue() {
        return destination == null && value != null;
    }

    /**
     * Retrieve the stored data value.
     *
     * <p>
     * This method is intended for passing data values on to other expression code results. To get the value as a
     * string, use {@link #getData} or {@link #getReference}.
     * </p>
     *
     * @return The stored data value.
     */
    public DataValue getRawDataValue() {
        Assert.check(codeRetrieved || code == null || code.isEmpty());
        Assert.check(destination == null && value != null);
        DataValue retValue = value;
        value = null;
        return retValue;
    }

    /**
     * Get a code fragment that represents the data value itself.
     *
     * <p>
     * Data value can only be retrieved after getting the generated code.
     * </p>
     *
     * @return Code fragment representing the data value.
     */
    public String getData() {
        return getRawDataValue().getData();
    }

    /**
     * Get a code fragment that represents a reference to the data value. It should be clear from the context whether
     * this method will yield a result.
     *
     * <p>
     * Reference can only be retrieved after getting the generated code.
     * </p>
     *
     * @return Code fragment representing a reference to the data value.
     */
    public String getReference() {
        return getRawDataValue().getReference();
    }
}
