//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2023 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.setext.runtime;

import static org.eclipse.escet.common.app.framework.output.OutputProvider.out;
import static org.eclipse.escet.common.java.Lists.first;
import static org.eclipse.escet.common.java.Lists.last;
import static org.eclipse.escet.common.java.Lists.listc;
import static org.eclipse.escet.common.java.Pair.pair;
import static org.eclipse.escet.common.java.Sets.set;
import static org.eclipse.escet.common.java.Sets.setc;
import static org.eclipse.escet.common.java.Sets.sortedgeneric;
import static org.eclipse.escet.common.java.Sets.sortedstrings;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.Set;

import org.eclipse.escet.common.app.framework.Paths;
import org.eclipse.escet.common.app.framework.exceptions.InputOutputException;
import org.eclipse.escet.common.app.framework.exceptions.InvalidInputException;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.Pair;
import org.eclipse.escet.common.java.TextPosition;
import org.eclipse.escet.setext.runtime.exceptions.ParseException;
import org.eclipse.escet.setext.runtime.exceptions.SyntaxException;

/**
 * Base class for parsers generated by SeText.
 *
 * @param <T> The type of the abstract syntax tree that results from parsing.
 */
public abstract class Parser<T> {
    /** The syntax warnings found so far. Is modified in-place. */
    private final Set<SyntaxWarning> warnings = set();

    /** The scanner to use to read tokens from the input. */
    protected final Scanner scanner;

    /**
     * Information about the expected terminals, per parser state:
     * <ul>
     * <li>Outer array: per parser state (complete).</li>
     * <li>Inner array: the union of the terminal ids of the terminals in {@code first(Z1 Z2 ... Zn)} for all grammar
     * productions {@code X: Y1 Y2 ... Ym . Z1 Z2 ... Zn} of the parser state.</li>
     * </ul>
     */
    protected int[][] firstTerminals;

    /**
     * Information about the expected terminals, per parser state, per reduced non-terminal {@code T} via which the
     * parser state is reached:
     * <ul>
     * <li>Outer array: per parser state (complete).</li>
     * <li>Middle array: per unique non-terminal {@code T}, from all grammar productions
     * {@code X: Y1 Y2 ... Ym . T Z1 Z2 ... Zn} of the parser state.</li>
     * <li>Inner array: the id of non-terminal {@code T} (at index {@code 0}), and the union of the terminal ids of the
     * terminals in {@code first(Z1 Z2 ... Zn)} (at the remaining indices).</li>
     * </ul>
     */
    protected int[][][] firstTerminalsReduced;

    /**
     * Information about the reducible non-terminals, per parser state:
     * <ul>
     * <li>Outer array: per parser state (complete).</li>
     * <li>Middle array: per unique non-terminal {@code X}, from all grammar productions
     * {@code X: Y1 Y2 ... Ym . Z1 Z2 ... Zn} of the parser state, where {@code Z1 Z2 ... Zn} accepts empty (0 symbols
     * or 1 or more non-terminals that all accept empty).</li>
     * <li>Inner array: the id of non-terminal {@code X} (at index {@code 0}), and the number of states to pop
     * ({@code m}, the number of terminals and non-terminals before the '{@code .}') (at the remaining indices).</li>
     * </ul>
     */
    protected int[][][] reducibleNonTerminals;

    /**
     * Information about the reducible non-terminals, per parser state, per reduced non-terminal {@code T} via which the
     * parser state is reached:
     * <ul>
     * <li>Outer array: per parser state (complete).</li>
     * <li>Middle array: per unique pair of non-terminals {@code T} and {@code X}, from all grammar productions
     * {@code X: Y1 Y2 ... Ym . T Z1 Z2 ... Zn} of the parser state, where {@code Z1 Z2 ... Zn} accepts empty (0 symbols
     * or 1 or more non-terminals that all accept empty).</li>
     * <li>Inner array: the id of non-terminal {@code T} (at index {@code 0}), the id of non-terminal {@code X} (at
     * index {@code 1}), and the number of states to pop ({@code m}, the number of terminals and non-terminals before
     * the '{@code .}') (at the remaining indices).</li>
     * </ul>
     */
    protected int[][][] reducibleNonTerminalsReduced;

    /** The entry symbol names for each of the parser states, and {@code null} for the initial state. */
    protected String[] entrySymbolNames;

    /** Whether to output debug information for the parser. */
    protected boolean debugParser;

    /** Parser state id stack. This is the current up-to-date state stack, used during parsing. */
    protected LinkedList<Integer> stateStack;

    /**
     * Parser state id stack, without final reductions. This state stack is used in case of parse failures, to obtain
     * the expected terminals.
     *
     * <p>
     * Only the difference with {@link #stateStack} is stored. The {@link #stateStackPrefixCount common prefix} is not
     * stored. After a non-reduce action, this state stack is conceptually equal to the {@link #stateStack reduced state
     * stack}, and thus stored as an empty list. The non-reduced state stack is conceptually not updated for reduce
     * actions, but since the {@link #stateStack reduced state stack} is updated for reduce actions, reduce actions
     * introduce differences, which <em>are</em> stored. Is {@code null} if not available.
     * </p>
     */
    protected LinkedList<Integer> stateStackNonReduced;

    /**
     * The length of the common prefix of the {@link #stateStack reduced state stack} and the
     * {@link #stateStackNonReduced non-reduced state stack}. The common prefix is only stored in the reduced state
     * stack, and not in the non-reduced state stack. Is zero if not available.
     */
    protected int stateStackPrefixCount;

    /** Parser object stack. */
    protected LinkedList<Object> objectStack;

    /**
     * Collected fold ranges. If {@code null}, no fold ranges will be collected. Users of parsers may set this to an
     * empty list to indicate that fold ranges should be collected.
     */
    public List<org.eclipse.jface.text.Position> foldRanges = null;

    /**
     * Constructor for the {@link Parser} class.
     *
     * @param scanner The scanner to use to read tokens from the input.
     */
    public Parser(Scanner scanner) {
        this.scanner = scanner;
    }

    /**
     * Add a fold range to the collection. The range covers the given tokens, as well as the space between them, if
     * applicable.
     *
     * @param start Token denoting the start of the range (inclusive).
     * @param end Token denoting the end of the range (inclusive).
     * @throws IllegalArgumentException If the start token does not start before the end token.
     * @throws IllegalArgumentException If the end token does not end after the start token.
     */
    public void addFoldRange(Token start, Token end) {
        addFoldRange(start.position, end.position);
    }

    /**
     * Add a fold range to the collection. The range covers the given positions, as well as the space between them, if
     * applicable.
     *
     * @param start Position denoting the start of the range (inclusive).
     * @param end Position denoting the end of the range (inclusive).
     * @throws IllegalArgumentException If the start position does not start before the end position.
     * @throws IllegalArgumentException If the end position does not end after the start position.
     */
    public void addFoldRange(TextPosition start, TextPosition end) {
        // Skip if we're not collecting fold ranges.
        if (foldRanges == null) {
            return;
        }

        // Validation.
        if (start.startOffset >= end.startOffset) {
            throw new IllegalArgumentException("start.start >= end.start");
        }
        if (end.endOffset <= start.endOffset) {
            throw new IllegalArgumentException("end.end <= start.end");
        }

        // Add range.
        int first = start.startOffset;
        int last = end.endOffset;
        addFoldRange(first, last + 1 - first);
    }

    /**
     * Add a fold range to the collection.
     *
     * @param start Start offset of the fold range in the file.
     * @param length Length of the fold range in the file.
     * @throws IllegalArgumentException If {@code start} is negative.
     * @throws IllegalArgumentException If {@code length} is not positive.
     */
    public void addFoldRange(int start, int length) {
        // Skip if we're not collecting fold ranges.
        if (foldRanges == null) {
            return;
        }

        // Validation.
        if (start < 0) {
            throw new IllegalArgumentException("start < 0");
        }
        if (length <= 0) {
            throw new IllegalArgumentException("length <= 0");
        }

        // Add range.
        foldRanges.add(new org.eclipse.jface.text.Position(start, length));
    }

    /**
     * Adds a syntax warning to the list of warnings found so far.
     *
     * @param message The message describing the syntax warning.
     * @param position Position information.
     */
    public void addWarning(String message, TextPosition position) {
        warnings.add(new SyntaxWarning(message, position));
    }

    /**
     * Returns the syntax warnings found so far. The resulting list must never be modified in-place!
     *
     * @return The syntax warnings found so far (sorted).
     */
    public List<SyntaxWarning> getWarnings() {
        return sortedgeneric(warnings);
    }

    /**
     * Returns the object with the parser call back hooks.
     *
     * @return The object with the parser call back hooks.
     */
    public abstract ParserHooksBase getHooks();

    /**
     * Returns the source of the input data. This text is prefixed to exception messages. May be {@code null} to
     * indicate no source information is available.
     *
     * @return The source of the input data, or {@code null}.
     */
    public String getSource() {
        return scanner.src;
    }

    /**
     * Returns the location of the source file being parsed, as an absolute local file system path, with platform
     * specific file separators. The path does not necessarily refer to an existing file.
     *
     * @return The location of the source file.
     */
    public String getLocation() {
        return scanner.location;
    }

    /**
     * Parses the given textual input and returns the parse result, without source information, and without outputting
     * debug information.
     *
     * @param inputText The textual input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws SyntaxException If parsing failed.
     */
    public T parseString(String inputText, String location) {
        return parseString(inputText, location, null, DebugMode.NONE);
    }

    /**
     * Parses the given textual input and returns the parse result, without outputting debug information.
     *
     * @param inputText The textual input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param src The source of the data to be parsed. This text is prefixed to scanning and parsing exception messages.
     *     May be {@code null} to indicate no source information is available.
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws SyntaxException If parsing failed.
     */
    public T parseString(String inputText, String location, String src) {
        return parseString(inputText, location, src, DebugMode.NONE);
    }

    /**
     * Parses the given textual input and returns the parse result, without source information.
     *
     * @param inputText The textual input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param debug The debug mode (what debug output to generate).
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws SyntaxException If parsing failed.
     */
    public T parseString(String inputText, String location, DebugMode debug) {
        return parseString(inputText, location, null, debug);
    }

    /**
     * Parses the given textual input and returns the parse result.
     *
     * @param inputText The textual input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param src The source of the data to be parsed. This text is prefixed to scanning and parsing exception messages.
     *     May be {@code null} to indicate no source information is available.
     * @param debug The debug mode (what debug output to generate).
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws SyntaxException If parsing failed.
     */
    public T parseString(String inputText, String location, String src, DebugMode debug) {
        // Construct input reader. No need to buffer in-memory strings.
        StringReader strReader = new StringReader(inputText);
        CodePointReader cpReader = new CodePointReader(strReader, false);

        // Defer to default parsing method.
        return parseReader(cpReader, location, src, debug);
    }

    /**
     * Parses a text file and returns the parse result, without source information, and without outputting debug
     * information. Assumes a UTF-8 encoding for the file.
     *
     * @param filePath The path to the input file to parse. May be an absolute or relative local file system path. The
     *     path is {@link Paths#resolve resolved} against the application's current working directory. This (unresolved)
     *     path is also used in the source for position information.
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InvalidInputException If the file can not be found, is a directory rather than a regular file, or for
     *     some other reason can not be opened for reading.
     * @throws InputOutputException If reading or closing the file failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseFile(String filePath) {
        return parseFile(Paths.resolve(filePath), filePath, DebugMode.NONE);
    }

    /**
     * Parses a text file and returns the parse result, without outputting debug information. Assumes a UTF-8 encoding
     * for the file.
     *
     * @param filePath The path to the input file to parse. Must be an absolute local file system path.
     * @param srcFilePath The path to the input file to parse. May be an absolute or relative local file system path.
     *     This path is used in the source for position information.
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InvalidInputException If the file can not be found, is a directory rather than a regular file, or for
     *     some other reason can not be opened for reading.
     * @throws InputOutputException If reading or closing the file failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseFile(String filePath, String srcFilePath) {
        return parseFile(filePath, srcFilePath, DebugMode.NONE);
    }

    /**
     * Parses a text file and returns the parse result, without source information. Assumes a UTF-8 encoding for the
     * file.
     *
     * @param filePath The path to the input file to parse. May be an absolute or relative local file system path. The
     *     path is {@link Paths#resolve resolved} against the application's current working directory. This (unresolved)
     *     path is also used in the source for position information.
     * @param debug The debug mode (what debug output to generate).
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InvalidInputException If the file can not be found, is a directory rather than a regular file, or for
     *     some other reason can not be opened for reading.
     * @throws InputOutputException If reading or closing the file failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseFile(String filePath, DebugMode debug) {
        return parseFile(Paths.resolve(filePath), filePath, debug);
    }

    /**
     * Parses a text file and returns the parse result. Assumes a UTF-8 encoding for the file.
     *
     * @param filePath The path to the input file to parse. Must be an absolute local file system path.
     * @param srcFilePath The path to the input file to parse. May be an absolute or relative local file system path.
     *     This path is used in the source for position information.
     * @param debug The debug mode (what debug output to generate).
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InvalidInputException If the file can not be found, is a directory rather than a regular file, or for
     *     some other reason can not be opened for reading.
     * @throws InputOutputException If reading or closing the file failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseFile(String filePath, String srcFilePath, DebugMode debug) {
        // Open file.
        FileInputStream fileStream;
        try {
            fileStream = new FileInputStream(filePath);
        } catch (FileNotFoundException e) {
            String msg = fmt("Could not open file \"%s\", as the file does not exist, is a directory rather than a "
                    + "regular file, or for some other reason cannot be opened for reading.", filePath);
            throw new InvalidInputException(msg, e);
        }

        // Parse the stream.
        String src = fmt("File \"%s\": ", srcFilePath);
        return parseStream(fileStream, filePath, src, debug);
    }

    /**
     * Parses the given textual input and returns the parse result, without source information, and without outputting
     * debug information. Assumes a UTF-8 encoding for the stream. Stream is closed after parsing.
     *
     * @param stream The stream to use to read bytes from the input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InputOutputException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseStream(InputStream stream, String location) {
        return parseStream(stream, location, null, DebugMode.NONE);
    }

    /**
     * Parses the given textual input and returns the parse result, without outputting debug information. Assumes a
     * UTF-8 encoding for the stream. Stream is closed after parsing.
     *
     * @param stream The stream to use to read bytes from the input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param src The source of the data to be parsed. This text is prefixed to scanning and parsing exception messages.
     *     May be {@code null} to indicate no source information is available.
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InputOutputException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseStream(InputStream stream, String location, String src) {
        return parseStream(stream, location, src, DebugMode.NONE);
    }

    /**
     * Parses the given textual input and returns the parse result, without source information. Assumes a UTF-8 encoding
     * for the stream. Stream is closed after parsing.
     *
     * @param stream The stream to use to read bytes from the input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param debug The debug mode (what debug output to generate).
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InputOutputException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseStream(InputStream stream, String location, DebugMode debug) {
        return parseStream(stream, location, null, debug);
    }

    /**
     * Parses the given textual input and returns the parse result. Assumes a UTF-8 encoding for the stream. Stream is
     * closed after parsing.
     *
     * @param stream The stream to use to read bytes from the input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param src The source of the data to be parsed. This text is prefixed to scanning and parsing exception messages.
     *     May be {@code null} to indicate no source information is available.
     * @param debug The debug mode (what debug output to generate).
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InputOutputException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseStream(InputStream stream, String location, String src, DebugMode debug) {
        // Construct code point reader for file, with UTF-8 encoding, and
        // buffering.
        CodePointReader reader = new CodePointReader(stream, true);

        // Defer to default parsing method, but always close the stream.
        try {
            return parseReader(reader, location, src, debug);
        } finally {
            // Always close the stream.
            try {
                stream.close();
            } catch (IOException e) {
                String msg = fmt("Could not close \"%s\".", location);
                throw new InputOutputException(msg);
            }
        }
    }

    /**
     * Parses the given textual input and returns the parse result, without source information, and without outputting
     * debug information.
     *
     * @param reader The reader to use to read code points from the input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InputOutputException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseReader(CodePointReader reader, String location) {
        return parseReader(reader, location, null, DebugMode.NONE);
    }

    /**
     * Parses the given textual input and returns the parse result, without outputting debug information.
     *
     * @param reader The reader to use to read code points from the input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param src The source of the data to be parsed. This text is prefixed to scanning and parsing exception messages.
     *     May be {@code null} to indicate no source information is available.
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InputOutputException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseReader(CodePointReader reader, String location, String src) {
        return parseReader(reader, location, src, DebugMode.NONE);
    }

    /**
     * Parses the given textual input and returns the parse result, without source information.
     *
     * @param reader The reader to use to read code points from the input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param debug The debug mode (what debug output to generate).
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InputOutputException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseReader(CodePointReader reader, String location, DebugMode debug) {
        return parseReader(reader, location, null, debug);
    }

    /**
     * Parses the given textual input and returns the parse result.
     *
     * @param reader The reader to use to read code points from the input.
     * @param location The location of the source file being parsed. Must be an absolute local file system path, with
     *     platform specific file separators. The path does not have to refer to an existing file.
     * @param src The source of the data to be parsed. This text is prefixed to scanning and parsing exception messages.
     *     May be {@code null} to indicate no source information is available.
     * @param debug The debug mode (what debug output to generate).
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws InputOutputException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    public T parseReader(CodePointReader reader, String location, String src, DebugMode debug) {
        // Initialization.
        debugParser = debug == DebugMode.PARSER || debug == DebugMode.BOTH;

        if (debugParser) {
            out("%s: %sParsing...", getClass().getSimpleName(), (src == null) ? "" : src);
        }

        scanner.initScanner(reader, location, src, debug, true);

        stateStack = new LinkedList<>();
        stateStack.addLast(0);

        stateStackNonReduced = new LinkedList<>();
        stateStackPrefixCount = 1;

        objectStack = new LinkedList<>();

        getHooks().setParser(this);

        // Perform parsing.
        T rslt;
        try {
            rslt = parse();
        } catch (IOException e) {
            String msg = fmt("%sFailed to read input.", (src == null) ? "" : src);
            throw new InputOutputException(msg, e);
        }

        // Return the root of the abstract syntax tree that resulted from
        // parsing.
        return rslt;
    }

    /**
     * Parses the input and returns the parse result.
     *
     * @return The parse result, usually and abstract syntax tree.
     *
     * @throws IOException If reading the input failed due to an I/O error.
     * @throws SyntaxException If parsing failed.
     */
    protected abstract T parse() throws IOException;

    /**
     * Returns the next token from the scanner. Tokens that correspond to terminals without a name are skipped (except
     * for end-of-file terminals).
     *
     * @return The next token.
     * @throws IOException If reading the input failed due to an I/O error.
     */
    protected Token nextToken() throws IOException {
        Token token = scanner.nextToken();
        while (true) {
            // Accept named terminals.
            String terminalName = scanner.terminalNames[token.id];
            if (terminalName != null) {
                return token;
            }

            // Accept end-of-file terminals.
            if (token.isEof()) {
                return token;
            }

            // Skip this token and get the next one.
            token = scanner.nextToken();
        }
    }

    /**
     * Returns the current scanner state id. This is based on the {@link #stateStack reduced state stack}.
     *
     * @return The current scanner state id.
     */
    protected int getCurrentState() {
        return stateStack.peekLast();
    }

    /**
     * Perform a shift action.
     *
     * @param token The current input token, to shift.
     * @param stateId The state to add to the state stack.
     * @return The next token from the scanner. See {@link #nextToken}.
     * @throws IOException If reading the input failed due to an I/O error.
     */
    protected final Token doShift(Token token, int stateId) throws IOException {
        // If parser debugging is enabled, debug the shift action.
        if (debugParser) {
            debugShift(token);
        }

        // Update the reduced state stacks.
        stateStack.addLast(stateId);
        objectStack.addLast(token);

        // Update non-reduced state stack to be equal to the reduced state
        // stack.
        stateStackNonReduced.clear();
        stateStackPrefixCount = stateStack.size();

        // Return the next token.
        return nextToken();
    }

    /**
     * Perform the first phase of a reduce action.
     *
     * @param token The current input token.
     * @param reduceId The id of the non-terminal to reduce.
     */
    protected final void doReduce1(Token token, int reduceId) {
        // If parser debugging is enabled, debug the reduce action.
        if (debugParser) {
            debugReduce(token, reduceId);
        }
    }

    /**
     * Perform the second phase of a reduce action. Must be repeatedly invoked for all entries to pop from the stack.
     *
     * @return The object popped from the object stack, potentially to be provided to a parser call-back function.
     */
    protected final Object doReduce2() {
        // Update state stacks.
        int size = stateStack.size();
        if (size > stateStackPrefixCount) {
            // Beyond the common prefix. Only update reduced state stack for
            // reduce actions. The popped state was most likely added after a
            // previous reduce (by a goto action).
            stateStack.removeLast();
        } else if (size == stateStackPrefixCount) {
            // At the end of the common prefix. Move state from the reduced
            // state stack to the non-reduced state stack.
            int stateId = stateStack.removeLast();
            stateStackNonReduced.addFirst(stateId);
            stateStackPrefixCount--;
        } else {
            // Removing beyond the end of the common prefix. Should never
            // happen.
            String msg = fmt("Reduced state stack size (%,d) < common prefix count (%,d).", size,
                    stateStackPrefixCount);
            throw new RuntimeException(msg);
        }

        // Update object stack, and return popped element.
        return objectStack.removeLast();
    }

    /**
     * Perform the third phase of a reduce action.
     *
     * @param obj The object to add to the object stack, obtained from a parser call-back function.
     * @return The current parser state, after the reduce action, but before the corresponding goto action.
     */
    protected final int doReduce3(Object obj) {
        // Push object to the object stack.
        objectStack.addLast(obj);

        // Return the current parser state.
        return getCurrentState();
    }

    /**
     * Perform a goto action.
     *
     * @param stateId The state to add to the state stack.
     */
    protected final void doGoto(int stateId) {
        // Update the reduced state stack. Keep non-reduced stack intact.
        stateStack.addLast(stateId);
    }

    /**
     * Perform an accept action.
     *
     * @param token The current input token. Must be an end-of-file token.
     * @return The parse result.
     */
    protected final Object doAccept(Token token) {
        // If parser debugging is enabled, debug the accept action.
        if (debugParser) {
            debugAccept(token);
        }

        // Update the object stack.
        Object rslt = objectStack.removeLast();
        Assert.check(objectStack.isEmpty());

        // Parsing is done. No need for the non-reduced state stack, as it is
        // only needed in case of a parse error.
        stateStackNonReduced = null;
        stateStackPrefixCount = 0;

        // Return the parse result.
        return rslt;
    }

    /**
     * Collects the terminal ids of the terminals expected in the current parser state. It also checks the expected
     * terminals after any reductions that are possible.
     *
     * @return Expected terminal ids.
     */
    private Set<Integer> collectExpectedTerminalIds() {
        // Create ArrayList for the non-reduced state stack, for fast indexing.
        // Use the non-reduced state stack to ensure we get all expected
        // terminals.
        int stackSize = stateStackPrefixCount + stateStackNonReduced.size();
        List<Integer> stateStack = listc(stackSize);
        Iterator<Integer> stateIter = new NonReducedStateStackIterator();
        while (stateIter.hasNext()) {
            stateStack.add(stateIter.next());
        }

        // Create queue of pairs, of 0-based state stack indices, and 0-based
        // non-terminal ids for non-terminal reductions via which the state was
        // reached. Initialize to last state stack state (current state),
        // without a non-terminal reduction.
        Queue<Pair<Integer, Integer>> todo = new LinkedList<>();
        todo.add(pair(stackSize - 1, -1));

        // Process queue, collecting expected terminal ids.
        Set<Integer> expTermIds = set();
        while (!todo.isEmpty()) {
            // Get next state and reduced non-terminal.
            Pair<Integer, Integer> item = todo.remove();
            int stateIdx = item.left;
            int stateId = stateStack.get(stateIdx);
            int reducedNonTermId = item.right;

            // Different handling based on whether a non-terminal is reduced.
            if (reducedNonTermId == -1) {
                // First state to process (last one to be added to the stack).

                // Add 'first' terminals of the state.
                int[] termIds = firstTerminals[stateId];
                for (int termId: termIds) {
                    expTermIds.add(termId);
                }

                // If any non-terminals can be reduced, continue with them.
                int[][] reductions = reducibleNonTerminals[stateId];
                for (int[] reduction: reductions) {
                    int reductionNonTermId = reduction[0];
                    Assert.check(reductionNonTermId >= 0);

                    // Continue with states of the state stack reached after
                    // reducing the non-terminal with different pop counts.
                    for (int i = 1; i < reduction.length; i++) {
                        int reductionPopCount = reduction[i];
                        todo.add(pair(stateIdx - reductionPopCount, reductionNonTermId));
                    }
                }
            } else {
                // Other states to process (after reducing non-terminals).

                // Add 'first' terminals of the state, after reduced
                // non-terminal.
                int[][] idsArray = firstTerminalsReduced[stateId];
                for (int[] ids: idsArray) {
                    // Check matching reduced non-terminal id.
                    if (ids[0] != reducedNonTermId) {
                        continue;
                    }

                    // Add all 'first' terminals.
                    for (int i = 1; i < ids.length; i++) {
                        expTermIds.add(ids[i]);
                    }
                }

                // If any non-terminals can be reduced, continue with them.
                int[][] reductions = reducibleNonTerminalsReduced[stateId];
                for (int[] reduction: reductions) {
                    // Check matching reduced non-terminal id.
                    if (reduction[0] != reducedNonTermId) {
                        continue;
                    }

                    int reductionNonTermId = reduction[1];
                    Assert.check(reductionNonTermId >= 0);

                    // Continue with states of the state stack reached after
                    // reducing the non-terminal with different pop counts.
                    for (int i = 2; i < reduction.length; i++) {
                        int reductionPopCount = reduction[i];
                        todo.add(pair(stateIdx - reductionPopCount, reductionNonTermId));
                    }
                }
            }
        }

        // Return collected terminal ids.
        return expTermIds;
    }

    /**
     * Throws a {@link ParseException} to indicate parsing failure.
     *
     * @param token The token at which parsing failed.
     * @throws ParseException Always thrown.
     */
    protected void parsingFailed(Token token) {
        // Print debug output for parsing failure.
        if (debugParser) {
            debugFailed(token, true);
            debugFailed(token, false);
        }

        // Get found token text.
        String tokenText;
        if (token.isEof()) {
            tokenText = null;
        } else {
            tokenText = token.originalText;
            Assert.notNull(tokenText);
            Assert.check(!tokenText.isEmpty());
        }

        // Get found terminal text.
        String foundTermTxt = scanner.terminalDescriptions[token.id];

        // Get expected terminals text.
        Set<Integer> expTermIds = collectExpectedTerminalIds();

        Set<String> expTermsTxtsSet = setc(expTermIds.size());
        for (int termId: expTermIds) {
            expTermsTxtsSet.add(scanner.terminalDescriptions[termId]);
        }
        Assert.check(!expTermsTxtsSet.isEmpty());
        List<String> expTermsTxts = sortedstrings(expTermsTxtsSet);

        String expTermsTxt;
        if (expTermsTxts.size() == 1) {
            expTermsTxt = first(expTermsTxts);
        } else if (expTermsTxts.size() == 2) {
            expTermsTxt = first(expTermsTxts) + " or " + last(expTermsTxts);
        } else {
            List<String> ts = expTermsTxts.subList(0, expTermsTxts.size() - 1);
            expTermsTxt = String.join(", ", ts);
            expTermsTxt += ", or " + last(expTermsTxts);
        }

        // Throw exception to indicate parse failure.
        throw new ParseException(tokenText, token.position, foundTermTxt, expTermsTxt);
    }

    /**
     * Outputs debugging information for the parser, for a shift action. Must only be called if {@link #debugParser} is
     * {@code true}.
     *
     * @param token The current input token.
     */
    protected void debugShift(Token token) {
        String tokenTxt;
        if (token.isEof()) {
            tokenTxt = "\u00B6";
        } else {
            tokenTxt = scanner.terminalNames[token.id];
        }

        out("%s: %s (shift %s)", getClass().getSimpleName(), getStackTxt(token, true), tokenTxt);
    }

    /**
     * Outputs debugging information for the parser, for a reduce action. Must only be called if {@link #debugParser} is
     * {@code true}.
     *
     * @param token The current input token.
     * @param reduceId The id of the non-terminal being reduced.
     */
    protected void debugReduce(Token token, int reduceId) {
        String nonterminalName = getNonTerminalName(reduceId);
        out("%s: %s (reduce %s)", getClass().getSimpleName(), getStackTxt(token, true), nonterminalName);
    }

    /**
     * Outputs debugging information for the parser, for an accept action. Must only be called if {@link #debugParser}
     * is {@code true}.
     *
     * @param token The current input token. Must be an end-of-file token.
     */
    protected void debugAccept(Token token) {
        Assert.check(token.isEof());
        out("%s: %s (accept)", getClass().getSimpleName(), getStackTxt(token, true));
    }

    /**
     * Outputs debugging information for the parser, for a parse failure. Must only be called if {@link #debugParser} is
     * {@code true}.
     *
     * @param token The current input token.
     * @param reduced Whether to use the {@link #stateStack reduced state stack} ({@code true}) or the
     *     {@link #stateStackNonReduced non-reduced state stack} ({@code false}).
     */
    protected void debugFailed(Token token, boolean reduced) {
        out("%s: %s (parsing failed, %sreduced state stack)", getClass().getSimpleName(), getStackTxt(token, reduced),
                reduced ? "" : "non-");
    }

    /**
     * Returns the name of a non-terminal, given its unique id.
     *
     * @param nonTerminalId The unique id of the non-terminal.
     * @return The name of the non-terminal.
     */
    protected abstract String getNonTerminalName(int nonTerminalId);

    /**
     * Returns a textual representation of the current parser stack.
     *
     * @param token The current input token.
     * @param reduced Whether to use the {@link #stateStack reduced state stack} ({@code true}) or the
     *     {@link #stateStackNonReduced non-reduced state stack} ({@code false}).
     * @return A textual representation of the current parser stack.
     */
    private String getStackTxt(Token token, boolean reduced) {
        StringBuilder builder = new StringBuilder();
        Iterator<Integer> iter = reduced ? stateStack.iterator() : new NonReducedStateStackIterator();
        while (iter.hasNext()) {
            int state = iter.next();
            String entrySymbolName = entrySymbolNames[state];
            if (entrySymbolName != null) {
                builder.append(entrySymbolName);
            }
            builder.append(fmt("/%d ", state));
        }
        builder.append(". ");
        String tokenTxt;
        if (token.isEof()) {
            tokenTxt = "\u00B6";
        } else {
            tokenTxt = scanner.terminalNames[token.id];
        }
        builder.append(tokenTxt);
        return builder.toString();
    }

    /** State stack iterator for the non-reduced state stack. */
    private class NonReducedStateStackIterator implements Iterator<Integer> {
        /**
         * Reduced state stack iterator. Used to iterator over the common prefix.
         *
         * @see #stateStack
         * @see #stateStackPrefixCount
         */
        private final Iterator<Integer> reducedIterator;

        /**
         * Non-reduced state stack iterator. Used to iterate over the part of the non-reduced state stack that differs
         * from the prefix that it has in common with the reduced state stack.
         *
         * @see #stateStackNonReduced
         */
        private final Iterator<Integer> nonReducedIterator;

        /**
         * The number of states already returned by the {@link #next} method, for the prefix of the stack that the
         * reduced and non-reduced state stack share.
         */
        private int count;

        /** Constructor for the {@link NonReducedStateStackIterator} class. */
        public NonReducedStateStackIterator() {
            reducedIterator = stateStack.iterator();
            nonReducedIterator = stateStackNonReduced.iterator();
            count = 0;
        }

        @Override
        public boolean hasNext() {
            // Exhaust prefix of the reduced state stack first.
            if (count < stateStackPrefixCount) {
                return true;
            }

            // Use the part of the non-reduced state stack that differs from
            // prefix that it has in common with the reduced state stack.
            return nonReducedIterator.hasNext();
        }

        @Override
        public Integer next() {
            // Exhaust prefix of the reduced state stack first.
            if (count < stateStackPrefixCount) {
                count++;
                return reducedIterator.next();
            }

            // Use the part of the non-reduced state stack that differs from
            // prefix that it has in common with the reduced state stack. No
            // need to increment count, it is high enough already.
            return nonReducedIterator.next();
        }
    }
}
