//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2022, 2024 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.controllercheck.checks.confluence;

import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Lists.set2list;
import static org.eclipse.escet.common.java.Maps.mapc;
import static org.eclipse.escet.common.java.Strings.SORTER;

import java.math.BigInteger;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.bdd.spec.CifBddEdge;
import org.eclipse.escet.cif.bdd.spec.CifBddEdgeApplyDirection;
import org.eclipse.escet.cif.bdd.spec.CifBddSpec;
import org.eclipse.escet.cif.bdd.spec.CifBddVariable;
import org.eclipse.escet.cif.common.CifTextUtils;
import org.eclipse.escet.cif.controllercheck.checks.ControllerCheckerBddBasedCheck;
import org.eclipse.escet.cif.metamodel.cif.declarations.Event;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.Lists;
import org.eclipse.escet.common.java.Pair;
import org.eclipse.escet.common.java.Termination;
import org.eclipse.escet.common.java.output.DebugNormalOutput;

import com.github.javabdd.BDD;
import com.github.javabdd.BDDDomain;
import com.github.javabdd.BDDVarSet;

/** Class to check confluence of the specification. */
public class ConfluenceCheck extends ControllerCheckerBddBasedCheck<ConfluenceCheckConclusion> {
    /** Debug global flow of the checks, which pairs are tested, where are they matched. */
    private static final boolean DEBUG_GLOBAL = false;

    /** Whether to enable debugging output for independence checking. */
    private static final boolean DEBUG_INDENPENCE = false;

    /** Whether to enable debugging output for reversible checking. */
    private static final boolean DEBUG_REVERSIBLE = false;

    /** Whether to enable debugging output for update equivalence. */
    private static final boolean DEBUG_UPDATE_EQUIVALENCE = false;

    /** The name of the property being checked. */
    public static final String PROPERTY_NAME = "confluence";

    @Override
    public String getPropertyName() {
        return PROPERTY_NAME;
    }

    @Override
    public ConfluenceCheckConclusion performCheck(CifBddSpec cifBddSpec) {
        // Get information from BDD specification.
        Termination termination = cifBddSpec.settings.getTermination();
        DebugNormalOutput out = cifBddSpec.settings.getNormalOutput();
        DebugNormalOutput dbg = cifBddSpec.settings.getDebugOutput();
        List<Event> controllableEvents = set2list(cifBddSpec.controllables);

        // If no controllable events, then confluence trivially holds.
        if (controllableEvents.isEmpty()) {
            dbg.line("No controllable events. Confluence trivially holds.");
            return new ConfluenceCheckConclusion(List.of());
        }

        // Add 'zero' variables, variables that hold the values of the variables before taking any transitions (zero
        // transitions taken) to the BDD factory, and create an 'x0 = x and y0 = y and z0 = z and ...' identity
        // relations.
        BDD zeroToOldVarRelations = createZeroToOldVarsRelations(cifBddSpec);

        // Get single edge per controllable event.
        Map<Event, CifBddEdge> eventToEdge = cifBddSpec.eventEdges.entrySet().stream()
                .collect(Collectors.toMap(e -> e.getKey(), e -> Lists.single(e.getValue())));

        // Storage of test results.
        List<Pair<String, String>> mutualExclusives = list(); // List with pairs that are mutual exclusive.
        List<Pair<String, String>> updateEquivalents = list(); // List with pairs that are update equivalent.
        List<Pair<String, String>> independents = list(); // List with pairs that are independent.
        List<Pair<String, String>> skippables = list(); // List with pairs that are skippable.
        List<Pair<String, String>> reversibles = list(); // List with pairs that are reversible.
        List<Pair<String, String>> cannotProves = list(); // List with pairs where confluence could not be proven.

        // For all pairs of events, perform the checks. But, avoid checking both both (A, B) and (B, A).
        for (int i = 0; i < controllableEvents.size(); i++) {
            Event event1 = controllableEvents.get(i);
            String evt1Name = CifTextUtils.getAbsName(event1);
            CifBddEdge edge1 = eventToEdge.get(event1);

            for (int j = i + 1; j < controllableEvents.size(); j++) {
                Event event2 = controllableEvents.get(j);
                String evt2Name = CifTextUtils.getAbsName(event2);
                CifBddEdge edge2 = eventToEdge.get(event2);

                if (termination.isRequested()) {
                    return null;
                }

                if (DEBUG_GLOBAL) {
                    dbg.line("Trying event pair (" + evt1Name + ", " + evt2Name + ").");
                }

                // Check for mutual exclusiveness (never both guards are enabled at the same time).
                BDD commonEnabledGuards = edge1.guard.and(edge2.guard);
                if (commonEnabledGuards.isZero()) {
                    mutualExclusives.add(makeSortedPair(evt1Name, evt2Name));
                    if (DEBUG_GLOBAL) {
                        dbg.line("  -> event pair (" + evt1Name + ", " + evt2Name + ") is mutual exclusive.");
                    }
                    continue;
                }

                if (termination.isRequested()) {
                    return null;
                }

                // Connect 'zero' variables to the 'old' variables.
                commonEnabledGuards = zeroToOldVarRelations.id().andWith(commonEnabledGuards);

                if (termination.isRequested()) {
                    return null;
                }

                // Compute the 'source states', the states before where both events are enabled (but no transitions are
                // taken for them just yet), expressed in terms of 'zero' variables.
                BDD commonEnabledZeroStates = commonEnabledGuards.exist(cifBddSpec.varSetOld);
                if (DEBUG_GLOBAL) {
                    dbg.line("Zero states: %s", commonEnabledZeroStates.toString());
                }

                if (termination.isRequested()) {
                    return null;
                }

                // Check for update equivalence (both events make the same changes).
                if (DEBUG_UPDATE_EQUIVALENCE) {
                    dbg.line("Update equivalence: edge1 guard: %s", edge1.guard.toString());
                    dbg.line("Update equivalence: edge2 guard: %s", edge2.guard.toString());
                    dbg.line("Update equivalence: edge1 update/guard: %s", edge1.updateGuard.toString());
                    dbg.line("Update equivalence: edge2 update/guard: %s", edge1.updateGuard.toString());
                    dbg.line("Update equivalence: commonEnabledGuards: %s", commonEnabledGuards.toString());
                }

                BDD event1Done = edge1.apply(commonEnabledGuards.id(), CifBddEdgeApplyDirection.FORWARD, null);
                BDD event2Done = edge2.apply(commonEnabledGuards.id(), CifBddEdgeApplyDirection.FORWARD, null);

                if (DEBUG_UPDATE_EQUIVALENCE) {
                    dbg.line("Update equivalence: event1 done: %s", event1Done.toString());
                    dbg.line("Update equivalence: event2 done: %s", event2Done.toString());
                }

                if (!event1Done.isZero() && event1Done.equals(event2Done)
                        && allStatesCovered(commonEnabledZeroStates, event1Done, cifBddSpec.varSetOld))
                {
                    if (DEBUG_GLOBAL) {
                        dbg.line("  -> event pair (" + evt1Name + ", " + evt2Name + ") is update equivalent.");
                    }
                    updateEquivalents.add(makeSortedPair(evt1Name, evt2Name));

                    commonEnabledGuards.free();
                    commonEnabledZeroStates.free();
                    event1Done.free();
                    event2Done.free();
                    continue;
                }

                if (termination.isRequested()) {
                    return null;
                }

                // Check for independence (diamond shape edges leading to the same changes).
                if (DEBUG_INDENPENCE) {
                    dbg.line("Independence: edge1 guard: %s", edge1.guard.toString());
                    dbg.line("Independence: edge2 guard: %s", edge2.guard.toString());
                    dbg.line("Independence: edge1 update/guard: %s", edge1.updateGuard.toString());
                    dbg.line("Independence: edge2 update/guard: %s", edge1.updateGuard.toString());
                }

                // First event1 then event2.
                BDD event1Enabled2 = event1Done.and(edge2.guard);
                BDD event12Done = (event1Enabled2.isZero()) ? cifBddSpec.factory.zero()
                        : edge2.apply(event1Enabled2.id(), CifBddEdgeApplyDirection.FORWARD, null);

                if (DEBUG_INDENPENCE) {
                    dbg.line("Independence: event1 enabled2: %s", event1Enabled2.toString());
                    dbg.line("Independence: event1+2 done: %s", event12Done.toString());
                }

                event1Enabled2.free();

                if (termination.isRequested()) {
                    return null;
                }

                // First event2 then event1.
                BDD event2Enabled1 = event2Done.and(edge1.guard);
                BDD event21Done = (event2Enabled1.isZero()) ? cifBddSpec.factory.zero()
                        : edge1.apply(event2Enabled1.id(), CifBddEdgeApplyDirection.FORWARD, null);

                if (DEBUG_INDENPENCE) {
                    dbg.line("Independence: event2 enabled1: %s", event2Enabled1.toString());
                    dbg.line("Independence: event2+1 done: %s", event21Done.toString());
                }

                event2Enabled1.free();

                // Decide on independence.
                if (!event12Done.isZero() && event12Done.equals(event21Done)
                        && allStatesCovered(commonEnabledZeroStates, event12Done, cifBddSpec.varSetOld))
                {
                    independents.add(makeSortedPair(evt1Name, evt2Name));
                    if (DEBUG_GLOBAL) {
                        dbg.line("  -> event pair (" + evt1Name + ", " + evt2Name + ") is independent.");
                    }

                    commonEnabledGuards.free();
                    commonEnabledZeroStates.free();
                    event1Done.free();
                    event2Done.free();
                    event12Done.free();
                    event21Done.free();
                    continue;
                }

                if (termination.isRequested()) {
                    return null;
                }

                // Check for skippable (may perform a second event, but its changes are undone).

                // Check skippable event2 (events 2, 1 versus event 1).
                if (!event21Done.isZero() && event1Done.equals(event21Done)
                        && allStatesCovered(commonEnabledZeroStates, event1Done, cifBddSpec.varSetOld))
                {
                    skippables.add(makeSortedPair(evt1Name, evt2Name));
                    if (DEBUG_GLOBAL) {
                        dbg.line("  -> event pair (" + evt1Name + ", " + evt2Name + ") is skippable.");
                    }

                    commonEnabledGuards.free();
                    commonEnabledZeroStates.free();
                    event1Done.free();
                    event2Done.free();
                    event12Done.free();
                    event21Done.free();
                    continue;
                }

                if (termination.isRequested()) {
                    return null;
                }

                // Check skippable event1 (events 1, 2 versus event 2).
                if (!event12Done.isZero() && event2Done.equals(event12Done)
                        && allStatesCovered(commonEnabledZeroStates, event2Done, cifBddSpec.varSetOld))
                {
                    skippables.add(makeSortedPair(evt1Name, evt2Name));
                    if (DEBUG_GLOBAL) {
                        dbg.line("  -> event pair (" + evt1Name + ", " + evt2Name + ") is skippable.");
                    }

                    commonEnabledGuards.free();
                    commonEnabledZeroStates.free();
                    event1Done.free();
                    event2Done.free();
                    event12Done.free();
                    event21Done.free();
                    continue;
                }

                if (termination.isRequested()) {
                    return null;
                }

                // Check reversible (if event2 is performed before event1 its effect is reverted by event3 after
                // event1).
                if (DEBUG_REVERSIBLE) {
                    dbg.line("Reversible: edge1 guard: %s", edge1.guard.toString());
                    dbg.line("Reversible: edge2 guard: %s", edge2.guard.toString());
                    dbg.line("Reversible: edge1 update/guard: %s", edge1.updateGuard.toString());
                    dbg.line("Reversible: edge2 update/guard: %s", edge1.updateGuard.toString());
                }

                boolean foundReversible = false;
                for (int k = 0; k < controllableEvents.size(); k++) {
                    Event event3 = controllableEvents.get(k);
                    if (event3 == event1 || event3 == event2) {
                        continue;
                    }

                    if (termination.isRequested()) {
                        return null;
                    }

                    String evt3Name = CifTextUtils.getAbsName(event3);
                    CifBddEdge edge3 = eventToEdge.get(event3);

                    if (DEBUG_REVERSIBLE) {
                        dbg.line("Reversible (" + evt1Name + ", " + evt2Name + "), trying event3 = " + evt3Name);
                        dbg.line("Reversible: event 2+1 done: %s", event21Done.toString());
                        dbg.line("Reversible: event 1+2 done: %s", event12Done.toString());
                        dbg.line("Reversible: edge3 guard: %s", edge3.guard.toString());
                        dbg.line("Reversible: edge3 update/guard: %s", edge3.updateGuard.toString());
                    }

                    // Check for reversible (events 2, 1, 3 versus event 1).
                    if (!event21Done.isZero()) {
                        BDD event21Enabled3 = event21Done.and(edge3.guard);
                        BDD event213Done = (event21Enabled3.isZero()) ? cifBddSpec.factory.zero()
                                : edge3.apply(event21Enabled3.id(), CifBddEdgeApplyDirection.FORWARD, null);

                        if (DEBUG_REVERSIBLE) {
                            dbg.line("Reversible: event2+1 enabled3: %s", event21Enabled3.toString());
                            dbg.line("Reversible: event2+1+3 done: %s", event213Done.toString());
                            dbg.line("Reversible: event1 done: %s", event1Done.toString());
                        }

                        event21Enabled3.free();

                        if (!event213Done.isZero() && event213Done.equals(event1Done)
                                && allStatesCovered(commonEnabledZeroStates, event1Done, cifBddSpec.varSetOld))
                        {
                            if (DEBUG_REVERSIBLE) {
                                dbg.line("reversible: event213Done != false && event213Done == event1Done -> "
                                        + "Found a match.");
                            }
                            foundReversible = true;

                            event213Done.free();
                            break;
                        }

                        event213Done.free();
                    }

                    if (termination.isRequested()) {
                        return null;
                    }

                    // Check for reversible (events 1, 2, 3 versus event 2).
                    if (!event12Done.isZero()) {
                        BDD event12Enabled3 = event12Done.and(edge3.guard);
                        BDD event123Done = (event12Enabled3.isZero()) ? cifBddSpec.factory.zero()
                                : edge3.apply(event12Enabled3.id(), CifBddEdgeApplyDirection.FORWARD, null);

                        if (DEBUG_REVERSIBLE) {
                            dbg.line("Reversible: event1+2 enabled3: %s", event12Enabled3.toString());
                            dbg.line("Reversible: event1+2+3 done: %s", event123Done.toString());
                            dbg.line("Reversible: event2 done: %s", event2Done.toString());
                        }

                        event12Enabled3.free();

                        if (!event123Done.isZero() && event123Done.equals(event2Done)
                                && allStatesCovered(commonEnabledZeroStates, event2Done, cifBddSpec.varSetOld))
                        {
                            if (DEBUG_REVERSIBLE) {
                                dbg.line("reversible: event123Done != false && event123Done == event2Done -> "
                                        + "Found a match.");
                            }
                            foundReversible = true;

                            event123Done.free();
                            break;
                        }

                        event123Done.free();
                    }

                    if (termination.isRequested()) {
                        return null;
                    }
                }

                commonEnabledGuards.free();
                commonEnabledZeroStates.free();
                event1Done.free();
                event2Done.free();
                event12Done.free();
                event21Done.free();

                if (foundReversible) {
                    reversibles.add(makeSortedPair(evt1Name, evt2Name));
                    if (DEBUG_GLOBAL) {
                        dbg.line("  -> event pair (" + evt1Name + ", " + evt2Name + ") is reversible.");
                    }

                    continue;
                }

                // None of the checks hold: failed to prove confluence.
                cannotProves.add(makeSortedPair(evt1Name, evt2Name));
                if (DEBUG_GLOBAL) {
                    dbg.line("  -> event pair (" + evt1Name + ", " + evt2Name + ") is CANNOT-PROVE.");
                }
            }
        }
        if (termination.isRequested()) {
            return null;
        }

        // Cleanup.
        zeroToOldVarRelations.free();

        // Dump results.
        boolean needEmptyLine = false;
        needEmptyLine = dumpMatches(mutualExclusives, "Mutual exclusive event pairs", out, needEmptyLine);
        needEmptyLine = dumpMatches(updateEquivalents, "Update equivalent event pairs", out, needEmptyLine);
        needEmptyLine = dumpMatches(independents, "Independent event pairs", out, needEmptyLine);
        needEmptyLine = dumpMatches(skippables, "Skippable event pairs", out, needEmptyLine);
        needEmptyLine = dumpMatches(reversibles, "Reversible event pairs", out, needEmptyLine);

        if (mutualExclusives.isEmpty() && updateEquivalents.isEmpty() && independents.isEmpty() && skippables.isEmpty()
                && reversibles.isEmpty())
        {
            dbg.line("No proven pairs.");
        }

        dbg.line();
        if (cannotProves.isEmpty()) {
            dbg.line("All pairs proven. Confluence holds.");
        } else {
            dbg.line("Some pairs unproven. Confluence may not hold.");
        }

        // Return check conclusion.
        return new ConfluenceCheckConclusion(cannotProves);
    }

    /**
     * Create a 'zero' variables to the 'old' variables identity relation: 'x0 = x and y0 = y and z0 = z and ...'.
     *
     * <p>
     * That is, create 'zero' variables, variables that hold the values of the variables before taking any transitions
     * (zero transitions taken). We add for each variable 'x' in the CIF/BDD specification an 'x0' variable, such that
     * we have 3 sets of variables ('x', 'x+' and 'x0'). The new variables are added to the BDD factory. Then, using the
     * existing 'x' variables and the new 'x0' variables, identity relation 'x0 = x and y0 = y and z0 = z and ...' is
     * created, and returned by this method.
     * </p>
     *
     * <p>
     * The reason for adding these 'zero' variables is as follows. To prove confluence, we consider pairs of events, and
     * their global guards and updates. We start with states where the guards of both edges are enabled. If there are
     * any (they are not mutually exclusive), we connect the 'zero' variables to the 'old' variables through an 'x0 = x
     * and y0 = y and z0 = z and ...' identity relation, and add this to the combined guard of the two edges. This
     * allows us to keep track of what values the variables had before taking any edges (the possible 'source states').
     * If we then apply edges to the combined guard predicate, the values of the 'x' variables may get updated, but the
     * 'x0' variables keep their values. This allows detection of different paths through the state space that lead to
     * the same set of states when considering all possible source states and target states for certain events, while
     * for specific source states the different paths lead to different target states. For instance, consider the
     * following example:
     * <ul>
     * <li>Event 'a' has guard 'true' and update 'x := not x'.</li>
     * <li>Event 'b' has guard 'true' and update 'x := x'.</li>
     * <li>Their combined guard is 'true'. If we don't consider the 'zero' variables, and apply the update of each edge
     * to predicate 'true', both edges result in set of target states 'true'. However, one edge inverts the value of
     * 'x', while the other keeps it the same. They thus don't have the same effect.</li>
     * <li>If we do consider 'zero' variables, we first change the combined guard 'true' to 'x0 = x'. Then we apply the
     * update of each edge, which gives us 'x0 = not x' and 'x0 = x'. In this case, we can detect that for different
     * source states the target states are different when applying these two edges.</li>
     * </ul>
     * </p>
     *
     * <p>
     * The 'x0' variables are added here, but they can't be removed after this check, since JavaBDD doesn't allow
     * removing variables from the BDD factory after they have been added. They thus remain after this check. However,
     * they are not used in predicates of other checks, so they are simply ignored there. It shouldn't impact the
     * representation of BDDs, nor the performance of the other checks, since these new variables are added 'on top' of
     * the existing ones (closer to the root of BDDs).
     * </p>
     *
     * @param cifBddSpec The CIF/BDD specification.
     * @return The zero variables to old variables identity relations.
     */
    private BDD createZeroToOldVarsRelations(CifBddSpec cifBddSpec) {
        // Create 'zero' variables and them to the BDD factory.
        Map<CifBddVariable, BDDDomain> domains0 = mapc(cifBddSpec.variables.length);
        for (CifBddVariable var: cifBddSpec.variables) {
            BigInteger size = var.domain.size();
            BDDDomain domain0 = cifBddSpec.factory.extDomain(size);
            Assert.areEqual(var.domain.varNum(), domain0.varNum());
            domains0.put(var, domain0);
        }
        Assert.areEqual(cifBddSpec.factory.varNum() % 3, 0);

        // Create and return the 'zero' variables to 'old' variables identity relations.
        return domains0.entrySet().stream().map(entry -> entry.getKey().domain.buildEquals(entry.getValue()))
                .reduce(cifBddSpec.factory.one(), BDD::andWith);
    }

    /**
     * Check whether that the given 'result' states cover all given 'zero' states.
     *
     * <p>
     * This check is to make sure that no source states are lost along the way, by taking transitions. That is, if some
     * path of transitions can only be executed from certain states, while another path can be executed from different
     * states, then we may not correctly detect confluence for all source states. As an example of this, consider the
     * following:
     * </p>
     * <pre>
     * controllable a, b;
     *
     * supervisor A:
     *   location p:
     *     initial;
     *     edge a do z := true goto q;
     *     edge b do v := 2 goto q;
     *
     *   location q:
     *     edge a do z := true goto s;
     *     edge b do v := 2 goto s;
     *
     *   location s:
     *     marked;
     * </pre>
     * <p>
     * Then:
     * <ul>
     * <li>The updates are independent in relation to 'z' and 'v'.</li>
     * <li>If we first execute 'a' and then 'b', it must be the case that we started in 'p' and went to 'q' with 'a',
     * and then went to 's' with 'b'. We can compute the target states reached after taking 'a' and then 'b', but given
     * that we could only start in 'p', this doesn't give us the information for all possible source states.</li>
     * </ul>
     * This method detects such cases, such that we can prevent deciding confluence for them.
     * </p>
     *
     * @param commonEnabledZeroStates The 'zero' states, the source states where the combined guards holds, expressed in
     *     terms of 'zero' variables.
     * @param resultStates The 'result' states, the target states for one of the confluence pattern checks, as a
     *     relation between surviving 'zero' states and the target 'old' states.
     * @param varSetOld The BDD variable set containing all old variables.
     * @return Whether the {@code resultStates} cover all {@code commonEnabledZeroStates}, and thus no 'zero' states are
     *     lost along the way.
     */
    private boolean allStatesCovered(BDD commonEnabledZeroStates, BDD resultStates, BDDVarSet varSetOld) {
        // Compute the set of 'zero' states that are associated with the target 'old' states.
        BDD zeroResultStates = resultStates.exist(varSetOld);

        // Check whether the target 'zero' states cover all source 'zero' states.
        boolean result = commonEnabledZeroStates.equals(zeroResultStates);
        zeroResultStates.free();
        return result;
    }

    /**
     * Construct an event pair where its members are alphabetically sorted.
     *
     * @param evt1Name First name to use.
     * @param evt2Name Second name to use.
     * @return Pair with the names in alphabetical order.
     */
    private Pair<String, String> makeSortedPair(String evt1Name, String evt2Name) {
        if (SORTER.compare(evt1Name, evt2Name) < 0) {
            return new Pair<>(evt1Name, evt2Name);
        } else {
            return new Pair<>(evt2Name, evt1Name);
        }
    }

    /**
     * Output the collection of event pairs with the stated reason.
     *
     * @param pairs Event pairs to output.
     * @param reasonText Description of the reason what the collection means.
     * @param out Callback to send normal output to the user.
     * @param needEmptyLine Whether an empty line is needed if matches are dumped by this method.
     * @return Whether an empty line is needed if matches are dumped after this method.
     */
    private boolean dumpMatches(List<Pair<String, String>> pairs, String reasonText, DebugNormalOutput out,
            boolean needEmptyLine)
    {
        // Nothing to do if no pairs. Preserves the need for an empty line for the next dump.
        if (pairs.isEmpty()) {
            return needEmptyLine;
        }

        // Sort the pairs for easier reading.
        pairs.sort(
                Comparator.comparing((Pair<String, String> p) -> p.left, SORTER).thenComparing(p -> p.right, SORTER));

        // Dump the pairs.
        if (needEmptyLine) {
            out.line();
        }
        out.line(reasonText + ":");
        out.inc();
        out.line(pairs.stream().map(Pair::toString).collect(Collectors.joining(", ")));
        out.dec();
        return true;
    }
}
