//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.conversion.bitvectors;

import org.eclipse.escet.common.java.Assert;

import com.github.javabdd.BDD;
import com.github.javabdd.BDDDomain;
import com.github.javabdd.BDDFactory;

/**
 * Two's complement BDD bit vector.
 *
 * @see <a href="https://en.wikipedia.org/wiki/Two%27s_complement">Two&apos;s complement (Wikipedia)</a>
 */
public class TwosComplementCifBddBitVector
        extends CifBddBitVector<TwosComplementCifBddBitVector, TwosComplementCifBddBitVectorAndCarry>
{
    /** The minimum length (in number of bits) of any {@link TwosComplementCifBddBitVector}. */
    public static final int MINIMUM_LENGTH = 2;

    /**
     * Constructor for the {@link TwosComplementCifBddBitVector} class.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @throws IllegalArgumentException If the length is less than two.
     */
    private TwosComplementCifBddBitVector(BDDFactory factory, int length) {
        super(factory, length);
    }

    @Override
    protected int getMinimumLength() {
        return MINIMUM_LENGTH;
    }

    /**
     * Returns the minimum length (in number of bits) needed to represent the given integer value as a
     * {@link TwosComplementCifBddBitVector}.
     *
     * @param value The integer value.
     * @return The minimum length.
     */
    public static int getMinimumLength(int value) {
        if (value >= 0) {
            // Needs extra sign bit, besides the number of bits required for the unsigned bit vector. At least two
            // bits are required.
            return Math.max(2, 1 + UnsignedCifBddBitVector.getMinimumLength(value));
        } else {
            // Requires sign bit, besides number of bits to represent the inverse, while accounting for the fact that we
            // can represent one more negative number than we can positive ones. At least two bits are required.
            return Math.max(2, 1 + UnsignedCifBddBitVector.getMinimumLength(-(value + 1)));
        }
    }

    @Override
    protected TwosComplementCifBddBitVector createEmpty(int length) {
        return new TwosComplementCifBddBitVector(this.factory, length);
    }

    /**
     * Creates a {@link TwosComplementCifBddBitVector}. Initializes the bits of the bit vector to 'false'.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @return The created bit vector.
     * @throws IllegalArgumentException If the length is less than two.
     */
    public static TwosComplementCifBddBitVector create(BDDFactory factory, int length) {
        return create(factory, length, false);
    }

    /**
     * Creates a {@link TwosComplementCifBddBitVector}. Initializes each bit of the bit vector to the given boolean
     * value.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @param value The value to use for each bit.
     * @return The created bit vector.
     * @throws IllegalArgumentException If the length is less than two.
     */
    public static TwosComplementCifBddBitVector create(BDDFactory factory, int length, boolean value) {
        // Create.
        TwosComplementCifBddBitVector vector = new TwosComplementCifBddBitVector(factory, length);

        // Initialize.
        for (int i = 0; i < vector.bits.length; i++) {
            vector.bits[i] = value ? factory.one() : factory.zero();
        }

        // Return.
        return vector;
    }

    /**
     * Creates a {@link TwosComplementCifBddBitVector} from an integer value. Initializes the bits of the bit vector to
     * the given integer value. Uses an as small as possible bit vector to represent the integer value.
     *
     * @param factory The BDD factory to use.
     * @param value The integer value to represent using a bit vector.
     * @return The created bit vector.
     */
    public static TwosComplementCifBddBitVector createFromInt(BDDFactory factory, int value) {
        int length = getMinimumLength(value);
        return createFromInt(factory, length, value);
    }

    /**
     * Creates a {@link TwosComplementCifBddBitVector} from an integer value. Initializes the bits of the bit vector to
     * the given integer value, creating a bit vector of the given length. If the requested length is larger than the
     * needed number of bits, the remaining/highest bits are 'false' if the value is non-negative and 'true' if the
     * value is negative.
     *
     * @param factory The BDD factory to use.
     * @param length The number of bits of the bit vector.
     * @param value The integer value to represent using a bit vector.
     * @return The created bit vector.
     * @throws IllegalArgumentException If the length is insufficient to store the given value.
     */
    public static TwosComplementCifBddBitVector createFromInt(BDDFactory factory, int length, int value) {
        // Precondition check.
        if (length < getMinimumLength(value)) {
            throw new IllegalArgumentException("Length is insufficient.");
        }

        // Create.
        TwosComplementCifBddBitVector vector = new TwosComplementCifBddBitVector(factory, length);

        // Initialize.
        for (int i = 0; i < vector.bits.length; i++) {
            vector.bits[i] = ((value & 0x1) != 0) ? factory.one() : factory.zero();
            value >>= 1;
        }
        Assert.check(value == 0 || value == -1, value); // All zeros or all ones.

        // Return.
        return vector;
    }

    /**
     * Creates a {@link TwosComplementCifBddBitVector} from a non-empty {@link BDDDomain}. Initializes the bits of the
     * bit vector to the variables of the given domain. The bit vector is one bit longer than the number of variables in
     * the given domain, to account for the sign bit, which is always 'false' as BDD domains represent non-negative
     * values.
     *
     * @param domain The domain to use.
     * @return The created bit vector.
     * @throws IllegalArgumentException If the domain is empty.
     */
    public static TwosComplementCifBddBitVector createFromDomain(BDDDomain domain) {
        // Precondition check.
        int varCnt = domain.varNum();
        if (varCnt == 0) {
            throw new IllegalArgumentException("Domain is empty.");
        }

        // Create bit vector.
        int length = varCnt + 1;
        TwosComplementCifBddBitVector vector = new TwosComplementCifBddBitVector(domain.getFactory(), length);

        // Initialize non-sign bits.
        int[] vars = domain.vars();
        for (int i = 0; i < vars.length; i++) {
            vector.bits[i] = vector.factory.ithVar(vars[i]);
        }

        // Set sign bit to 'false' (domains represent non-negative values).
        vector.bits[length - 1] = vector.factory.zero();

        // Return.
        return vector;
    }

    /**
     * Creates a {@link TwosComplementCifBddBitVector} from an {@link UnsignedCifBddBitVector}. The two's complement bit
     * vector is one bit longer than the unsigned bit vector, to account for the sign bit, which is always 'false' as
     * unsigned bit vectors in all cases represent non-negative values.
     *
     * @param unsignedVector The unsigned bit vector to use.
     * @return The created two's complement bit vector.
     */
    public static TwosComplementCifBddBitVector createFromUnsignedBitVector(UnsignedCifBddBitVector unsignedVector) {
        // Create.
        TwosComplementCifBddBitVector twosComplementVector = new TwosComplementCifBddBitVector(unsignedVector.factory,
                unsignedVector.length() + 1);

        // Initialize non-sign bits.
        for (int i = 0; i < unsignedVector.length(); i++) {
            twosComplementVector.bits[i] = unsignedVector.getBit(i);
        }

        // Set sign bit to 'false' (no negative values).
        twosComplementVector.bits[twosComplementVector.bits.length - 1] = twosComplementVector.factory.zero();

        // Return.
        return twosComplementVector;
    }

    @Override
    public Integer getInt() {
        // Check for enough room to represent the value.
        if (bits.length > 32) {
            throw new IllegalStateException("More than 32 bits in vector.");
        }

        // Return value.
        Long value = getLong();
        return (value == null) ? null : (int)(long)value;
    }

    @Override
    public Long getLong() {
        // Check for enough room to represent the value.
        if (bits.length > 64) {
            throw new IllegalStateException("More than 64 bits in vector.");
        }

        // Initialize all result bits with the sign bit.
        long value;
        int bitIndex = bits.length - 1;
        if (bits[bitIndex].isOne()) {
            // Negative value: set all bits to '1'.
            value = -1;
        } else if (bits[bitIndex].isZero()) {
            // Non-negative value: set all bits to '0'.
            value = 0;
        } else {
            // Not a constant value.
            return null;
        }
        bitIndex--;

        // Shift remaining bits in from the right.
        for (; bitIndex >= 0; bitIndex--) {
            if (bits[bitIndex].isOne()) {
                // Shift already-considered higher bits. Add current '1' bit.
                value = (value << 1) | 1;
            } else if (bits[bitIndex].isZero()) {
                // Shift already-considered higher bits. Nothing to add for '0' bit.
                value = (value << 1);
            } else {
                // Not a constant value.
                return null;
            }
        }

        // Return the value.
        return value;
    }

    @Override
    public void setInt(int value) {
        // Precondition check.
        if (bits.length < getMinimumLength(value)) {
            throw new IllegalArgumentException("Length is insufficient.");
        }

        // Set value.
        for (int i = 0; i < bits.length; i++) {
            bits[i].free();
            bits[i] = ((value & 0x1) != 0) ? factory.one() : factory.zero();
            value >>= 1;
        }
        Assert.check(value == 0 || value == -1, value); // All zeros or all ones.
    }

    @Override
    public void setDomain(BDDDomain domain) {
        // Precondition checks.
        int varCnt = domain.varNum();
        if (varCnt == 0) {
            throw new IllegalArgumentException("Domain is empty.");
        }
        if (varCnt > bits.length - 1) { // The '- 1' accounts for the sign bit.
            throw new IllegalArgumentException("Domain doesn't fit.");
        }

        // Set domain.
        int[] vars = domain.vars();
        Assert.areEqual(varCnt, vars.length);
        int i = 0;
        for (; i < varCnt; i++) {
            bits[i].free();
            bits[i] = factory.ithVar(vars[i]);
        }
        for (; i < bits.length; i++) {
            bits[i].free();
            bits[i] = factory.zero();
        }
    }

    /**
     * {@inheritDoc}
     *
     * <p>
     * For two's complement bit vectors, the additional (most significant) bits are set to the sign bit.
     * </p>
     *
     * @throws IllegalArgumentException If the new length is less than two.
     */
    @Override
    public void resize(int length) {
        // Optimization.
        if (length == bits.length) {
            return;
        }

        // Precondition check.
        if (length < 2) {
            throw new IllegalArgumentException("Length is less than two.");
        }

        // Allocate new bits.
        BDD[] newBits = new BDD[length];

        // Copy the common bits.
        int numberOfCommonBits = Math.min(bits.length, length);
        System.arraycopy(bits, 0, newBits, 0, numberOfCommonBits);

        // If new length is larger, set additional bits to the sign bit.
        BDD signBit = bits[bits.length - 1];
        for (int i = numberOfCommonBits; i < length; i++) {
            newBits[i] = signBit.id();
        }

        // If new length is smaller, free dropped bits.
        for (int i = numberOfCommonBits; i < bits.length; i++) {
            bits[i].free();
        }

        // Replace the bits.
        bits = newBits;
    }

    @Override
    public TwosComplementCifBddBitVectorAndCarry negate() {
        // To negate a two's complement bit vector, we negate each bit and perform an addition with 'one'. This
        // implementation inlines the 'one' addition and merges negating the bits with the addition, for better
        // performance.

        // Compute resulting vector.
        TwosComplementCifBddBitVector rslt = new TwosComplementCifBddBitVector(factory, bits.length);
        BDD carry = factory.one(); // Add 'one' using the carry-in of bit 0.
        BDD lastCarry = factory.zero();
        BDD beforeLastCarry = factory.zero();
        for (int i = 0; i < bits.length; i++) {
            // notThisI = !this[i]
            BDD notThisI = this.bits[i].not();

            // rslt[i] = !this[i] ^ 0 ^ carry = ![this] ^ carry = notThisI ^ carry
            rslt.bits[i] = notThisI.id().xorWith(carry.id());

            // carry = (!this[i] & 0) | (carry & (!this[i] | 0)) = carry & !this[i] = carry & notThisI
            carry = carry.andWith(notThisI);

            // Save the last two carry bits.
            if (i == bits.length - 1) {
                lastCarry = carry.id();
            } else if (i == bits.length - 2) {
                beforeLastCarry = carry.id();
            }
        }

        // Compute resulting carry.
        carry.free();
        carry = beforeLastCarry.xorWith(lastCarry);

        // Return the result.
        return new TwosComplementCifBddBitVectorAndCarry(rslt, carry);
    }

    @Override
    public TwosComplementCifBddBitVectorAndCarry abs() {
        // Compute 'this < 0'. We can use the sign bit, as it is 'true' when the value is negative and 'false' when it
        // is non-negative.
        BDD cmp = bits[bits.length - 1];

        // Compute '-this', including overflow.
        TwosComplementCifBddBitVectorAndCarry negated = negate();
        BDD overflow = negated.carry;

        // Compute result vector 'if this < 0: -this else this end'.
        TwosComplementCifBddBitVector resultVector = negated.vector.ifThenElse(this, cmp);
        negated.vector.free();

        // Return the result.
        return new TwosComplementCifBddBitVectorAndCarry(resultVector, overflow);
    }

    @Override
    public TwosComplementCifBddBitVectorAndCarry add(TwosComplementCifBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute resulting vector.
        TwosComplementCifBddBitVector rslt = new TwosComplementCifBddBitVector(factory, bits.length);
        BDD carry = factory.zero();
        BDD lastCarry = factory.zero();
        BDD beforeLastCarry = factory.zero();
        for (int i = 0; i < bits.length; i++) {
            // rslt[i] = this[i] ^ other[i] ^ carry
            rslt.bits[i] = this.bits[i].xor(other.bits[i]).xorWith(carry.id());

            // carry = (this[i] & other[i]) | (carry & (this[i] | other[i]))
            carry = this.bits[i].and(other.bits[i]).orWith(carry.andWith(this.bits[i].or(other.bits[i])));

            // Save the last two carry bits.
            if (i == bits.length - 1) {
                lastCarry = carry.id();
            } else if (i == bits.length - 2) {
                beforeLastCarry = carry.id();
            }
        }

        // Compute resulting carry.
        carry.free();
        carry = beforeLastCarry.xorWith(lastCarry);

        // Return the result.
        return new TwosComplementCifBddBitVectorAndCarry(rslt, carry);
    }

    @Override
    public TwosComplementCifBddBitVectorAndCarry subtract(TwosComplementCifBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute resulting vector.
        TwosComplementCifBddBitVector rslt = new TwosComplementCifBddBitVector(factory, bits.length);
        BDD carry = factory.zero();
        BDD lastCarry = factory.zero();
        BDD beforeLastCarry = factory.zero();
        for (int i = 0; i < bits.length; i++) {
            // rslt[i] = this[i] ^ other[i] ^ carry
            rslt.bits[i] = this.bits[i].xor(other.bits[i]).xorWith(carry.id());

            // carry = (this[n] & other[n] & carry) | (!this[n] & (other[n] | carry))
            BDD tmp1 = other.bits[i].or(carry);
            BDD tmp2 = this.bits[i].apply(tmp1, BDDFactory.less);
            tmp1.free();
            carry = this.bits[i].and(other.bits[i]).andWith(carry).orWith(tmp2);

            // Save the last two carry bits.
            if (i == bits.length - 1) {
                lastCarry = carry.id();
            } else if (i == bits.length - 2) {
                beforeLastCarry = carry.id();
            }
        }

        // Compute resulting carry.
        carry.free();
        carry = beforeLastCarry.xorWith(lastCarry);

        // Return the result.
        return new TwosComplementCifBddBitVectorAndCarry(rslt, carry);
    }

    /**
     * {@inheritDoc}
     *
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalArgumentException If the highest bit of the {@link #abs absolute value} of this vector is not
     *     'false', and the highest two bits of the divisor are not 'false' (based on the length of this vector).
     */
    @Override
    public TwosComplementCifBddBitVector div(int divisor) {
        return divmod(divisor, true);
    }

    /**
     * {@inheritDoc}
     *
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalStateException If the highest bit of the {@link #abs absolute value} of this bit vector is not
     *     'false'.
     */
    @Override
    public TwosComplementCifBddBitVector mod(int divisor) {
        return divmod(divisor, false);
    }

    /**
     * Computes the quotient ('div' result) or remainder ('mod' result) of dividing this vector (the dividend) by the
     * given value (the divisor). This operation returns a new bit vector. The bit vector on which the operation is
     * performed is not modified or {@link #free freed}.
     *
     * @param divisor The value by which to divide this bit vector.
     * @param isDiv Whether to compute and return the quotient/'div' ({@code true}) or remainder/'mod' ({@code false}).
     * @return The quotient ('div' result) or remainder ('mod' result).
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalArgumentException If the quotient/'div' is computed, and the highest bit of the {@link #abs
     *     absolute value} of this vector is not 'false', and the highest two bits of the divisor are not 'false' (based
     *     on the length of this vector).
     * @throws IllegalStateException If the remainder/'mod' is computed, and the highest bit of the {@link #abs absolute
     *     value} of this bit vector is not 'false'.
     */
    private TwosComplementCifBddBitVector divmod(int divisor, boolean isDiv) {
        // Get absolute value.
        TwosComplementCifBddBitVectorAndCarry absThis = abs();
        absThis.carry.free();

        // Compute unsigned 'div' or 'mod' result on absolute value.
        TwosComplementCifBddBitVector resultUnsigned = absThis.vector.divmodUnsigned(divisor, isDiv);
        absThis.vector.free();

        // Compute signed 'div' or 'mod' result: 'if signBit: -resultUnsigned else resultUnsigned end'.
        BDD signBit = bits[bits.length - 1];
        TwosComplementCifBddBitVectorAndCarry resultUnsignedNeg = resultUnsigned.negate();
        TwosComplementCifBddBitVector resultSigned = resultUnsignedNeg.vector.ifThenElse(resultUnsigned, signBit);
        resultUnsigned.free();
        resultUnsignedNeg.vector.free();
        resultUnsignedNeg.carry.free();

        // Return signed 'div' or 'mod' result.
        return resultSigned;
    }

    /**
     * Computes the quotient ('div' result) or remainder ('mod' result) of dividing this vector (the dividend) by the
     * given value (the divisor), considering the bit vectors as being unsigned. This operation returns a new bit
     * vector. The bit vector on which the operation is performed is not modified or {@link #free freed}.
     *
     * @param divisorValue The value by which to divide this bit vector.
     * @param isDiv Whether to compute and return the quotient/'div' ({@code true}) or remainder/'mod' ({@code false}).
     * @return The quotient ('div' result) or remainder ('mod' result).
     * @throws IllegalArgumentException If the divisor is not positive.
     * @throws IllegalArgumentException If the divisor doesn't fit within this bit vector.
     * @throws IllegalArgumentException If the quotient/'div' is computed, and the highest bit of this bit vector is not
     *     'false', and the highest two bits of the divisor are not 'false' (based on the length of this vector).
     * @throws IllegalStateException If the remainder/'mod' is computed, and the highest bit of this bit vector is not
     *     'false'.
     */
    private TwosComplementCifBddBitVector divmodUnsigned(int divisorValue, boolean isDiv) {
        // Precondition checks.
        if (divisorValue <= 0) {
            throw new IllegalArgumentException("Divisor is not positive.");
        }
        if (bits.length < getMinimumLength(divisorValue)) {
            throw new IllegalArgumentException("Divisor doesn't fit.");
        }

        // Get divisor vector.
        TwosComplementCifBddBitVector divisor = createFromInt(factory, bits.length, divisorValue);

        // More precondition checks.
        if (isDiv && !bits[bits.length - 1].isZero()
                && (!divisor.bits[divisor.length() - 1].isZero() || !divisor.bits[divisor.length() - 2].isZero()))
        {
            throw new IllegalArgumentException("Computing the quotient/'div', the highest bit of the dividend's "
                    + "absolute value is not 'false', and the highest two bits of the divisor are not both "
                    + "'false'.");
        }
        if (!isDiv && !bits[bits.length - 1].isZero()) {
            throw new IllegalStateException("Computing the remainder/'mod', and the highest bit of the dividend's "
                    + "absolute value is not 'false'.");
        }

        // Create result vectors.
        TwosComplementCifBddBitVector quotient = shiftLeft(1, factory.zero());
        TwosComplementCifBddBitVector remainderZero = create(factory, bits.length);
        TwosComplementCifBddBitVector remainder = remainderZero.shiftLeft(1, bits[bits.length - 1]);
        remainderZero.free();

        // Compute result.
        divModUnsignedRecursive(divisor, quotient, remainder, bits.length);
        divisor.free();

        // Return requested result.
        if (isDiv) {
            remainder.free();
            return quotient;
        } else {
            quotient.free();
            TwosComplementCifBddBitVector shiftedRemainder = remainder.shiftRight(1, factory.zero());
            remainder.free();
            return shiftedRemainder;
        }
    }

    /**
     * Computes the quotient ('div' result) and remainder ('mod' result) of dividing a bit vector (the dividend) by the
     * given other bit vector (the divisor), considering the bit vectors as being unsigned.
     *
     * @param divisor The divisor bit vector. Is not modified.
     * @param quotient The quotient/'div' bit vector, as computed so far. Is modified in-place.
     * @param remainder The remainder/'mod' bit vector, as computed so far. Is modified in-place.
     * @param step The number of steps to perform.
     */
    private void divModUnsignedRecursive(TwosComplementCifBddBitVector divisor, TwosComplementCifBddBitVector quotient,
            TwosComplementCifBddBitVector remainder, int step)
    {
        int divLen = divisor.bits.length;
        BDD isSmaller = divisor.lessOrEqual(remainder);
        TwosComplementCifBddBitVector newQuotient = quotient.shiftLeft(1, isSmaller);

        TwosComplementCifBddBitVector sub = create(factory, divLen);
        for (int i = 0; i < divLen; i++) {
            sub.bits[i] = isSmaller.ite(divisor.bits[i], factory.zero());
        }

        TwosComplementCifBddBitVectorAndCarry tmp = remainder.subtract(sub);
        TwosComplementCifBddBitVector newRemainder = tmp.vector.shiftLeft(1, quotient.bits[divLen - 1]);

        if (step > 1) {
            divModUnsignedRecursive(divisor, newQuotient, newRemainder, step - 1);
        }

        tmp.vector.free();
        tmp.carry.free();
        sub.free();
        isSmaller.free();

        quotient.replaceBy(newQuotient);
        remainder.replaceBy(newRemainder);
    }

    @Override
    public BDD lessThan(TwosComplementCifBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute carry.
        BDD carry = factory.zero();
        for (int i = 0; i < bits.length; i++) {
            // carry = (!this[i] & other[i]) | biimp(this[i], other[i]) & carry
            BDD lt = this.bits[i].apply(other.bits[i], BDDFactory.less);
            BDD eq = this.bits[i].biimp(other.bits[i]);
            carry = lt.orWith(eq.andWith(carry));
        }

        // Compute result: (this[n-1] & !other[n-1]) | ((this[n-1] | !other[n-1]) & carry)
        BDD thisSignBit = this.bits[this.bits.length - 1];
        BDD otherSignBit = other.bits[other.bits.length - 1];
        BDD lhs = thisSignBit.id().andWith(otherSignBit.not());
        BDD rhs = thisSignBit.id().orWith(otherSignBit.not()).andWith(carry);
        return lhs.orWith(rhs);
    }

    @Override
    public BDD lessOrEqual(TwosComplementCifBddBitVector other) {
        // Precondition check.
        if (this.bits.length != other.bits.length) {
            throw new IllegalArgumentException("Different lengths.");
        }

        // Compute carry.
        BDD carry = factory.one();
        for (int i = 0; i < bits.length; i++) {
            // carry = (!this[i] & other[i]) | biimp(this[i], other[i]) & carry
            BDD lt = this.bits[i].apply(other.bits[i], BDDFactory.less);
            BDD eq = this.bits[i].biimp(other.bits[i]);
            carry = lt.orWith(eq.andWith(carry));
        }

        // Compute result: (this[n-1] & !other[n-1]) | ((this[n-1] | !other[n-1]) & carry)
        BDD thisSignBit = this.bits[this.bits.length - 1];
        BDD otherSignBit = other.bits[other.bits.length - 1];
        BDD lhs = thisSignBit.id().andWith(otherSignBit.not());
        BDD rhs = thisSignBit.id().orWith(otherSignBit.not()).andWith(carry);
        return lhs.orWith(rhs);
    }
}
