/*******************************************************************************
 * Copyright (c) 2011 Draeger Medical GmbH (http://www.draeger.com).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * CONTRIBUTORS:
 * 		Peter Karlitschek (initial contribution)
 * 
 *******************************************************************************/

package org.eclipse.etrice.generator.cpp.gen

import com.google.inject.Inject
import com.google.inject.Singleton
import java.util.ArrayList
import org.eclipse.etrice.core.genmodel.etricegen.Root
import org.eclipse.etrice.core.genmodel.fsm.base.ILogger
import org.eclipse.etrice.core.room.CommunicationType
import org.eclipse.etrice.core.room.DataClass
import org.eclipse.etrice.core.room.InterfaceItem
import org.eclipse.etrice.core.room.Message
import org.eclipse.etrice.core.room.Port
import org.eclipse.etrice.core.room.PortClass
import org.eclipse.etrice.core.room.PrimitiveType
import org.eclipse.etrice.core.room.ProtocolClass
import org.eclipse.etrice.core.room.SAP
import org.eclipse.etrice.core.room.SPP
import org.eclipse.etrice.generator.cpp.Main
import org.eclipse.etrice.generator.generic.GenericProtocolClassGenerator
import org.eclipse.etrice.generator.generic.ProcedureHelpers
import org.eclipse.etrice.generator.generic.RoomExtensions
import org.eclipse.etrice.generator.generic.TypeHelpers
import org.eclipse.xtext.generator.JavaIoFileSystemAccess

/**
 * @author Peter Karlitschek
 *
 */
@Singleton
class ProtocolClassGen extends GenericProtocolClassGenerator {

	@Inject extension JavaIoFileSystemAccess fileAccess
	@Inject extension CppExtensions stdExt
	@Inject extension RoomExtensions roomExt
	@Inject extension ProcedureHelpers helpers
	@Inject extension TypeHelpers
	@Inject extension Initialization
	@Inject ILogger logger
	
	def doGenerate(Root root) { 
		for (pc: root.usedProtocolClasses) {
			var path = pc.generationTargetPath+pc.getPath

			logger.logInfo("generating ProtocolClass header '"+pc.getCppHeaderFileName+"' in '"+path+"'")
			fileAccess.setOutputPath(path)
			fileAccess.generateFile(pc.getCppHeaderFileName, root.generateHeaderFile(pc))

			logger.logInfo("generating ProtocolClass source '"+pc.getCppSourceFileName+"' in '"+path+"'")
			fileAccess.setOutputPath(path)
			fileAccess.generateFile(pc.getCppSourceFileName, root.generateSourceFile(pc))
		}
	}

	
	def private generateHeaderFile(Root root, ProtocolClass pc) {'''
		/**
		 * @author generated by eTrice
		 *
		 * Header File of ProtocolClass pc.name
		 * 
		 */

		generateIncludeGuardBegin(pc.name)
		
		#include "platforms/generic/etDatatypes.h"
		#include "common/messaging/IRTObject.h"
		#include "common/modelbase/PortBase.h"
		#include "common/modelbase/InterfaceItemBase.h"
		#include "common/messaging/Address.h"
		#include "common/messaging/Message.h"
		#include <vector>
		#include <string>
				
		namespace etRuntime {
			class IEventReceiver;
			
		}
		
		helpers.userCode(pc.userCode1)
		
		FOR dataClass : root.getReferencedDataClasses(pc)
		#include "dataClass.pathdataClass.name.h"
		ENDFOR

		class pc.name {
		   public:
		IF pc.commType==CommunicationType::EVENT_DRIVEN	   /* message IDs */
				genMessageIDs(pc)
				static bool isValidEvtID(int evtId) {
					return ((MSG_MIN < evtId) && (evtId < MSG_MAX));
				};
				static bool isValidOutgoingEvtID(int evtId) {
					return ((MSG_MIN < evtId) && (evtId < IF pc.incomingMessages.size == 0MSG_MAXELSEIN_pc.incomingMessages.get(0).nameENDIF));
				};
				static bool isValidIncomingEvtID(int evtId) {
					return ((IF pc.incomingMessages.size == 0MSG_MAXELSEIN_pc.incomingMessages.get(0).nameENDIF <= evtId) && (evtId < MSG_MAX));
				};
				static std::string getMessageString(int msg_id);
			
			private:
				static std::string s_messageStrings[];
		ENDIF
				helpers.userCode(pc.userCode2)
		};
		
		portClassDeclaration(pc, false)
		portClassDeclaration(pc, true)
		generateIncludeGuardEnd(pc.name)
	'''
	}
	
	def private portClassDeclaration(ProtocolClass pc, Boolean conj) {
		var pclass = pc.getPortClass(conj)
		var portClassName = pc.getPortClassName(conj)
		var replPortClassName = pc.getPortClassName(conj, true)
	'''
	//------------------------------------------------------------------------------------------------------------
	// IF conjconjugated ENDIFport class
	//------------------------------------------------------------------------------------------------------------
	class portClassName : public etRuntime::PortBase {
		IF pclass!=null
			helpers.userCode(pclass.userCode)
		ENDIF
	   public:
		// constructors
		 portClassName(etRuntime::IEventReceiver& actor, etRuntime::IRTObject* parent, std::string name, int localId, etRuntime::Address addr, etRuntime::Address peerAddress, bool doRegistration = true); 
		 portClassName(etRuntime::IEventReceiver& actor, etRuntime::IRTObject* parent, std::string name, int localId, int idx, etRuntime::Address addr, etRuntime::Address peerAddress, bool doRegistration = true);
	
		 virtual void receive(etRuntime::Message* m);
		IF pclass!=null
			helpers.attributes(pclass.attributes)
			helpers.operationsDeclaration(pclass.operations, portClassName)
		ENDIF
		
		  // outgoing messages
		FOR m : pc.getAllMessages(conj)
		  	sendMessageDeclaration(m,conj)
		ENDFOR
	};
	
	//------------------------------------------------------------------------------------------------------------
	// IF conjconjugated ENDIFreplicated port class
	//------------------------------------------------------------------------------------------------------------
	class replPortClassName {
		private:
		    int m_replication;
		    portClassName* m_ports;  //dynamic array used instead of vector to avoid copy construction
	
		public:
			replPortClassName(etRuntime::IEventReceiver& actor, etRuntime::IRTObject* parent, std::string name, int localId, std::vector<etRuntime::Address> addr, std::vector<etRuntime::Address> peerAddress);
			virtual ~replPortClassName() {};
			
			int getReplication() {	return m_replication; }
			int getIndexOf(const etRuntime::InterfaceItemBase& ifitem){ return ifitem.getIdx();	}
			portClassName get(int i) {return m_ports[i];}
			
			IF pc.commType==CommunicationType::EVENT_DRIVEN
				 // outgoing messages
				FOR m : pc.getAllMessages(conj)
				  	sendMessageDeclaration(m,conj)
				ENDFOR
			ENDIF
			
	};
	'''
	}

	def private generateSourceFile(Root root, ProtocolClass pc) {'''
		/**
		 * @author generated by eTrice
		 *
		 * Source File of ProtocolClass pc.name
		 * 
		 */

		#include "pc.getCppHeaderFileName"
		#include "common/debugging/DebuggingService.h"
		#include <iostream>
		IF Main::settings.isUseEtUnit
			extern "C" {
				#include "etUnit.h"
			}
		ENDIF

		using namespace etRuntime;
		
		helpers.userCode(pc.userCode3)
		
	IF pc.commType==CommunicationType::EVENT_DRIVEN
		helpers.userCode(pc.userCode2)
	
		portClassImplementation(pc, false)
		portClassImplementation(pc, true)
		
		/*--------------------- debug helpers */
		generateDebugHelpersImplementation(root, pc)
	ENDIF
		
		
	'''
	}
	

	def portClassImplementation(ProtocolClass pc, Boolean conj) {
		var pclass = pc.getPortClass(conj)
		var portClassName = pc.getPortClassName(conj)
		var replPortClassName = pc.getPortClassName(conj, true)
	'''
	//------------------------------------------------------------------------------------------------------------
	// IF conjconjugated ENDIFport class
	//------------------------------------------------------------------------------------------------------------
	
	portClassName::portClassName(etRuntime::IEventReceiver& actor, etRuntime::IRTObject* parent, std::string name, int localId, Address addr, Address peerAddress, bool doRegistration)
		: pclass.generateConstructorInitalizerList("0")
	{
		IF pclass!=nullpclass.attributes.attributeInitialization(false)ENDIF
		if (doRegistration) {
			DebuggingService::getInstance().addPortInstance(*this);
		}
	}

	portClassName::portClassName(etRuntime::IEventReceiver& actor, etRuntime::IRTObject* parent, std::string name, int localId, int idx, Address addr, Address peerAddress, bool doRegistration)
		: pclass.generateConstructorInitalizerList("idx")
	{
		IF pclass!=nullpclass.attributes.attributeInitialization(false)ENDIF
		if (doRegistration) {
			DebuggingService::getInstance().addPortInstance(*this);
		}
	}
		
	void portClassName::receive(Message* msg) {
		if (! pc.name::IF conjisValidOutgoingEvtIDELSEisValidIncomingEvtIDENDIF(msg->getEvtId())) {
			std::cout << "unknown" << std::endl;
		}
		else {
			if (msg->hasDebugFlagSet()) {			// TODO: model switch for activation of this flag
				DebuggingService::getInstance().addMessageAsyncIn(getPeerAddress(), getAddress(), pc.name::getMessageString(msg->getEvtId()));
			}
			
			IF pc.handlesReceive(conj)
			switch (msg->getEvtId()) {
				FOR hdlr : pc.getReceiveHandlers(conj)
				case pc.name::hdlr.msg.getCodeName():
					{
						FOR command : hdlr.detailCode.lines
						command
						ENDFOR
					}
					break;
				ENDFOR
				default:
			ENDIF
					getEventReceiver().receiveEvent(this, msg->getEvtId(),	msg->getData());
			IF pc.handlesReceive(conj)
					break;
			}
			ENDIF
		}
	};

	IF pclass!=null
		helpers.operationsImplementation(pclass.operations, portClassName)
	ENDIF
			
	// sent messages
	FOR m : pc.getAllMessages(conj)
		sendMessage(m, pc.name, portClassName, conj)
	ENDFOR
		
	//------------------------------------------------------------------------------------------------------------
	// IF conjconjugated ENDIFreplicated port class
	//------------------------------------------------------------------------------------------------------------
	replPortClassName::replPortClassName(etRuntime::IEventReceiver& actor, etRuntime::IRTObject* parent, std::string name, int localId, std::vector<Address> addr, std::vector<Address> peerAddress) 
		: m_replication(addr.size()),
	  	  m_ports()
	{
		char numstr[10]; // enough to hold all numbers up to 32-bits
	
		m_ports = reinterpret_cast<portClassName*> (new char[sizeof(portClassName) * addr.size()]);
		for (int i = 0; i < m_replication; ++i) {
			snprintf(numstr, sizeof(numstr), "%d", i);
			//placement new to avoid copy construction, therefore no vector is used
			new  (&m_ports[i]) portClassName(actor, parent, name + numstr, localId, i, addr[i], peerAddress[i]);
		}
	};
	
		
	// outgoing messages
	FOR m : pc.getAllMessages(conj)
	messageSignatureDefinition(m, replPortClassName){
		for (int i=0; i<m_replication; ++i) {
			m_ports[i].messageCall(m);
		}
	}
	ENDFOR
	'''
	}
	
	def generateConstructorInitalizerList(PortClass pc, String index) { 
		var initializerList = new ArrayList<CharSequence>();
		initializerList.add('''PortBase(actor, parent, name, localId, index, addr, peerAddress)''')
		if (pc != null) {
			for (attrib: pc.attributes) {
				initializerList.add(attrib.attributeInitialization(false))
			}
		}
		return 
		'''
		  initializerList.join(',\n')
		'''
	}

	
	

	def private messageCall(Message m) {
	'''m.name(IF m.data!=null m.data.nameENDIF)'''
	}
	

	
	def private generateDebugHelpersImplementation(Root root, ProtocolClass pc){'''
		
		TODO: make this optional or different for smaller footprint
		/* message names as strings for debugging (generate MSC) */
		std::string pc.name::s_messageStrings[] 
				= {"MIN", 
				   FOR m : pc.getAllOutgoingMessages()
				   "m.name",
				   ENDFOR 
				   FOR m : pc.getAllIncomingMessages()
				   "m.name",
				   ENDFOR
				   "MAX"};
		
		std::string pc.name::getMessageString(int msg_id) {
			if ((MSG_MIN < msg_id ) && ( msg_id < MSG_MAX )) {
				return s_messageStrings[msg_id];
			} else {
				// id out of range
				return "Message ID out of range";
			}
		}
		
		'''
	}
	 

	def messageSignature(Message m) {
		'''IF m.privprivate:ELSEpublic:ENDIF void m.name(IF m.data!=nullm.data.refType.type.typeName m.data.nameENDIF)'''
	}

	def messageSignatureExplicit(Message m) {
		var dc = (m.data.refType.type as DataClass)
		'''public: void m.name(IF dc.base!=nulldc.base.typeName _super, ENDIFFOR a : dc.attributes SEPARATOR ", "a.type.type.typeNameIF a.size>1[]ENDIF a.nameENDFOR)'''
	}

	def messageSignatureDefinition(Message m, String classPrefix) {
		'''void classPrefix::m.name(IF m.data!=nullm.data.refType.type.typeName m.data.nameENDIF)'''
	}

	def messageSignatureExplicitDefinition(Message m, String classPrefix) {
		var dc = (m.data.refType.type as DataClass)
		'''void classPrefix::m.name(IF dc.base!=nulldc.base.typeName _super, ENDIFFOR a : dc.attributes SEPARATOR ", "a.type.type.typeNameIF a.size>1[]ENDIF a.nameENDFOR)'''
	}

//	def messageCall(Message m) {
//		'''m.name(IF m.data!=null m.data.nameENDIF)'''
//	}
	
	def sendMessageDeclaration(Message m, boolean conj) {
		'''
			messageSignature(m);
			IF m.data!=null && m.data.refType.type instanceof DataClass
				messageSignatureExplicit(m);
			ENDIF
		'''
	}
	
	def sendMessage(Message m, String portClassName, String classPrefix, boolean conj) {
		var dir = if (conj) "IN" else "OUT"
		var hdlr = m.getSendHandler(conj)
		'''
			messageSignatureDefinition(m, classPrefix) {
				IF hdlr!=null
					FOR command : hdlr.detailCode.lines	command
					ENDFOR
				ELSE
					DebuggingService::getInstance().addMessageAsyncOut(getAddress(), getPeerAddress(),
																	   portClassName::getMessageString(portClassName::dir_m.name));
					if (getPeerAddress().isValid()){
						IF m.data==nullgetPeerMsgReceiver()->receive(new Message(getPeerAddress(), portClassName::dir_m.name));
						ELSEgetPeerMsgReceiver()->receive(new Message(getPeerAddress(),portClassName::dir_m.name, 
						                                                reinterpret_cast<void*>(IF (!m.data.refType.ref && !(m.data.refType.type instanceof PrimitiveType))&ENDIFm.data.name),
						                                                sizeof(m.data.refType.type.typeName)));
						ENDIF
					}
				ENDIF
			}
			
			IF m.data!=null && m.data.refType.type instanceof DataClass
				messageSignatureExplicitDefinition(m, classPrefix) {
					m.name(m.data.refType.type.name(IF (m.data.refType.type as DataClass).base!=null_super, ENDIFFOR a : (m.data.refType.type as DataClass).attributes SEPARATOR ", "a.nameENDFOR));
				}
			ENDIF
		'''
		//TODO derived data classes are not yet handled correctly
	}
	
	override getMessageID(Message msg, InterfaceItem item) {
		if (item instanceof Port) {
			var p = item as Port;
			var direction = if (p.isConjugated())"OUT_" else "IN_"
			return enumInUse(p.getProtocol().getName(), direction+msg.getName())
		}
		else if (item instanceof SAP) {
			var sap = item as SAP;
			return enumInUse(sap.getProtocol().getName(), "OUT_"+msg.getName())
		}
		else if (item instanceof SPP) {
			var spp = item as SPP;
			return enumInUse(spp.getProtocol().getName(), "IN_"+msg.getName())
		}

		return "unknown interface item";
	}
	
	def String enumInUse(String namespace, String member) {
		return namespace+"::"+member
	}
}