/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.AspectExtensions;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.utils.TypeReferencesHelper;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmAnnotationValue;
import org.eclipse.xtext.common.types.JvmBooleanAnnotationValue;
import org.eclipse.xtext.common.types.JvmCustomAnnotationValue;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmEnumerationLiteral;
import org.eclipse.xtext.common.types.JvmEnumerationType;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmVisibility;
import org.eclipse.xtext.xbase.XBooleanLiteral;
import org.eclipse.xtext.xbase.XStringLiteral;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Infers the minimal Ecore file (an {@link EPackage}) corresponding to the
 * "modeling intention" of a K3 aspect. For example, from the following aspect:
 * 
 * <code>
 * \@Aspect(className = A)
 * class AspectA {
 *     public int foo
 *     def void bar() {}
 * }
 * </code>
 * 
 * it will infer a new {@link EPackage} containing an {@link EClass} {@code A}
 * with an {@link EAttribute} {@code foo} and an {@link EOperation} {@code foo}.
 */
@SuppressWarnings("all")
public class AspectToEcore {
  @Inject
  @Extension
  private AspectExtensions _aspectExtensions;
  
  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;
  
  @Inject
  @Extension
  private TypeReferencesHelper _typeReferencesHelper;
  
  @Inject
  private JvmTypeReferenceBuilder.Factory typeRefBuilderFactory;
  
  private final static String CONTAINMENT_ANNOTATION_FQN = "fr.inria.diverse.melange.annotation.Containment";
  
  private final static String UNIQUE_ANNOTATION_FQN = "fr.inria.diverse.melange.annotation.Unique";
  
  private final static String OPPOSITE_ANNOTATION_FQN = "fr.inria.diverse.melange.annotation.Opposite";
  
  private final static String STEP_ANNOTATION_FQN = "fr.inria.diverse.k3.al.annotationprocessor.Step";
  
  private final static List<String> K3_PREFIXES = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("_privk3", "super_"));
  
  public final static String PROP_NAME = "AspectProperties";
  
  /**
   * Analyzes the aspect {@code aspect}, woven on the {@link EClass}
   * {@code baseCls} contained in the {@link EPackage} {@code basePkg}, and
   * returns the corresponding {@link EPackage} describing its modeling intention.
   */
  public EPackage inferEcoreFragment(final JvmDeclaredType aspect, final EClass baseCls, final Set<EPackage> basePkgs) {
    final JvmTypeReferenceBuilder typeRefBuilder = this.typeRefBuilderFactory.create(aspect.eResource().getResourceSet());
    EPackage _xifexpression = null;
    if ((baseCls != null)) {
      _xifexpression = this.copyPackage(baseCls);
    } else {
      EPackage _createEPackage = EcoreFactory.eINSTANCE.createEPackage();
      final Procedure1<EPackage> _function = new Procedure1<EPackage>() {
        @Override
        public void apply(final EPackage it) {
          it.setName(IterableExtensions.<EPackage>head(basePkgs).getName());
          it.setNsPrefix(IterableExtensions.<EPackage>head(basePkgs).getNsPrefix());
          it.setNsURI(IterableExtensions.<EPackage>head(basePkgs).getNsURI());
        }
      };
      _xifexpression = ObjectExtensions.<EPackage>operator_doubleArrow(_createEPackage, _function);
    }
    final EPackage aspPkg = _xifexpression;
    final EPackage aspTopPkg = this._ecoreExtensions.getRootPackage(aspPkg);
    EClass _createEClass = EcoreFactory.eINSTANCE.createEClass();
    final Procedure1<EClass> _function_1 = new Procedure1<EClass>() {
      @Override
      public void apply(final EClass cls) {
        String _xifexpression = null;
        if ((baseCls != null)) {
          _xifexpression = baseCls.getName();
        } else {
          _xifexpression = aspect.getSimpleName();
        }
        cls.setName(_xifexpression);
        boolean _xifexpression_1 = false;
        if ((baseCls != null)) {
          _xifexpression_1 = baseCls.isAbstract();
        } else {
          _xifexpression_1 = aspect.isAbstract();
        }
        cls.setAbstract(_xifexpression_1);
        boolean _xifexpression_2 = false;
        if ((baseCls != null)) {
          _xifexpression_2 = baseCls.isInterface();
        } else {
          _xifexpression_2 = false;
        }
        cls.setInterface(_xifexpression_2);
        if ((baseCls == null)) {
          AspectToEcore.this._ecoreExtensions.addAspectAnnotation(cls);
          if (((aspect.getExtendedClass() != null) && (!Objects.equal(aspect.getExtendedClass().getSimpleName(), "Object")))) {
            EList<EClass> _eSuperTypes = cls.getESuperTypes();
            EClass _orCreateClass = AspectToEcore.this._ecoreExtensions.getOrCreateClass(aspTopPkg, 
              aspect.getExtendedClass().getQualifiedName());
            _eSuperTypes.add(_orCreateClass);
          }
        }
      }
    };
    final EClass aspCls = ObjectExtensions.<EClass>operator_doubleArrow(_createEClass, _function_1);
    EList<EClassifier> _eClassifiers = aspPkg.getEClassifiers();
    _eClassifiers.add(aspCls);
    final Function1<JvmField, Boolean> _function_2 = new Function1<JvmField, Boolean>() {
      @Override
      public Boolean apply(final JvmField it) {
        return Boolean.valueOf((Objects.equal(it.getVisibility(), JvmVisibility.PUBLIC) && (!it.isStatic())));
      }
    };
    final Consumer<JvmField> _function_3 = new Consumer<JvmField>() {
      @Override
      public void accept(final JvmField field) {
        final JvmTypeReference fieldType = field.getType();
        int _xifexpression = (int) 0;
        boolean _isList = AspectToEcore.this._typeReferencesHelper.isList(fieldType);
        if (_isList) {
          _xifexpression = (-1);
        } else {
          _xifexpression = 1;
        }
        final int upperB = _xifexpression;
        JvmType _xifexpression_1 = null;
        boolean _isList_1 = AspectToEcore.this._typeReferencesHelper.isList(fieldType);
        if (_isList_1) {
          _xifexpression_1 = AspectToEcore.this._typeReferencesHelper.getContainedElementsType(fieldType);
        } else {
          _xifexpression_1 = fieldType.getType();
        }
        final JvmType realType = _xifexpression_1;
        EClass _xifexpression_2 = null;
        String _qualifiedName = realType.getQualifiedName();
        String _uniqueId = AspectToEcore.this._ecoreExtensions.getUniqueId(aspCls);
        boolean _equals = Objects.equal(_qualifiedName, _uniqueId);
        if (_equals) {
          _xifexpression_2 = aspCls;
        } else {
          _xifexpression_2 = AspectToEcore.this._ecoreExtensions.findClass(basePkgs, realType.getQualifiedName());
        }
        final EClass find = _xifexpression_2;
        if ((find != null)) {
          EList<EStructuralFeature> _eStructuralFeatures = aspCls.getEStructuralFeatures();
          EReference _createEReference = EcoreFactory.eINSTANCE.createEReference();
          final Procedure1<EReference> _function = new Procedure1<EReference>() {
            @Override
            public void apply(final EReference it) {
              it.setName(field.getSimpleName());
              it.setEType(AspectToEcore.this._ecoreExtensions.getOrCreateClass(aspTopPkg, AspectToEcore.this.toQualifiedName(find)));
              it.setUpperBound(upperB);
              it.setContainment(AspectToEcore.this.isContainment(field));
              AspectToEcore.this._ecoreExtensions.addAspectAnnotation(it);
              it.setUnique(AspectToEcore.this.isUnique(field));
            }
          };
          EReference _doubleArrow = ObjectExtensions.<EReference>operator_doubleArrow(_createEReference, _function);
          _eStructuralFeatures.add(_doubleArrow);
        } else {
          EList<EStructuralFeature> _eStructuralFeatures_1 = aspCls.getEStructuralFeatures();
          EAttribute _createEAttribute = EcoreFactory.eINSTANCE.createEAttribute();
          final Procedure1<EAttribute> _function_1 = new Procedure1<EAttribute>() {
            @Override
            public void apply(final EAttribute it) {
              it.setName(field.getSimpleName());
              EClassifier _xifexpression = null;
              if ((realType instanceof JvmEnumerationType)) {
                final Function1<JvmEnumerationLiteral, String> _function = new Function1<JvmEnumerationLiteral, String>() {
                  @Override
                  public String apply(final JvmEnumerationLiteral it) {
                    return it.getSimpleName();
                  }
                };
                _xifexpression = AspectToEcore.this._ecoreExtensions.getOrCreateEnum(aspTopPkg, ((JvmEnumerationType)realType).getSimpleName(), 
                  ListExtensions.<JvmEnumerationLiteral, String>map(((JvmEnumerationType)realType).getLiterals(), _function));
              } else {
                _xifexpression = AspectToEcore.this._ecoreExtensions.getOrCreateDataType(aspTopPkg, realType.getSimpleName(), 
                  realType.getQualifiedName());
              }
              it.setEType(_xifexpression);
              it.setUpperBound(upperB);
              AspectToEcore.this._ecoreExtensions.addAspectAnnotation(it);
              it.setUnique(AspectToEcore.this.isUnique(field));
            }
          };
          EAttribute _doubleArrow_1 = ObjectExtensions.<EAttribute>operator_doubleArrow(_createEAttribute, _function_1);
          _eStructuralFeatures_1.add(_doubleArrow_1);
        }
      }
    };
    IterableExtensions.<JvmField>filter(aspect.getDeclaredFields(), _function_2).forEach(_function_3);
    final Function1<JvmOperation, Boolean> _function_4 = new Function1<JvmOperation, Boolean>() {
      @Override
      public Boolean apply(final JvmOperation it) {
        return Boolean.valueOf(((!AspectToEcore.this.isK3Specific(it)) && Objects.equal(it.getVisibility(), JvmVisibility.PUBLIC)));
      }
    };
    final Consumer<JvmOperation> _function_5 = new Consumer<JvmOperation>() {
      @Override
      public void accept(final JvmOperation op) {
        final String featureName = AspectToEcore.this.findFeatureNameFor(aspect, op, typeRefBuilder);
        if ((featureName == null)) {
          boolean _isEvent = AspectToEcore.this.isEvent(op);
          if (_isEvent) {
            final ArrayList<EClassifier> parameterClassifiers = new ArrayList<EClassifier>();
            final Consumer<JvmFormalParameter> _function = new Consumer<JvmFormalParameter>() {
              @Override
              public void accept(final JvmFormalParameter p) {
                final String parameterTypeName = p.getParameterType().getType().getSimpleName();
                final Function1<EPackage, Boolean> _function = new Function1<EPackage, Boolean>() {
                  @Override
                  public Boolean apply(final EPackage pkg) {
                    EClassifier _eClassifier = pkg.getEClassifier(parameterTypeName);
                    return Boolean.valueOf((_eClassifier != null));
                  }
                };
                EPackage _findFirst = IterableExtensions.<EPackage>findFirst(basePkgs, _function);
                EClassifier _eClassifier = null;
                if (_findFirst!=null) {
                  _eClassifier=_findFirst.getEClassifier(parameterTypeName);
                }
                final EClassifier parameterClassifier = _eClassifier;
                if ((parameterClassifier != null)) {
                  parameterClassifiers.add(parameterClassifier);
                }
              }
            };
            op.getParameters().forEach(_function);
            String _firstUpper = StringExtensions.toFirstUpper(baseCls.getName());
            String _firstUpper_1 = StringExtensions.toFirstUpper(op.getSimpleName());
            String _plus = (_firstUpper + _firstUpper_1);
            final String eventName = (_plus + "Event");
            EClass _createEClass = EcoreFactory.eINSTANCE.createEClass();
            final Procedure1<EClass> _function_1 = new Procedure1<EClass>() {
              @Override
              public void apply(final EClass cls) {
                cls.setName(eventName);
                cls.setAbstract(false);
                cls.setInterface(false);
                final Consumer<EClassifier> _function = new Consumer<EClassifier>() {
                  @Override
                  public void accept(final EClassifier p) {
                    EList<EStructuralFeature> _eStructuralFeatures = cls.getEStructuralFeatures();
                    EReference _createEReference = EcoreFactory.eINSTANCE.createEReference();
                    final Procedure1<EReference> _function = new Procedure1<EReference>() {
                      @Override
                      public void apply(final EReference ref) {
                        ref.setName(StringExtensions.toFirstLower(p.getName()));
                        ref.setLowerBound(0);
                        ref.setUpperBound(1);
                        ref.setEType(p);
                      }
                    };
                    EReference _doubleArrow = ObjectExtensions.<EReference>operator_doubleArrow(_createEReference, _function);
                    _eStructuralFeatures.add(_doubleArrow);
                  }
                };
                parameterClassifiers.forEach(_function);
              }
            };
            final EClass evtCls = ObjectExtensions.<EClass>operator_doubleArrow(_createEClass, _function_1);
            EList<EClassifier> _eClassifiers = aspPkg.getEClassifiers();
            _eClassifiers.add(evtCls);
          }
          int _xifexpression = (int) 0;
          boolean _isList = AspectToEcore.this._typeReferencesHelper.isList(op.getReturnType());
          if (_isList) {
            _xifexpression = (-1);
          } else {
            _xifexpression = 1;
          }
          final int upperB = _xifexpression;
          JvmType _xifexpression_1 = null;
          boolean _isList_1 = AspectToEcore.this._typeReferencesHelper.isList(op.getReturnType());
          if (_isList_1) {
            _xifexpression_1 = AspectToEcore.this._typeReferencesHelper.getContainedElementsType(op.getReturnType());
          } else {
            _xifexpression_1 = op.getReturnType().getType();
          }
          final JvmType realType = _xifexpression_1;
          EClass _xifexpression_2 = null;
          String _qualifiedName = realType.getQualifiedName();
          String _uniqueId = AspectToEcore.this._ecoreExtensions.getUniqueId(aspCls);
          boolean _equals = Objects.equal(_qualifiedName, _uniqueId);
          if (_equals) {
            _xifexpression_2 = aspCls;
          } else {
            _xifexpression_2 = AspectToEcore.this._ecoreExtensions.findClass(basePkgs, realType.getQualifiedName());
          }
          final EClass retCls = _xifexpression_2;
          final Function1<EOperation, Boolean> _function_2 = new Function1<EOperation, Boolean>() {
            @Override
            public Boolean apply(final EOperation it) {
              String _name = it.getName();
              String _simpleName = op.getSimpleName();
              return Boolean.valueOf(Objects.equal(_name, _simpleName));
            }
          };
          boolean _exists = IterableExtensions.<EOperation>exists(aspCls.getEOperations(), _function_2);
          boolean _not = (!_exists);
          if (_not) {
            EList<EOperation> _eOperations = aspCls.getEOperations();
            EOperation _createEOperation = EcoreFactory.eINSTANCE.createEOperation();
            final Procedure1<EOperation> _function_3 = new Procedure1<EOperation>() {
              @Override
              public void apply(final EOperation it) {
                it.setName(op.getSimpleName());
                final Procedure2<JvmFormalParameter, Integer> _function = new Procedure2<JvmFormalParameter, Integer>() {
                  @Override
                  public void apply(final JvmFormalParameter p, final Integer i) {
                    if (((!AspectToEcore.this._aspectExtensions.hasAspectAnnotation(aspect)) || ((i).intValue() > 0))) {
                      final JvmType pType = p.getParameterType().getType();
                      int _xifexpression = (int) 0;
                      boolean _isList = AspectToEcore.this._typeReferencesHelper.isList(p.getParameterType());
                      if (_isList) {
                        _xifexpression = (-1);
                      } else {
                        _xifexpression = 1;
                      }
                      final int upperBP = _xifexpression;
                      JvmType _xifexpression_1 = null;
                      boolean _isList_1 = AspectToEcore.this._typeReferencesHelper.isList(p.getParameterType());
                      if (_isList_1) {
                        _xifexpression_1 = AspectToEcore.this._typeReferencesHelper.getContainedElementsType(p.getParameterType());
                      } else {
                        _xifexpression_1 = pType;
                      }
                      final JvmType realTypeP = _xifexpression_1;
                      EClass _xifexpression_2 = null;
                      String _qualifiedName = realTypeP.getQualifiedName();
                      String _uniqueId = AspectToEcore.this._ecoreExtensions.getUniqueId(aspCls);
                      boolean _equals = Objects.equal(_qualifiedName, _uniqueId);
                      if (_equals) {
                        _xifexpression_2 = aspCls;
                      } else {
                        _xifexpression_2 = AspectToEcore.this._ecoreExtensions.findClass(basePkgs, realTypeP.getQualifiedName());
                      }
                      final EClass attrCls = _xifexpression_2;
                      EList<EParameter> _eParameters = it.getEParameters();
                      EParameter _createEParameter = EcoreFactory.eINSTANCE.createEParameter();
                      final Procedure1<EParameter> _function = new Procedure1<EParameter>() {
                        @Override
                        public void apply(final EParameter pp) {
                          pp.setName(p.getSimpleName());
                          pp.setUpperBound(upperBP);
                          EClassifier _xifexpression = null;
                          if ((attrCls != null)) {
                            _xifexpression = AspectToEcore.this._ecoreExtensions.getOrCreateClass(aspTopPkg, AspectToEcore.this._ecoreExtensions.getUniqueId(attrCls));
                          } else {
                            EClassifier _xifexpression_1 = null;
                            if ((realTypeP instanceof JvmEnumerationType)) {
                              final Function1<JvmEnumerationLiteral, String> _function = new Function1<JvmEnumerationLiteral, String>() {
                                @Override
                                public String apply(final JvmEnumerationLiteral it) {
                                  return it.getSimpleName();
                                }
                              };
                              _xifexpression_1 = AspectToEcore.this._ecoreExtensions.getOrCreateEnum(aspTopPkg, ((JvmEnumerationType)realTypeP).getSimpleName(), 
                                ListExtensions.<JvmEnumerationLiteral, String>map(((JvmEnumerationType)realTypeP).getLiterals(), _function));
                            } else {
                              _xifexpression_1 = AspectToEcore.this._ecoreExtensions.getOrCreateDataType(aspTopPkg, realTypeP.getSimpleName(), 
                                realTypeP.getQualifiedName());
                            }
                            _xifexpression = _xifexpression_1;
                          }
                          pp.setEType(_xifexpression);
                        }
                      };
                      EParameter _doubleArrow = ObjectExtensions.<EParameter>operator_doubleArrow(_createEParameter, _function);
                      _eParameters.add(_doubleArrow);
                    }
                  }
                };
                IterableExtensions.<JvmFormalParameter>forEach(op.getParameters(), _function);
                if (((!Objects.equal(op.getReturnType().getSimpleName(), "void")) && (op.getReturnType().getSimpleName() != "null"))) {
                  it.setUpperBound(upperB);
                  EClassifier _xifexpression = null;
                  if ((retCls != null)) {
                    _xifexpression = AspectToEcore.this._ecoreExtensions.getOrCreateClass(aspTopPkg, AspectToEcore.this._ecoreExtensions.getUniqueId(retCls));
                  } else {
                    EClassifier _xifexpression_1 = null;
                    if ((realType instanceof JvmEnumerationType)) {
                      final Function1<JvmEnumerationLiteral, String> _function_1 = new Function1<JvmEnumerationLiteral, String>() {
                        @Override
                        public String apply(final JvmEnumerationLiteral it) {
                          return it.getSimpleName();
                        }
                      };
                      _xifexpression_1 = AspectToEcore.this._ecoreExtensions.getOrCreateEnum(aspTopPkg, ((JvmEnumerationType)realType).getSimpleName(), 
                        ListExtensions.<JvmEnumerationLiteral, String>map(((JvmEnumerationType)realType).getLiterals(), _function_1));
                    } else {
                      _xifexpression_1 = AspectToEcore.this._ecoreExtensions.getOrCreateDataType(aspTopPkg, realType.getSimpleName(), 
                        realType.getQualifiedName());
                    }
                    _xifexpression = _xifexpression_1;
                  }
                  it.setEType(_xifexpression);
                }
                AspectToEcore.this._ecoreExtensions.addAspectAnnotation(it);
              }
            };
            EOperation _doubleArrow = ObjectExtensions.<EOperation>operator_doubleArrow(_createEOperation, _function_3);
            _eOperations.add(_doubleArrow);
          }
        } else {
          final Function1<EStructuralFeature, Boolean> _function_4 = new Function1<EStructuralFeature, Boolean>() {
            @Override
            public Boolean apply(final EStructuralFeature it) {
              String _name = it.getName();
              return Boolean.valueOf(Objects.equal(_name, featureName));
            }
          };
          boolean _exists_1 = IterableExtensions.<EStructuralFeature>exists(aspCls.getEStructuralFeatures(), _function_4);
          boolean _not_1 = (!_exists_1);
          if (_not_1) {
            JvmTypeReference _xifexpression_3 = null;
            if ((op.getSimpleName().startsWith("get") || (op.getParameters().size() == 1))) {
              _xifexpression_3 = op.getReturnType();
            } else {
              _xifexpression_3 = op.getParameters().get(1).getParameterType();
            }
            final JvmTypeReference retType = _xifexpression_3;
            int _xifexpression_4 = (int) 0;
            boolean _isList_2 = AspectToEcore.this._typeReferencesHelper.isList(op.getReturnType());
            if (_isList_2) {
              _xifexpression_4 = (-1);
            } else {
              _xifexpression_4 = 1;
            }
            final int upperB_1 = _xifexpression_4;
            JvmType _xifexpression_5 = null;
            boolean _isList_3 = AspectToEcore.this._typeReferencesHelper.isList(op.getReturnType());
            if (_isList_3) {
              _xifexpression_5 = AspectToEcore.this._typeReferencesHelper.getContainedElementsType(retType);
            } else {
              _xifexpression_5 = retType.getType();
            }
            final JvmType realType_1 = _xifexpression_5;
            EClass _xifexpression_6 = null;
            String _qualifiedName_1 = realType_1.getQualifiedName();
            String _uniqueId_1 = AspectToEcore.this._ecoreExtensions.getUniqueId(aspCls);
            boolean _equals_1 = Objects.equal(_qualifiedName_1, _uniqueId_1);
            if (_equals_1) {
              _xifexpression_6 = aspCls;
            } else {
              _xifexpression_6 = AspectToEcore.this._ecoreExtensions.findClass(basePkgs, realType_1.getQualifiedName());
            }
            final EClass find = _xifexpression_6;
            if ((find != null)) {
              EList<EStructuralFeature> _eStructuralFeatures = aspCls.getEStructuralFeatures();
              EReference _createEReference = EcoreFactory.eINSTANCE.createEReference();
              final Procedure1<EReference> _function_5 = new Procedure1<EReference>() {
                @Override
                public void apply(final EReference it) {
                  it.setName(featureName);
                  it.setEType(AspectToEcore.this._ecoreExtensions.getOrCreateClass(aspTopPkg, AspectToEcore.this.toQualifiedName(find)));
                  it.setUpperBound(upperB_1);
                  it.setContainment(AspectToEcore.this.isContainment(op));
                  AspectToEcore.this._ecoreExtensions.addAspectAnnotation(it);
                  it.setUnique(AspectToEcore.this.isUnique(op));
                  boolean _isOpposite = AspectToEcore.this.isOpposite(op);
                  if (_isOpposite) {
                    AspectToEcore.this._ecoreExtensions.addOppositeAnnotation(it, AspectToEcore.this.getOppositeValue(op));
                  }
                }
              };
              EReference _doubleArrow_1 = ObjectExtensions.<EReference>operator_doubleArrow(_createEReference, _function_5);
              _eStructuralFeatures.add(_doubleArrow_1);
            } else {
              EList<EStructuralFeature> _eStructuralFeatures_1 = aspCls.getEStructuralFeatures();
              EAttribute _createEAttribute = EcoreFactory.eINSTANCE.createEAttribute();
              final Procedure1<EAttribute> _function_6 = new Procedure1<EAttribute>() {
                @Override
                public void apply(final EAttribute it) {
                  it.setName(featureName);
                  EClassifier _xifexpression = null;
                  if ((realType_1 instanceof JvmEnumerationType)) {
                    final Function1<JvmEnumerationLiteral, String> _function = new Function1<JvmEnumerationLiteral, String>() {
                      @Override
                      public String apply(final JvmEnumerationLiteral it) {
                        return it.getSimpleName();
                      }
                    };
                    _xifexpression = AspectToEcore.this._ecoreExtensions.getOrCreateEnum(aspTopPkg, ((JvmEnumerationType)realType_1).getSimpleName(), 
                      ListExtensions.<JvmEnumerationLiteral, String>map(((JvmEnumerationType)realType_1).getLiterals(), _function));
                  } else {
                    _xifexpression = AspectToEcore.this._ecoreExtensions.getOrCreateDataType(aspTopPkg, realType_1.getSimpleName(), 
                      realType_1.getQualifiedName());
                  }
                  it.setEType(_xifexpression);
                  it.setUpperBound(upperB_1);
                  AspectToEcore.this._ecoreExtensions.addAspectAnnotation(it);
                  boolean _isContainment = AspectToEcore.this.isContainment(op);
                  if (_isContainment) {
                    AspectToEcore.this._ecoreExtensions.addContainmentAnnotation(it);
                  }
                  it.setUnique(AspectToEcore.this.isUnique(op));
                }
              };
              EAttribute _doubleArrow_2 = ObjectExtensions.<EAttribute>operator_doubleArrow(_createEAttribute, _function_6);
              _eStructuralFeatures_1.add(_doubleArrow_2);
            }
          }
        }
      }
    };
    IterableExtensions.<JvmOperation>filter(aspect.getDeclaredOperations(), _function_4).forEach(_function_5);
    return aspTopPkg;
  }
  
  /**
   * For the getter or setter {@code op} in {@code type}, infers the
   * corresponding feature name (eg. getXyz()/setXyz() associated to the
   * "xyz" feature).
   * 
   * @param type The {@link JvmDeclaredType} of an aspect.
   * @param op   A {@link JvmOperation} of {@code type}.
   * @return the corresponding feature name, or null if it cannot be determined.
   */
  public String findFeatureNameFor(final JvmDeclaredType type, final JvmOperation op, final JvmTypeReferenceBuilder typeRefBuilder) {
    if ((((((op.getSimpleName().startsWith("get") && Character.isUpperCase(op.getSimpleName().charAt(3))) && (op.getParameters().size() == 1)) && (!Objects.equal(op.getReturnType().getSimpleName(), "void"))) && IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), new Function1<JvmOperation, Boolean>() {
      @Override
      public Boolean apply(final JvmOperation opp) {
        return Boolean.valueOf(((Objects.equal(opp.getSimpleName(), op.getSimpleName().replaceFirst("get", "set")) && Objects.equal(opp.getParameters().get(1).getParameterType().getQualifiedName(), op.getReturnType().getQualifiedName())) && Objects.equal(opp.getReturnType().getSimpleName(), "void")));
      }
    })) || ((((op.getSimpleName().startsWith("set") && Character.isUpperCase(op.getSimpleName().charAt(3))) && (op.getParameters().size() == 2)) && Objects.equal(op.getReturnType().getSimpleName(), "void")) && IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), new Function1<JvmOperation, Boolean>() {
      @Override
      public Boolean apply(final JvmOperation opp) {
        return Boolean.valueOf((Objects.equal(opp.getSimpleName(), op.getSimpleName().replaceFirst("set", "get")) && Objects.equal(opp.getReturnType().getQualifiedName(), op.getParameters().get(1).getParameterType().getQualifiedName())));
      }
    })))) {
      return StringExtensions.toFirstLower(op.getSimpleName().substring(3, op.getSimpleName().length()));
    } else {
      final Function1<JvmOperation, Boolean> _function = new Function1<JvmOperation, Boolean>() {
        @Override
        public Boolean apply(final JvmOperation opp) {
          return Boolean.valueOf((((opp != op) && Objects.equal(opp.getSimpleName(), op.getSimpleName())) && 
            ((((((op.getParameters().size() == 1) && (!Objects.equal(op.getReturnType().getSimpleName(), "void"))) && (opp.getParameters().size() == 2)) && Objects.equal(opp.getReturnType().getSimpleName(), "void")) && Objects.equal(op.getReturnType().getQualifiedName(), opp.getParameters().get(1).getParameterType().getQualifiedName())) || 
              (((((opp.getParameters().size() == 1) && (!Objects.equal(opp.getReturnType().getSimpleName(), "void"))) && (op.getParameters().size() == 2)) && Objects.equal(op.getReturnType().getSimpleName(), "void")) && Objects.equal(op.getParameters().get(1).getParameterType().getQualifiedName(), opp.getReturnType().getQualifiedName())))));
        }
      };
      boolean _exists = IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), _function);
      if (_exists) {
        return op.getSimpleName();
      } else {
        boolean _isGetter = this.isGetter(op, typeRefBuilder);
        if (_isGetter) {
          return op.getSimpleName();
        } else {
          if ((((((op.getSimpleName().startsWith("get") && Character.isUpperCase(op.getSimpleName().charAt(3))) && (op.getParameters().size() == 0)) && (!Objects.equal(op.getReturnType().getSimpleName(), "void"))) && IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), new Function1<JvmOperation, Boolean>() {
            @Override
            public Boolean apply(final JvmOperation opp) {
              return Boolean.valueOf((((Objects.equal(opp.getSimpleName(), op.getSimpleName().replaceFirst("get", "set")) && (opp.getParameters().size() == 1)) && Objects.equal(opp.getParameters().get(0).getParameterType().getQualifiedName(), op.getReturnType().getQualifiedName())) && Objects.equal(opp.getReturnType().getSimpleName(), "void")));
            }
          })) || ((((op.getSimpleName().startsWith("set") && Character.isUpperCase(op.getSimpleName().charAt(3))) && (op.getParameters().size() == 1)) && Objects.equal(op.getReturnType().getSimpleName(), "void")) && IterableExtensions.<JvmOperation>exists(type.getDeclaredOperations(), new Function1<JvmOperation, Boolean>() {
            @Override
            public Boolean apply(final JvmOperation opp) {
              return Boolean.valueOf(((Objects.equal(opp.getSimpleName(), op.getSimpleName().replaceFirst("set", "get")) && (opp.getParameters().size() == 0)) && Objects.equal(opp.getReturnType().getQualifiedName(), op.getParameters().get(0).getParameterType().getQualifiedName())));
            }
          })))) {
            return StringExtensions.toFirstLower(op.getSimpleName().substring(3, op.getSimpleName().length()));
          } else {
            return null;
          }
        }
      }
    }
  }
  
  /**
   * Checks whether the given field is annotated with @Containment
   */
  private boolean isContainment(final JvmMember field) {
    final Function1<JvmAnnotationReference, Boolean> _function = new Function1<JvmAnnotationReference, Boolean>() {
      @Override
      public Boolean apply(final JvmAnnotationReference it) {
        String _qualifiedName = it.getAnnotation().getQualifiedName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.CONTAINMENT_ANNOTATION_FQN));
      }
    };
    return IterableExtensions.<JvmAnnotationReference>exists(field.getAnnotations(), _function);
  }
  
  /**
   * Checks whether the given field is annotated with @Unique or @Containment or @Opposite
   */
  private boolean isUnique(final JvmMember field) {
    return ((this.isContainment(field) || this.isOpposite(field)) || IterableExtensions.<JvmAnnotationReference>exists(field.getAnnotations(), 
      new Function1<JvmAnnotationReference, Boolean>() {
        @Override
        public Boolean apply(final JvmAnnotationReference it) {
          String _qualifiedName = it.getAnnotation().getQualifiedName();
          return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.UNIQUE_ANNOTATION_FQN));
        }
      }));
  }
  
  /**
   * Checks whether the given field is annotated with @Opposite
   */
  private boolean isOpposite(final JvmMember field) {
    return (this.isContainment(field) || IterableExtensions.<JvmAnnotationReference>exists(field.getAnnotations(), 
      new Function1<JvmAnnotationReference, Boolean>() {
        @Override
        public Boolean apply(final JvmAnnotationReference it) {
          String _qualifiedName = it.getAnnotation().getQualifiedName();
          return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.OPPOSITE_ANNOTATION_FQN));
        }
      }));
  }
  
  /**
   * Return the 'value' parameter of the annotation @Opposite
   * or null if none
   */
  private String getOppositeValue(final JvmMember field) {
    Object _xblockexpression = null;
    {
      final Function1<JvmAnnotationReference, Boolean> _function = new Function1<JvmAnnotationReference, Boolean>() {
        @Override
        public Boolean apply(final JvmAnnotationReference it) {
          String _qualifiedName = it.getAnnotation().getQualifiedName();
          return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.OPPOSITE_ANNOTATION_FQN));
        }
      };
      final JvmAnnotationReference annot = IterableExtensions.<JvmAnnotationReference>findFirst(field.getAnnotations(), _function);
      EList<JvmAnnotationValue> _values = null;
      if (annot!=null) {
        _values=annot.getValues();
      }
      JvmAnnotationValue _findFirst = null;
      if (_values!=null) {
        final Function1<JvmAnnotationValue, Boolean> _function_1 = new Function1<JvmAnnotationValue, Boolean>() {
          @Override
          public Boolean apply(final JvmAnnotationValue it) {
            String _valueName = it.getValueName();
            return Boolean.valueOf(Objects.equal(_valueName, "value"));
          }
        };
        _findFirst=IterableExtensions.<JvmAnnotationValue>findFirst(_values, _function_1);
      }
      final JvmAnnotationValue annotVal = _findFirst;
      if ((annotVal instanceof JvmCustomAnnotationValue)) {
        EObject _head = IterableExtensions.<EObject>head(((JvmCustomAnnotationValue) annotVal).getValues());
        final XStringLiteral opRef = ((XStringLiteral) _head);
        String _value = null;
        if (opRef!=null) {
          _value=opRef.getValue();
        }
        return _value;
      }
      _xblockexpression = null;
    }
    return ((String)_xblockexpression);
  }
  
  /**
   * Checks whether the given operation is an event or not
   */
  private boolean isEvent(final JvmOperation operation) {
    boolean result = false;
    final Function1<JvmAnnotationReference, Boolean> _function = new Function1<JvmAnnotationReference, Boolean>() {
      @Override
      public Boolean apply(final JvmAnnotationReference a) {
        String _qualifiedName = a.getAnnotation().getQualifiedName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, AspectToEcore.STEP_ANNOTATION_FQN));
      }
    };
    final JvmAnnotationReference stepAnnotation = IterableExtensions.<JvmAnnotationReference>findFirst(operation.getAnnotations(), _function);
    if ((stepAnnotation != null)) {
      final Function1<JvmAnnotationValue, Boolean> _function_1 = new Function1<JvmAnnotationValue, Boolean>() {
        @Override
        public Boolean apply(final JvmAnnotationValue v) {
          String _valueName = v.getValueName();
          return Boolean.valueOf(Objects.equal(_valueName, "eventTriggerable"));
        }
      };
      final JvmAnnotationValue triggerableValue = IterableExtensions.<JvmAnnotationValue>findFirst(stepAnnotation.getValues(), _function_1);
      if ((triggerableValue != null)) {
        Boolean _switchResult = null;
        boolean _matched = false;
        if (triggerableValue instanceof JvmBooleanAnnotationValue) {
          _matched=true;
          EList<Boolean> _values = ((JvmBooleanAnnotationValue)triggerableValue).getValues();
          Boolean _head = null;
          if (_values!=null) {
            _head=IterableExtensions.<Boolean>head(_values);
          }
          _switchResult = _head;
        }
        if (!_matched) {
          if (triggerableValue instanceof JvmCustomAnnotationValue) {
            _matched=true;
            EObject _head = IterableExtensions.<EObject>head(((JvmCustomAnnotationValue)triggerableValue).getValues());
            _switchResult = Boolean.valueOf(((XBooleanLiteral) _head).isIsTrue());
          }
        }
        result = (_switchResult).booleanValue();
      }
    }
    return result;
  }
  
  /**
   * Checks whether the given operation is some obscure K3 code or not
   */
  private boolean isK3Specific(final JvmOperation op) {
    final Function1<String, Boolean> _function = new Function1<String, Boolean>() {
      @Override
      public Boolean apply(final String p) {
        return Boolean.valueOf(op.getSimpleName().startsWith(p));
      }
    };
    return IterableExtensions.<String>exists(AspectToEcore.K3_PREFIXES, _function);
  }
  
  /**
   * Create a copy of the hierachy of sub-packages containing {@link baseCls}
   * Return the deepest package
   */
  private EPackage copyPackage(final EClass baseCls) {
    EPackage res = null;
    EPackage currentPkg = baseCls.getEPackage();
    EPackage last = null;
    while ((currentPkg != null)) {
      {
        final EPackage pkgCopy = EcoreFactory.eINSTANCE.createEPackage();
        pkgCopy.setName(currentPkg.getName());
        pkgCopy.setNsPrefix(currentPkg.getNsPrefix());
        pkgCopy.setNsURI(currentPkg.getNsURI());
        if ((last != null)) {
          EList<EPackage> _eSubpackages = pkgCopy.getESubpackages();
          _eSubpackages.add(last);
        } else {
          res = pkgCopy;
        }
        last = pkgCopy;
        currentPkg = currentPkg.getESuperPackage();
      }
    }
    return res;
  }
  
  private String toQualifiedName(final EClass clazz) {
    final List<String> res = CollectionLiterals.<String>newArrayList();
    res.add(clazz.getName());
    EPackage pack = clazz.getEPackage();
    while ((pack != null)) {
      {
        res.add(pack.getName());
        pack = pack.getESuperPackage();
      }
    }
    return IterableExtensions.join(ListExtensions.<String>reverse(res), ".");
  }
  
  /**
   * Return true if {@link op} is an Aspect generated getter for final field
   */
  private boolean isGetter(final JvmOperation op, final JvmTypeReferenceBuilder typeRefBuilder) {
    boolean _xblockexpression = false;
    {
      try {
        int _size = op.getParameters().size();
        boolean _equals = (_size == 1);
        if (_equals) {
          final String eclass = op.getParameters().get(0).getParameterType().getSimpleName();
          final String className = op.getDeclaringType().getQualifiedName();
          final JvmTypeReference aspectProperties = typeRefBuilder.typeRef(((className + eclass) + AspectToEcore.PROP_NAME));
          JvmType _type = aspectProperties.getType();
          final JvmGenericType type = ((JvmGenericType) _type);
          final Function1<JvmField, Boolean> _function = new Function1<JvmField, Boolean>() {
            @Override
            public Boolean apply(final JvmField it) {
              return Boolean.valueOf((Objects.equal(it.getSimpleName(), op.getSimpleName()) && it.isFinal()));
            }
          };
          return IterableExtensions.<JvmField>exists(Iterables.<JvmField>filter(type.getMembers(), JvmField.class), _function);
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
}
