/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.processors;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.metamodel.melange.MelangeFactory;
import fr.inria.diverse.melange.metamodel.melange.Operator;
import fr.inria.diverse.melange.metamodel.melange.Weave;
import fr.inria.diverse.melange.processors.DispatchMelangeProcessor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jdt.core.search.SearchMatch;
import org.eclipse.jdt.core.search.SearchParticipant;
import org.eclipse.jdt.core.search.SearchPattern;
import org.eclipse.jdt.core.search.SearchRequestor;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Parses the wildcard imports of aspects ({@code with x.y.*}), retrieves
 * the actual aspects hidden behind the wildcard and adds them to the appropriate
 * {@link Language}.
 */
@SuppressWarnings("all")
public class WildcardAspectResolver extends DispatchMelangeProcessor {
  @Inject
  private JvmTypeReferenceBuilder.Factory builderFactory;
  
  private final static Logger log = Logger.getLogger(WildcardAspectResolver.class);
  
  protected void _preProcess(final Weave w, final boolean preLinkingPhase) {
    final JvmTypeReferenceBuilder typeRefBuilder = this.builderFactory.create(w.eResource().getResourceSet());
    String _aspectWildcardImport = w.getAspectWildcardImport();
    boolean _tripleNotEquals = (_aspectWildcardImport != null);
    if (_tripleNotEquals) {
      final Consumer<String> _function = new Consumer<String>() {
        @Override
        public void accept(final String typeRef) {
          EList<Operator> _operators = w.getOwningLanguage().getOperators();
          Weave _createWeave = MelangeFactory.eINSTANCE.createWeave();
          final Procedure1<Weave> _function = new Procedure1<Weave>() {
            @Override
            public void apply(final Weave it) {
              it.setAspectTypeRef(typeRefBuilder.typeRef(typeRef));
            }
          };
          Weave _doubleArrow = ObjectExtensions.<Weave>operator_doubleArrow(_createWeave, _function);
          _operators.add(_doubleArrow);
        }
      };
      this.resolveWildcardImport(w.getAspectWildcardImport()).forEach(_function);
    }
  }
  
  private List<String> resolveWildcardImport(final String wildcardImport) {
    final ArrayList<String> matches = CollectionLiterals.<String>newArrayList();
    final SearchEngine engine = new SearchEngine();
    final SearchRequestor requestor = new SearchRequestor() {
      @Override
      public void acceptSearchMatch(final SearchMatch match) {
        Object _element = match.getElement();
        if ((_element instanceof IPackageFragment)) {
          Object _element_1 = match.getElement();
          final IPackageFragment pkg = ((IPackageFragment) _element_1);
          try {
            final Function1<IJavaElement, Boolean> _function = new Function1<IJavaElement, Boolean>() {
              @Override
              public Boolean apply(final IJavaElement it) {
                return Boolean.valueOf(((!it.getElementName().endsWith("AspectContext.java")) && (!it.getElementName().endsWith("AspectProperties.java"))));
              }
            };
            final Function1<IJavaElement, String> _function_1 = new Function1<IJavaElement, String>() {
              @Override
              public String apply(final IJavaElement it) {
                String _xblockexpression = null;
                {
                  String _elementName = it.getElementName();
                  int _length = it.getElementName().length();
                  int _minus = (_length - 5);
                  final String trimmedAspectName = _elementName.substring(
                    0, _minus);
                  StringConcatenation _builder = new StringConcatenation();
                  String _elementName_1 = pkg.getElementName();
                  _builder.append(_elementName_1);
                  _builder.append(".");
                  _builder.append(trimmedAspectName);
                  _xblockexpression = _builder.toString();
                }
                return _xblockexpression;
              }
            };
            Iterable<String> _map = IterableExtensions.<IJavaElement, String>map(IterableExtensions.<IJavaElement>filter(((Iterable<IJavaElement>)Conversions.doWrapArray(pkg.getChildren())), _function), _function_1);
            Iterables.<String>addAll(matches, _map);
          } catch (final Throwable _t) {
            if (_t instanceof JavaModelException) {
              final JavaModelException e = (JavaModelException)_t;
              WildcardAspectResolver.log.error(e);
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        }
      }
    };
    int _length = wildcardImport.length();
    int _minus = (_length - 2);
    final SearchPattern pattern = SearchPattern.createPattern(
      wildcardImport.substring(0, _minus), 
      IJavaSearchConstants.PACKAGE, 
      IJavaSearchConstants.DECLARATIONS, 
      SearchPattern.R_EXACT_MATCH);
    try {
      SearchParticipant _defaultSearchParticipant = SearchEngine.getDefaultSearchParticipant();
      engine.search(pattern, 
        new SearchParticipant[] { _defaultSearchParticipant }, 
        SearchEngine.createWorkspaceScope(), requestor, 
        null);
    } catch (final Throwable _t) {
      if (_t instanceof CoreException) {
        final CoreException e = (CoreException)_t;
        WildcardAspectResolver.log.error(e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return matches;
  }
  
  public void preProcess(final EObject w, final boolean preLinkingPhase) {
    if (w instanceof Weave) {
      _preProcess((Weave)w, preLinkingPhase);
      return;
    } else if (w != null) {
      _preProcess(w, preLinkingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(w, preLinkingPhase).toString());
    }
  }
}
