/*******************************************************************************
 * Copyright (c) 2010 Oracle.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Apache License v2.0 which accompanies this distribution.
 * The Eclipse Public License is available at
 *     http://www.eclipse.org/legal/epl-v10.html
 * and the Apache License v2.0 is available at
 *     http://www.opensource.org/licenses/apache2.0.php.
 * You may elect to redistribute this code under either of these licenses.
 *
 * Contributors:
 *     mkeith - Gemini JPA work
 ******************************************************************************/
package org.eclipse.gemini.jpa.proxy;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import javax.persistence.EntityManagerFactory;


import org.eclipse.gemini.jpa.PUnitInfo;
import org.eclipse.gemini.jpa.proxy.EMFServiceProxyHandler;
import org.eclipse.persistence.config.PersistenceUnitProperties;

import static org.eclipse.gemini.jpa.GeminiUtil.*;

/**
 * Dynamic proxy class to proxy the EMFBuilder service
 */
public class EMFBuilderServiceProxyHandler extends EMFServiceProxyHandler
                                           implements InvocationHandler {

    // Keep around a copy of the props used to create an EMF through the EMF builder
    Map<String,Object> emfProps = new HashMap<String,Object>();

    public EMFBuilderServiceProxyHandler(PUnitInfo pUnitInfo,
                                         EMFServiceProxyHandler emfService) {
        super(pUnitInfo);
    }

    /*=========================*/
    /* InvocationProxy methods */
    /*=========================*/

    // Will only get calls for the method on the EntityManagerFactoryBuilder interface
    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {

        debug("EMFBuilderProxy invocation on method ", method.getName());

        if (method.getName().equals("hashCode"))
            return this.hashCode();
        if (method.getName().equals("toString"))
            return this.toString();
        if (method.getName().equals("equals"))
            return this.equals(args[0]);
        
        // Must be a createEntityManagerFactory(String, Map) call

        // The first arg must be the properties Map
        Map<String,Object> props = (Map<String,Object>)args[0];

        if ( props != null && (props.get(GEMINI_JPA_CREATE_SEPARATE_EMF) != null || props.get(PersistenceUnitProperties.SESSION_NAME) != null) ){
        	if ( pUnitInfo.getDriverClassName() == null && pUnitInfo.getDriverUrl() == null ) {
        		// create EMF for an incomplete persistence unit
        		return super.createEMF(props);
        	}else{
        		// Not allowed for a complete persistence unit
                throw new IllegalArgumentException();
        	}
        }

        // If we have an EMF and it has already been closed, discard it
        EntityManagerFactory emf;
        synchronized (pUnitInfo) {
            emf = pUnitInfo.getEmf();
            if ((emf != null) && (!emf.isOpen())) {
                syncUnsetEMF();
                emfProps.clear();
                emf = null;
            }
        }
        // Check if an EMF already exists
        if (emf != null) {

            // Verify the JDBC properties match the ones that may have been previously passed in
            verifyJDBCProperties((String) emfProps.get(JPA_JDBC_DRIVER_PROPERTY),
                                 (String) emfProps.get(JPA_JDBC_URL_PROPERTY),
                                 props);

            // If it was created using the builder service then just return it
            if (pUnitInfo.isEmfSetByBuilderService()) {
                return emf;
            }

            // It was created using the EMF service.
            // Verify the JDBC properties match the ones in the descriptor.
            verifyJDBCProperties(pUnitInfo.getDriverClassName(),
                                 pUnitInfo.getDriverUrl(),
                                 props);
            return emf;
        } else {
            // No EMF service existed; create another
            return syncGetEMFAndSetIfAbsent(true, props);
        }
    }

    /*================*/
    /* Helper methods */
    /*================*/

    public EntityManagerFactory createEMF(Map<String,Object> props) {

        emfProps = props;
        return super.createEMF(props);
    }

    // Local method to compare properties passed in Map to ones in persistence descriptor or in previously set props
    protected void verifyJDBCProperties(String driver, String driverUrl, Map<String,Object> props) {

        if (driver != null) {
            String propDriver = (String) props.get(JPA_JDBC_DRIVER_PROPERTY);
            if ((propDriver != null) && !driver.equals(propDriver)) {
                throw new IllegalArgumentException();
            }
        }
        if (driverUrl != null) {
            String propUrl = (String) props.get(JPA_JDBC_URL_PROPERTY);
            if ((propUrl != null) && !driverUrl.equals(propUrl)) {
                throw new IllegalArgumentException();
            }
        }
    }
}
