/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client;

import java.io.Closeable;
import java.io.IOException;
import java.net.URI;
import java.nio.channels.AsynchronousCloseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.HttpConnection;
import org.eclipse.jetty.client.HttpConversation;
import org.eclipse.jetty.client.HttpExchange;
import org.eclipse.jetty.client.HttpResponse;
import org.eclipse.jetty.client.HttpResponseException;
import org.eclipse.jetty.client.RequestNotifier;
import org.eclipse.jetty.client.ResponseNotifier;
import org.eclipse.jetty.client.api.Connection;
import org.eclipse.jetty.client.api.Destination;
import org.eclipse.jetty.client.api.ProxyConfiguration;
import org.eclipse.jetty.client.api.Request;
import org.eclipse.jetty.client.api.Response;
import org.eclipse.jetty.client.api.Result;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.HttpScheme;
import org.eclipse.jetty.util.BlockingArrayQueue;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class HttpDestination
implements Destination,
Closeable,
Dumpable {
    private static final Logger LOG = Log.getLogger(HttpDestination.class);
    private final AtomicInteger connectionCount = new AtomicInteger();
    private final HttpClient client;
    private final String scheme;
    private final String host;
    private final Destination.Address address;
    private final Queue<HttpExchange> exchanges;
    private final BlockingQueue<Connection> idleConnections;
    private final BlockingQueue<Connection> activeConnections;
    private final RequestNotifier requestNotifier;
    private final ResponseNotifier responseNotifier;
    private final Destination.Address proxyAddress;
    private final HttpField hostField;

    public HttpDestination(HttpClient client, String scheme, String host, int port) {
        this.client = client;
        this.scheme = scheme;
        this.host = host;
        this.address = new Destination.Address(host, port);
        int maxRequestsQueued = client.getMaxRequestsQueuedPerDestination();
        int capacity = Math.min(32, maxRequestsQueued);
        this.exchanges = new BlockingArrayQueue(capacity, capacity, maxRequestsQueued);
        int maxConnections = client.getMaxConnectionsPerDestination();
        capacity = Math.min(8, maxConnections);
        this.idleConnections = new BlockingArrayQueue(capacity, capacity, maxConnections);
        this.activeConnections = new BlockingArrayQueue(capacity, capacity, maxConnections);
        this.requestNotifier = new RequestNotifier(client);
        this.responseNotifier = new ResponseNotifier(client);
        ProxyConfiguration proxyConfig = client.getProxyConfiguration();
        Destination.Address address = this.proxyAddress = proxyConfig != null && proxyConfig.matches(host, port) ? new Destination.Address(proxyConfig.getHost(), proxyConfig.getPort()) : null;
        if (!client.isDefaultPort(scheme, port)) {
            host = host + ":" + port;
        }
        this.hostField = new HttpField(HttpHeader.HOST, host);
    }

    protected BlockingQueue<Connection> getIdleConnections() {
        return this.idleConnections;
    }

    protected BlockingQueue<Connection> getActiveConnections() {
        return this.activeConnections;
    }

    public RequestNotifier getRequestNotifier() {
        return this.requestNotifier;
    }

    public ResponseNotifier getResponseNotifier() {
        return this.responseNotifier;
    }

    @Override
    public String getScheme() {
        return this.scheme;
    }

    @Override
    public String getHost() {
        return this.host;
    }

    @Override
    public int getPort() {
        return this.address.getPort();
    }

    public Destination.Address getConnectAddress() {
        return this.isProxied() ? this.proxyAddress : this.address;
    }

    public boolean isProxied() {
        return this.proxyAddress != null;
    }

    public URI getProxyURI() {
        ProxyConfiguration proxyConfiguration = this.client.getProxyConfiguration();
        String uri = this.getScheme() + "://" + proxyConfiguration.getHost();
        if (!this.client.isDefaultPort(this.getScheme(), proxyConfiguration.getPort())) {
            uri = uri + ":" + proxyConfiguration.getPort();
        }
        return URI.create(uri);
    }

    public HttpField getHostField() {
        return this.hostField;
    }

    public void send(Request request, List<Response.ResponseListener> listeners) {
        if (!this.scheme.equals(request.getScheme())) {
            throw new IllegalArgumentException("Invalid request scheme " + request.getScheme() + " for destination " + this);
        }
        if (!this.getHost().equals(request.getHost())) {
            throw new IllegalArgumentException("Invalid request host " + request.getHost() + " for destination " + this);
        }
        int port = request.getPort();
        if (port >= 0 && this.getPort() != port) {
            throw new IllegalArgumentException("Invalid request port " + port + " for destination " + this);
        }
        HttpConversation conversation = this.client.getConversation(request.getConversationID(), true);
        HttpExchange exchange = new HttpExchange(conversation, this, request, listeners);
        if (this.client.isRunning()) {
            if (this.exchanges.offer(exchange)) {
                if (!this.client.isRunning() && this.exchanges.remove(exchange)) {
                    throw new RejectedExecutionException((Object)((Object)this.client) + " is stopping");
                }
                LOG.debug("Queued {}", new Object[]{request});
                this.requestNotifier.notifyQueued(request);
                Connection connection = this.acquire();
                if (connection != null) {
                    this.process(connection, false);
                }
            } else {
                LOG.debug("Max queued exceeded {}", new Object[]{request});
                this.abort(exchange, new RejectedExecutionException("Max requests per destination " + this.client.getMaxRequestsQueuedPerDestination() + " exceeded for " + this));
            }
        } else {
            throw new RejectedExecutionException((Object)((Object)this.client) + " is stopped");
        }
    }

    @Override
    public void newConnection(Promise<Connection> promise) {
        this.createConnection(new ProxyPromise(promise));
    }

    protected void createConnection(Promise<Connection> promise) {
        this.client.newConnection(this, promise);
    }

    protected Connection acquire() {
        int next;
        int current;
        Connection result = (Connection)this.idleConnections.poll();
        if (result != null) {
            return result;
        }
        final int maxConnections = this.client.getMaxConnectionsPerDestination();
        do {
            if ((next = (current = this.connectionCount.get()) + 1) <= maxConnections) continue;
            LOG.debug("Max connections per destination {} exceeded for {}", new Object[]{current, this});
            return (Connection)this.idleConnections.poll();
        } while (!this.connectionCount.compareAndSet(current, next));
        LOG.debug("Creating connection {}/{} for {}", new Object[]{next, maxConnections, this});
        Promise<Connection> promise = new Promise<Connection>(){

            public void succeeded(Connection connection) {
                HttpDestination.this.process(connection, true);
            }

            public void failed(final Throwable x) {
                HttpDestination.this.client.getExecutor().execute(new Runnable(){

                    @Override
                    public void run() {
                        HttpDestination.this.abort(x);
                    }
                });
            }
        };
        this.createConnection(new ProxyPromise((Promise)promise){

            @Override
            public void succeeded(Connection connection) {
                LOG.debug("Created connection {}/{} {} for {}", new Object[]{next, maxConnections, connection, HttpDestination.this});
                super.succeeded(connection);
            }

            @Override
            public void failed(Throwable x) {
                LOG.debug("Connection failed {} for {}", new Object[]{x, HttpDestination.this});
                HttpDestination.this.connectionCount.decrementAndGet();
                super.failed(x);
            }
        });
        return (Connection)this.idleConnections.poll();
    }

    private void abort(Throwable cause) {
        HttpExchange exchange;
        while ((exchange = this.exchanges.poll()) != null) {
            this.abort(exchange, cause);
        }
    }

    protected void process(Connection connection, boolean dispatch) {
        final HttpConnection httpConnection = (HttpConnection)connection;
        final HttpExchange exchange = this.exchanges.poll();
        if (exchange == null) {
            LOG.debug("{} idle", new Object[]{httpConnection});
            if (!this.idleConnections.offer(httpConnection)) {
                LOG.debug("{} idle overflow", new Object[0]);
                httpConnection.close();
            }
            if (!this.client.isRunning()) {
                LOG.debug("{} is stopping", new Object[]{this.client});
                this.remove(httpConnection);
                httpConnection.close();
            }
        } else {
            Request request = exchange.getRequest();
            Throwable cause = request.getAbortCause();
            if (cause != null) {
                this.abort(exchange, cause);
                LOG.debug("Aborted before processing {}: {}", new Object[]{exchange, cause});
            } else {
                LOG.debug("{} active", new Object[]{httpConnection});
                if (!this.activeConnections.offer(httpConnection)) {
                    LOG.warn("{} active overflow", new Object[0]);
                }
                if (dispatch) {
                    this.client.getExecutor().execute(new Runnable(){

                        @Override
                        public void run() {
                            httpConnection.send(exchange);
                        }
                    });
                } else {
                    httpConnection.send(exchange);
                }
            }
        }
    }

    public void release(Connection connection) {
        LOG.debug("{} released", new Object[]{connection});
        if (this.client.isRunning()) {
            boolean removed = this.activeConnections.remove(connection);
            if (removed) {
                this.process(connection, false);
            } else {
                LOG.debug("{} explicit", new Object[]{connection});
            }
        } else {
            LOG.debug("{} is stopped", new Object[]{this.client});
            this.remove(connection);
            connection.close();
        }
    }

    public void remove(Connection connection) {
        boolean removed = this.activeConnections.remove(connection);
        if (removed |= this.idleConnections.remove(connection)) {
            int open = this.connectionCount.decrementAndGet();
            LOG.debug("Removed connection {} for {} - open: {}", new Object[]{connection, this, open});
        }
        if (!this.exchanges.isEmpty() && (connection = this.acquire()) != null) {
            this.process(connection, false);
        }
    }

    @Override
    public void close() {
        for (Connection connection : this.idleConnections) {
            connection.close();
        }
        this.idleConnections.clear();
        for (Connection connection : this.activeConnections) {
            connection.close();
        }
        this.activeConnections.clear();
        this.abort(new AsynchronousCloseException());
        this.connectionCount.set(0);
        LOG.debug("Closed {}", new Object[]{this});
    }

    public boolean remove(HttpExchange exchange) {
        return this.exchanges.remove(exchange);
    }

    protected void abort(HttpExchange exchange, Throwable cause) {
        Request request = exchange.getRequest();
        HttpResponse response = exchange.getResponse();
        this.getRequestNotifier().notifyFailure(request, cause);
        List<Response.ResponseListener> listeners = exchange.getConversation().getResponseListeners();
        this.getResponseNotifier().notifyFailure(listeners, (Response)response, cause);
        this.getResponseNotifier().notifyComplete(listeners, new Result(request, cause, response, cause));
    }

    public String dump() {
        return ContainerLifeCycle.dump((Dumpable)this);
    }

    public void dump(Appendable out, String indent) throws IOException {
        ContainerLifeCycle.dumpObject((Appendable)out, (Object)(this + " - requests queued: " + this.exchanges.size()));
        ArrayList<String> connections = new ArrayList<String>();
        for (Connection connection : this.idleConnections) {
            connections.add(connection + " - IDLE");
        }
        for (Connection connection : this.activeConnections) {
            connections.add(connection + " - ACTIVE");
        }
        ContainerLifeCycle.dump((Appendable)out, (String)indent, (Collection[])new Collection[]{connections});
    }

    public String toString() {
        return String.format("%s(%s://%s:%d)%s", HttpDestination.class.getSimpleName(), this.getScheme(), this.getHost(), this.getPort(), this.proxyAddress == null ? "" : " via " + this.proxyAddress.getHost() + ":" + this.proxyAddress.getPort());
    }

    private class ProxyPromise
    implements Promise<Connection> {
        private final Promise<Connection> delegate;

        private ProxyPromise(Promise<Connection> delegate) {
            this.delegate = delegate;
        }

        public void succeeded(Connection connection) {
            boolean tunnel;
            boolean bl = tunnel = HttpDestination.this.isProxied() && HttpScheme.HTTPS.is(HttpDestination.this.getScheme()) && HttpDestination.this.client.getSslContextFactory() != null;
            if (tunnel) {
                this.tunnel(connection);
            } else {
                this.delegate.succeeded((Object)connection);
            }
        }

        public void failed(Throwable x) {
            this.delegate.failed(x);
        }

        private void tunnel(final Connection connection) {
            String target = HttpDestination.this.address.getHost() + ":" + HttpDestination.this.address.getPort();
            Request connect = HttpDestination.this.client.newRequest(HttpDestination.this.proxyAddress.getHost(), HttpDestination.this.proxyAddress.getPort()).scheme(HttpScheme.HTTP.asString()).method(HttpMethod.CONNECT).path(target).header(HttpHeader.HOST, target).timeout(HttpDestination.this.client.getConnectTimeout(), TimeUnit.MILLISECONDS);
            connection.send(connect, new Response.CompleteListener(){

                @Override
                public void onComplete(Result result) {
                    if (result.isFailed()) {
                        ProxyPromise.this.failed(result.getFailure());
                        connection.close();
                    } else {
                        Response response = result.getResponse();
                        if (response.getStatus() == 200) {
                            ProxyPromise.this.delegate.succeeded((Object)connection);
                        } else {
                            ProxyPromise.this.failed(new HttpResponseException("Received " + response + " for " + result.getRequest(), response));
                            connection.close();
                        }
                    }
                }
            });
        }
    }
}

