/*******************************************************************************
 * Copyright (c) 2011-2018 The University of York, Aston University.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License, v. 2.0 are satisfied: GNU General Public License, version 3.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-3.0
 *
 * Contributors:
 *     Konstantinos Barmpis - initial API and implementation
 *     Antonio Garcia-Dominguez - add isContainedWithin (for EMF-Splitter integration), use logging
 ******************************************************************************/
package org.eclipse.hawk.epsilon.emc.wrappers;

import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.Objects;

import org.eclipse.epsilon.eol.exceptions.EolRuntimeException;
import org.eclipse.hawk.core.IModelIndexer;
import org.eclipse.hawk.core.graph.IGraphEdge;
import org.eclipse.hawk.core.graph.IGraphNode;
import org.eclipse.hawk.core.graph.IGraphNodeReference;
import org.eclipse.hawk.core.graph.IGraphTransaction;
import org.eclipse.hawk.epsilon.emc.EOLQueryEngine;
import org.eclipse.hawk.graph.ModelElementNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GraphNodeWrapper implements IGraphNodeReference {

	private static final Logger LOGGER = LoggerFactory.getLogger(GraphNodeWrapper.class);

	protected String id;
	protected EOLQueryEngine containerModel;
	protected WeakReference<IGraphNode> node;

	public GraphNodeWrapper(IGraphNode n, EOLQueryEngine containerModel) {

		node = new WeakReference<IGraphNode>(n);
		this.id = n.getId().toString();
		this.containerModel = containerModel;

	}

	@Override
	public IGraphNode getNode() {
		IGraphNode ret = node.get();
		if (ret == null) {
			ret = containerModel.getBackend().getNodeById(id);
			node = new WeakReference<IGraphNode>(ret);
		}
		return ret;
	}

	@Override
	public String getId() {
		return id;
	}

	@Override
	public EOLQueryEngine getContainerModel() {
		return containerModel;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		GraphNodeWrapper other = (GraphNodeWrapper) obj;
		return Objects.equals(containerModel, other.containerModel) && Objects.equals(id, other.id);
	}

	public Object getFeature(String name) throws EolRuntimeException {
		return containerModel.getPropertyGetter().invoke(this, name);
	}

	/**
	 * Returns true if this model element is contained directly or indirectly
	 * within the specified path at the specified repository.
	 */
	public boolean isContainedWithin(String repository, String path) {
		try (IGraphTransaction t = containerModel.getBackend().beginTransaction()) {
			final ModelElementNode men = new ModelElementNode(getNode());
			return men.isContainedWithin(repository, path);
		} catch (Exception e) {
			LOGGER.error(e.getMessage(), e);
		}
		return false;
	}

	@Override
	public String getTypeName() {
		String type = "";

		try (IGraphTransaction t = containerModel.getBackend().beginTransaction()) {
			IGraphNode n = getNode();

			final Iterator<IGraphEdge> itTypeNode = n.getOutgoingWithType(ModelElementNode.EDGE_LABEL_OFTYPE).iterator();
			if (itTypeNode.hasNext()) {
				final IGraphNode typeNode = itTypeNode.next().getEndNode();
				type = typeNode.getProperty(IModelIndexer.IDENTIFIER_PROPERTY).toString();
			} else {
				LOGGER.error("No type node found for node {}", n);
			}

			t.success();
		} catch (Exception e) {
			LOGGER.error(e.getMessage(), e);
		}

		return type;
	}

	@Override
	public String toString() {
		String info = "";

		try (IGraphTransaction t = containerModel.getBackend().beginTransaction()) {
			info += "type:" + getTypeName();
			info += "";
			t.success();
		} catch (Exception e) {
			LOGGER.error(e.getMessage(), e);
		}
		return "GNW|id:" + id + "|" + (info.equals("") ? "[no meta-info]" : info) + "";
	}

	@Override
	public int hashCode() {
		return Objects.hash(containerModel, id);
	}
}
