/*******************************************************************************
 * Copyright (c) 2008 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Daniel Sanders - Initial cut
 *******************************************************************************/
package org.eclipse.higgins.audit.api;

import org.eclipse.higgins.configuration.api.IConfigurableComponent;

public interface AuditRecord extends IConfigurableComponent
{
	
	/**
	 * Set the originator information for the record.
	 *
	 * @param strOriginatorLocationName Location name component of the
	 * originator information.
	 * @param strOriginatorLocationAddress Location address component of the
	 * originator information.
	 * @param strOriginatorServiceType Service type component of the
	 * originator information.
	 * @param strOriginatorAuthAuthority Authorization authority component of the
	 * originator information.
	 * @param strOriginatorPrincipalName Principal name component of the
	 * originator information.
	 * @param strOriginatorPrincipalIdentity Principal identity component of the
	 * originator information.
	 */
	 public void setOriginatorInfo(
		String	strOriginatorLocationName,
		String	strOriginatorLocationAddress,
		String	strOriginatorServiceType,
		String 	strOriginatorAuthAuthority,
		String	strOriginatorPrincipalName,
		String	strOriginatorPrincipalIdentity) throws AuditException; 
	
	/**
	 * Set the event number for an audit record.
	 * 
	 * @param iEventNumber The event number of the detected event. Only event numbers
	 * defined in {@link org.eclipse.higgins.audit.api.AuditEvents AuditEvents} are valid.
	 * Any other event number will result in an exception being thrown.
	 */
	public void setEventNumber(
		int	iEventNumber) throws AuditException;

	/**
	 * Set the outcome in the audit record.
	 * 
	 * @param iOutcome The outcome of the event determined by the caller. Only the
	 * outcome codes defined in {@link org.eclipse.higgins.audit.api.AuditOutcomes
	 * AuditOutcomes} are valid.  Use AuditOutcomes.AUDIT_OUT_NOT_SPECIFIED for
	 * unspecified.
	 */
	public void setOutcome(
		int	iOutcome) throws AuditException;

	/**
	 * Set the initiator information in the audit record.
	 * 
	 * @param strAuthAuthority The authorization authority component of the initiator
	 * information.
	 * @param strDomainSpecificName The domain specific name component of the initiator
	 * information.
	 * @param strDomainSpecificId The domain specific ID component of the initiator information.
	 */
	public void setInitiatorInfo(
		String	strAuthAuthority,
		String	strDomainSpecificName,
		String	strDomainSpecificId) throws AuditException;

	/**
	 * Set the target information in the audit record.
	 * 
	 * @param strTargetLocationName The location name component of the target information.
	 * @param strTargetLocationAddress The location address component of the target information.
	 * @param strTargetServiceType The service type component of the target information.
	 * @param strTargetAuthAuthority The authorization authority component of the target information.
	 * @param strTargetPrincipalName The principal name component of the target information.
	 * @param strTargetPrincipalIdentity The principal identity component of the target information.
	 */
	public void setTargetInfo(
		String	strTargetLocationName,
		String	strTargetLocationAddress,
		String	strTargetServiceType,
		String	strTargetAuthAuthority,
		String	strTargetPrincipalName,
		String	strTargetPrincipalIdentity) throws AuditException;

	/**
	 * Add the event information to the audit record.
	 * 
	 * @param strInfoName The name of the event information.
	 * @param strInfoValue The value for the event information.
	 */
	public void addEventInfo(
		String	strInfoName,
		String	strInfoValue) throws AuditException;

	/**
	 * Format an audit record for output to the audit log.
	 * <p>
	 * <p>
	 * This method formats the audit record for the audit stream. If no prior call
	 * to the {@link org.eclipse.higgins.audit.api.AuditRecord#setTimeStamp setTimeStamp}
	 * method has been made, this method will also set the timestamp for the
	 * record.
	 */
	public String formatRecord() throws AuditException;

	/**
	 * Clear all information in the audit record.
	 */
	public void clearRecord();

	/**
	 * Set a timestamp for the audit record to current time.
	 */
	public void setTimeStamp();
}

