/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Michael McIntosh (IBM Corporation) - initial API and implementation
 *******************************************************************************/ 

package org.eclipse.higgins.configuration.common;

import org.eclipse.higgins.configuration.api.IConfigurableComponent;
import org.eclipse.higgins.configuration.api.IConfigurableComponentFactory;

/**
 * Convenience class that implements code required to get a singleton instance from an ObejctFactory.
 * 
 * @author mikemci at us dot ibm dot com
 */
public class ConfigurableComponentFactoryHelper
{
	
	/**
	 * Load the class by name 
	 * 
	 * @param className the name of the class to load and create an instance of
	 * @return the new instance
	 */
	static public IConfigurableComponent getInstanceFromClassName
		(final String className)
		throws Exception
	{
		final Class theClass = getClassByName(className);
		final Object theObject = theClass.newInstance();
		final IConfigurableComponent theConfigurableComponent  = (IConfigurableComponent)theObject;
		return theConfigurableComponent;
	}
	
	/**
	 * Load the class by name 
	 * 
	 * @param className the name of the class to load and create an instance of
	 * @return the singleton instance
	 */
	static public IConfigurableComponent getSingletonFromClassName
		(final String className)
		throws Exception
	{
		final Class theClass = getClassByName(className);
		final java.lang.reflect.Method theMethod = theClass.getMethod("getInstance", new Class [] {});
		final Object theObject = theMethod.invoke(null, new Class [] {});
		final IConfigurableComponent theConfigurableComponent  = (IConfigurableComponent)theObject;
		return theConfigurableComponent;
	}

	/**
	 * Load the class by name then creates an instance
	 * 
	 * @param factoryClassName the name of the class to load and create an instance from
	 * @return the singleton instance
	 */
	static public IConfigurableComponent getSingletonFromFactory
		(final String factoryClassName)
		throws Exception
	{
		final Class factoryClass = ConfigurableComponentFactoryHelper.getClassByName
			(factoryClassName);
		final Object objectFactory = factoryClass.newInstance();
		final IConfigurableComponentFactory factory = (IConfigurableComponentFactory)objectFactory;
		return factory.getSingletonInstance();
	}
	
	/**
	 * Load the class by name then creates an instance
	 * 
	 * @param factoryClassName the name of the class to load and create an instance from
	 * @return the new instance
	 */
	static public IConfigurableComponent getFromFactory
		(final String factoryClassName)
		throws Exception
	{
		final Class factoryClass = ConfigurableComponentFactoryHelper.getClassByName
			(factoryClassName);
		final Object objectFactory = factoryClass.newInstance();
		final IConfigurableComponentFactory factory = (IConfigurableComponentFactory)objectFactory;
		return factory.getNewInstance();
	}
	
	static private Class getClassByName
		(final String factoryClassName)
		throws Exception
	{
		ClassLoader classLoader = null;
		Exception throw1 = null;
 		Class factoryClass = null;
 		try
 		{
 			factoryClass = Class.forName(factoryClassName);
		}
 		catch (final Exception e)
 		{
 			throw1 = e;
 		}
 		if (factoryClass == null)
 		{
 			classLoader = Thread.currentThread().getContextClassLoader();
 			if (classLoader != null)
 			{
 				try
 				{
 					factoryClass = classLoader.loadClass(factoryClassName);
 				}
 				catch (final Exception e)
 				{
 					if (throw1 != null)
 					{
 						throw throw1;
 					}
 					else
 					{
 						throw e;
 					}
 				}
 			}
 		}
 		return factoryClass;
	}
}
