/*******************************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Greg Byrd (IBM Corporation) - initial implementation
 *******************************************************************************/ 

package org.eclipse.higgins.configuration.xml;

import java.util.Map;

/**
 * This handler returns an object that has already been processed -- i.e., one that
 * comes earlier in the XML configuration file.
 * 
 * <p>The desired object is referred to by its name, and its relative position in the
 * overall settings.  A path-like syntax is used, where '/' is used to denote a
 * subsetting.  A leading '/' means that the first string refers to a top-level setting.
 * No leading '/' means that the first string refers to a setting at the same level
 * as the reference (i.e., a sibling).  Subsettings of Map elements are referred to by name.  Subsettings of list
 * elements are referred to via their position in the list.
 * 
 * <p> Examples:
 * <p> x/y = setting <b>y</b> inside sibling setting <b>x</b></br>
 * /x/y = setting <b>y</b> inside top-level setting <x></br>
 * x/y/3 = element <b>3</b> of setting <b>y</b> of sibling setting <b>x</b>
 * 
 * @author gbyrd
 *
 */
public class ReferenceHandler implements ISettingHandler
{
	static private final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog
		(ReferenceHandler.class.getName());

	/* (non-Javadoc)
	 * @see org.eclipse.higgins.configuration.xml.SettingHandler#getSetting(java.lang.String, java.util.Map, java.util.Map, org.w3c.dom.Element)
	 */
	public Object getSetting
		(String strName,
		String strConfigurationBase,
		Map mapSettingHandlers,
		Map mapSettingsGlobal,
		Object settingsParent,
		org.apache.axiom.om.OMElement omSetting)
		throws Exception
	{
		String strValue = omSetting.getText();
		log.trace("Setting Name: " + strName + " Value: " + strValue);
		
		// References are relative to the top-level setting,
		// which is mapSettingsGlobal.  Setting names at each level
		// are separated by '/', as in: "first/second/third".
		
		String settingNames[] = strValue.split("/", 0);
		Object refObject = settingsParent;
		for (int i=0; i<settingNames.length; i++) {
			if (refObject == null) break;
			if ((i == 0) && (settingNames[0].equals(""))) {
				refObject = mapSettingsGlobal;
				continue;
			}
			// find out whether container is a Map or a List
			boolean isMap = false;
			boolean isList = false;
			Class interfaces[] = refObject.getClass().getInterfaces();
			for (int j=0; j<interfaces.length; j++) {
				if (interfaces[j] == Class.forName("java.util.Map")) {
					isMap = true; 
					break;  // no need to look further
				}
				else if (interfaces[j] == Class.forName("java.util.List")) {
					isList = true;
				}
			}
			
			if (isMap) {
				refObject = ((java.util.Map)refObject).get(settingNames[i]);
			}
			else if (isList) {
				// if list, settingName must be an integer index
				int index = Integer.parseInt(settingNames[i]);
				refObject = ((java.util.List)refObject).get(index);
			}
			else {
				refObject = null;
			}

		}
		
		return refObject;

	}
}
