/*******************************************************************************
 * Copyright (c) 2006-2007 Novell, Inc..
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Jim Sermersheim - Initial cut of interface
 *******************************************************************************/

package org.eclipse.higgins.idas.api;

import java.net.URI;
import java.util.Iterator;

/**
 */
public interface IHasAttributes
{
	/**
	 * Return all attributes of this object<p>
	 * 
	 * @return An Iterator of {@link IAttribute}s.
	 * @throws IdASException
	 */
	public Iterator getAttributes() throws IdASException;
	
	/**
	 * Returns the specified attribute of this object.
	 * @param attrID The identifier of the attribute to be returned.
	 * @return An {@link IAttribute} or {@code null} if none exits.
	 * @throws IdASException
	 */
	public IAttribute getAttribute(URI attrID) throws IdASException;
	
	/**
	 * Returns the specified attribute of this object as an ISingleValuedAttribute. 
	 * The caller of this is assumed to have knowledge that the attribute specified in 
	 * attrID is in fact a single-valued attribute (as defined by the model for
	 * that attribute).
	 * @param attrID The identifier of the attribute to be returned.
	 * @return An {@link IAttribute} or {@code null} if none exits.
	 * @throws IdASException
	 * @throws NotSingleValuedAttributeException when the specified attrID is not
	 * specified by its model as a single-valued attribute.
	 */
	public ISingleValuedAttribute getSingleValuedAttribute(URI attrID) throws IdASException, NotSingleValuedAttributeException;

	/**
	 * Creates a new Attribute for this container of attributes (typically {@link IEntity}).
	 * This call is typically followed by one or more calls to 
	 * {@link IAttribute#addValue}, {@link IAttribute#addSimpleValue}, 
	 * {@link IAttribute#addComplexValue}, (@link IHasMetadata#addMetadata} on
	 * the returned IAttribute
	 * Note: If the specified attribute already exists, the subsequently added 
	 * values are added to the entity, adding to the existing values of that attribute
	 * (subject to any model restrictions).
	 * Note: This operation is only applied to any backing data store after
	 * {@link IContext#applyUpdates()} is called.
	 * @param attrID The URI specifying the identifier of the Attribute being created
	 * @throws {@link IdASException}
	 * @throws {@link InvalidTypeException} when the attribute type is invalid
	 */
	public IAttribute addAttribute(URI attrID) throws IdASException, InvalidTypeException;	
	
	/**
	 * Creates a new Attribute for this container of 
	 * attributes (typically {@link IEntity}) by copying the data from
	 * the passed IAttribute.
	 * Note: This operation is only applied to any backing data store after
	 * IContext.applyUpdates() is called.
	 * @param copyFrom an {@link IAttribute} from which data is copied to construct a new 
	 * Attribute
	 * @throws {@link IdASException}
	 */
	public IAttribute addAttribute(IAttribute copyFrom) throws IdASException;
	
	/**
	 * Deletes the attribute specified by attrID. 
	 * On one hand, this is a convenience method (in that the caller can
	 * call this rather than fetching the attribute using 
	 * {@link #getAttribute(URI)} and then calling remove on the returned IAttribute. 
	 * More importantly however is that this method allows
	 * an attribute to be removed when the caller has permissions 
	 * to remove, but no permissions to read that attribute.
	 * Note: This operation is only applied to any backing data store after
	 * IContext.applyUpdates() is called.
	 * @param attrID identifies the attribute to be removed
	 */
	public void removeAttribute (URI attrID) throws IdASException;

	/**
	 * Deletes the specified value from the specified attrID. 
	 * This can only be used on simple attributes (not on complex attributes).  
	 * On one hand, this is a convenience method (in that the caller can
	 * call this rather than fetching the attribute using 
	 * {@link #getAttribute(URI)}, then fetching the 
	 * appropriate value from the returned IAttribute, 
	 * and then calling remove on that value.
	 * More importantly however is that this method allows
	 * an attribute value to be removed when the caller has 
	 * permissions to remove values, but no permissions to 
	 * read that attribute or its values.
	 * Note: This operation is only applied to any backing data store after
	 * IContext.applyUpdates() is called.
	 * @param attrID identifies the simple attribute to be removed
	 * @param value identifies the attribute value to be removed. 
	 * Note that this represents the data backing the simple attribute and is not an ISimpleAttribute itself.
	 */
	public void removeAttributeValue (URI attrID, Object value) throws IdASException;

	/**
	 * Deletes an attribute's value(s) based on the information passed in attr.
	 * The attribute type of the passed attr signifies which attribute is
	 * being affected, and the value or values passed in attr signify the
	 * attribute values to be removed.
	 * TODO: document the expected behavior when all passed in values are not
	 * found in the attribute.  Or remove this method and add one which may be 
	 * used to remove a single complex attribute value. 
	 * This is simply a convenience method which is equal to calling 
	 * {@link #removeAttributeValue(URI, Object)} once for each value in attr.
	 * Note: This operation is only applied to any backing data store after
	 * IContext.applyUpdates() is called.
	 * @param attr Holds one or more values to be deleted from the 
	 * attribute (named in attr)
	 */
	public void removeAttributeValue (IAttribute attr) throws IdASException;
	
	/**
	 * Returns true if the passed attribute set is equal to this one.
	 * The sets are compared for size, and then each attribute 
	 * in the set is compared for equality
	 * Note: This operation is only applied to any backing data store after
	 * IContext.applyUpdates() is called.
	 * @param attributes the set of attributes to compare to this one
	 * @return true if the sets are equal, false otherwise.
	 * @throws IdASException
	 */
	public boolean equals (IHasAttributes attributes) throws IdASException;
}
