/*******************************************************************************
 * Copyright (c) 2007-2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tie Li  - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.crpps.ui;

import higgins.login.demo.HigginsLogin;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.higgins.crpps.service.CardHistoryBean;
import org.eclipse.higgins.crpps.service.CardHistoryManager;
import org.eclipse.higgins.crpps.service.InfoCardController;
import org.eclipse.higgins.crpps.service.UserInterfaceInfoBean;
import org.eclipse.higgins.crpps.ui.dialogs.BrowserDialog;
import org.eclipse.higgins.crpps.ui.dialogs.ImportCardstoreDialog;
import org.eclipse.higgins.crpps.ui.dialogs.PersonalCardCreationDialog;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.CardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCard;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.forms.events.ExpansionAdapter;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.events.IExpansionListener;
import org.eclipse.ui.forms.events.IHyperlinkListener;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;
import org.eclipse.ui.part.ViewPart;

public class CardSelectorView extends ViewPart implements ISelectionProvider{
	private static Log log = LogFactory.getLog(CardSelectorView.class);
		
	private FormToolkit toolkit;
	private Form main_panel;
	private FormText rp_info;
	private ExpandableComposite rp_detail_comp; 
	private FormText rp_detail_info;
	private ScrolledForm rp_info_scroller;
	private ScrolledComposite scrolled_card_panel;
	private Composite card_panel;
	
    protected Composite currentWidget = null;
    private Vector cardComps = null;
	
    public static final String ICARD = "ICARD";
    public static final String ICARD_HISTORY = "ICARD_HISTORY";
    public static final String CARD_COMPOSITE = "CARD_COMPOSITE";
    public static final String CARD_LABEL = "CARD_LABEL";
    public static final String CARD_IMAGE = "CARD_IMAGE";
    public static final String SELECT_CARD_IMAGE = "SELECT_CARD_IMAGE";
    
    private final Point imageSize = new Point(100, 60);
    private final Point selectImageSize = new Point(100, 60);//new Point(120, 80);
    private final Point cardSize = new Point(110, 90);
    private final Point selectCardSize = new Point(110, 90);//new Point(130, 110);

    private Color blue = Display.getCurrent().getSystemColor(SWT.COLOR_DARK_BLUE);
    private Color white = Display.getCurrent().getSystemColor(SWT.COLOR_WHITE);
    private Color black = Display.getCurrent().getSystemColor(SWT.COLOR_BLACK);
        
    public Font sysFont;
    public Font selectFont;
    
    private UserInterfaceInfoBean uiib;
    
    private List listeners = new ArrayList();
    
    private String rp_purl = "";
    
	public CardSelectorView() {
		InfoCardController.getInst().setSelectorView(this);
		
		selectFont = new Font(Display.getCurrent(), "Ariel", 8, SWT.BOLD);
		sysFont = new Font(Display.getCurrent(), "Ariel", 7, SWT.NONE);
		cardComps = new Vector();
	}

	public void createPartControl(Composite parent) {
		getSite().setSelectionProvider(this);
		
		toolkit = new FormToolkit(parent.getDisplay());
		main_panel = toolkit.createForm(parent);
		main_panel.getBody().setLayout(new FormLayout());
		
		//----Title bar----
		main_panel.setText("Select an i-card for the relying party site");
		Path path = new Path("/image/form_banner.gif");
		URL url = FileLocator.find(Platform.getBundle("org.eclipse.higgins.crpps"), 
			path, null);
		ImageDescriptor imageDesc = ImageDescriptor.createFromURL(url);
		main_panel.setBackgroundImage(imageDesc.createImage());
		
		//----Tool bar----
		Composite toolBarComp = toolkit.createComposite(main_panel.getBody());		
		FormData td = new FormData();
		td.top = new FormAttachment(0, 0);
		td.left = new FormAttachment(0, 0);
		td.right = new FormAttachment(100, 0);
		toolBarComp.setLayoutData(td);
		toolBarComp.setLayout(new RowLayout());
		
			Button newCardButton = toolkit.createButton(toolBarComp, "New", SWT.PUSH);
			newCardButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					createNewCard();
				}
			});
			Button importCardButton = toolkit.createButton(toolBarComp, "Import", SWT.PUSH);
			importCardButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					importCard();
				}
			});
		
		Composite separator = toolkit.createCompositeSeparator(main_panel.getBody());
		td = new FormData();
		td.top = new FormAttachment(toolBarComp, 5);
		td.left = new FormAttachment(0, 0);
		td.right = new FormAttachment(100, 0);
		td.height = 1;
		separator.setLayoutData(td);
		
		//----RP info panel----
		rp_info_scroller = toolkit.createScrolledForm(main_panel.getBody());
		rp_info_scroller.setExpandHorizontal(false);
		td = new FormData();
		td.top = new FormAttachment(separator, 5);
		td.left = new FormAttachment(0, 0);
		td.right = new FormAttachment(100, 0);	
		td.bottom = new FormAttachment(22, 0);
		rp_info_scroller.setLayoutData(td);
		TableWrapLayout twl = new TableWrapLayout();
		twl.numColumns = 1;
		rp_info_scroller.getBody().setLayout(twl);
		
			StringBuffer buf = new StringBuffer();
			buf.append("<form>");
			buf.append("<p><b>Choose an i-card for site </b></p>");
			buf.append("<p><b>Required information : </b></p>");
			buf.append("<p><b>Optional information : </b></p>");
			buf.append("</form>");
			
			rp_info = toolkit.createFormText(rp_info_scroller.getBody(), true);
			TableWrapData twd = new TableWrapData();			
			twd.colspan = 1;
			rp_info.setLayoutData(twd);
			rp_info.setText(buf.toString(), true, false);
		
		Composite separator1 = toolkit.createCompositeSeparator(main_panel.getBody());
		td = new FormData();
		td.top = new FormAttachment(rp_info_scroller, 1);
		td.left = new FormAttachment(0, 0);
		td.right = new FormAttachment(100, 0);
		td.height = 1;
		separator1.setLayoutData(td);	
					
		//----The rp detail panel----
		this.rp_detail_comp = toolkit.createExpandableComposite(
				main_panel.getBody(),
				ExpandableComposite.TREE_NODE | ExpandableComposite.CLIENT_INDENT);
		td = new FormData();
		td.top = new FormAttachment(separator1, 5);
		td.left = new FormAttachment(0, 0);
		this.rp_detail_comp.setLayoutData(td);
		this.rp_detail_comp.setText("See the relying party detailed information");
		this.rp_detail_info = toolkit.createFormText(rp_detail_comp, true);
		this.rp_detail_comp.setClient(rp_detail_info);	
		buf = new StringBuffer();
		buf.append("<form>");
		buf.append("<p>Organization name : </p>");
		buf.append("<p>Country : </p>");
		buf.append("<p>Private statement : </p>");
		buf.append("<p>This site is verified by </p>");
		buf.append("</form>");
		rp_detail_info.setText(buf.toString(), true, false);
		rp_detail_comp.addExpansionListener(new ExpansionAdapter(){  
			public void expansionStateChanged(ExpansionEvent e) {
			   main_panel.layout(true);
			}
		});
													
		//----The card selector panel----
		scrolled_card_panel = new ScrolledComposite(main_panel.getBody(), 
				SWT.BORDER | SWT.V_SCROLL);
		scrolled_card_panel.setBackground(
				Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));
		td = new FormData();
		td.top = new FormAttachment(this.rp_detail_comp, 5);
		td.left = new FormAttachment(0, 0);
		td.right = new FormAttachment(100, 0);
		td.bottom = new FormAttachment(100, 0);
		scrolled_card_panel.setLayoutData(td);
		
		card_panel = new Composite(scrolled_card_panel, SWT.NONE);
		scrolled_card_panel.setContent(card_panel);
		scrolled_card_panel.setExpandVertical(true);
		scrolled_card_panel.setExpandHorizontal(true);
		scrolled_card_panel.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent e) {
				Rectangle r = scrolled_card_panel.getClientArea();
				scrolled_card_panel.setMinSize(
					scrolled_card_panel.computeSize(r.width, SWT.DEFAULT));
			}
		});
		
		RowLayout rowlo = new RowLayout();
		rowlo.wrap = true;
		card_panel.setLayout(rowlo);
		card_panel.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));
					
		rp_info_scroller.layout();
		card_panel.layout();
		main_panel.layout();
	}

	public void reset(){
		log.info("reset the card selector view part");
		CardSelection cs = new CardSelection(CardSelection.TYPE_RESET);
		this.fireSelectionChange(cs);
	}
	
	public void notifyUIInfo(String url, UserInterfaceInfoBean _uiib){
		this.uiib = _uiib;
		
		StringBuffer buf = new StringBuffer();
		buf.append("<form>");
		buf.append("<p>Choose an i-card for site <b>" + uiib.get_name() + "</b></p>");
		buf.append("<p><b>Required information : </b>");		
		IPolicy policy = uiib.getPolicy();
		String[] labels = policy.getSummary();	
		if(labels.length>0){
			for(int i=0; i<labels.length-1; i++){
				buf.append(labels[i]);
				buf.append(", ");
			}
			buf.append(labels[labels.length-1]);
		}
		buf.append("</p>");
				
		buf.append("<p><b>Optional information : </b>");
		if(policy instanceof CardSpacePolicy){
			CardSpacePolicy csp = (CardSpacePolicy) policy;
			String[] optClaims = csp.getOptionalClaimSummary();
			if(optClaims!=null && optClaims.length>0){
				for(int i=0; i<optClaims.length-1; i++){
					buf.append(optClaims[i]);
					buf.append(", ");
				}
				buf.append(optClaims[optClaims.length-1]);
			}
		}	
		buf.append("</p>");
		buf.append("</form>");
		rp_info.setText(buf.toString(), true, false);
		rp_info_scroller.layout();		
		rp_info_scroller.pack();
		
		if(policy instanceof CardSpacePolicy){
			CardSpacePolicy csp = (CardSpacePolicy) policy;
			rp_purl = csp.getPrivacyUrl();
		}
		
		buf = new StringBuffer();
		buf.append("<form>");
		buf.append("<p><b>Organization name : </b>" + uiib.get_orgnization() +"</p>");
		buf.append("<p><b>Country : </b>" + uiib.get_country() + "</p>");
		buf.append("<p><b>Private statement : </b><a href=\"" 
			+ rp_purl + "\">" + rp_purl +"</a></p>");
		buf.append("<p>This site is verified by <b>" + uiib.get_verifier() +"</b></p>");
		buf.append("</form>");
		rp_detail_info.setText(buf.toString(), true, true);
		rp_detail_info.addHyperlinkListener(new IHyperlinkListener(){
			public void linkActivated(HyperlinkEvent e) {
				BrowserDialog bd = new BrowserDialog(getSite().getShell());
				bd.open();
				bd.setURL(rp_purl);
			}

			public void linkEntered(HyperlinkEvent e) {}
			public void linkExited(HyperlinkEvent e) {}
		});
				
		Vector cards = (Vector)uiib.getMatchCards().clone();		
		Vector freq_cards = new Vector();
		Vector freq_card_hints = new Vector();
		
		Vector chbs= CardHistoryManager.getInst().searchCardHistoryBean(url);
		if(chbs!=null && chbs.size()>0){
			for(int i=0; i<chbs.size(); i++){
				CardHistoryBean chb = (CardHistoryBean)chbs.get(i);
				String id = chb.getCard_id();
				for(int j=0; j<cards.size(); j++){
					ICard card = (ICard) cards.get(j);
					if(card.getID().equals(id)){
						freq_cards.add(card);
						freq_card_hints.add(chb);
						cards.remove(card);
						continue;
					}
				}
			}
		}
		
		clearCardPanel();	
		
		if(freq_cards.size()>0){
			Label freqLabel = toolkit.createLabel(this.card_panel, 
				"The following cards has been used for this site:");
			freqLabel.setData("need_to_be_disposed");
			RowData rd = new RowData();
			rd.width = this.card_panel.getBounds().width;
			freqLabel.setLayoutData(rd);
			for(int i=0; i<freq_cards.size(); i++){
				ICard card = (ICard) freq_cards.get(i);
				CardHistoryBean chb = (CardHistoryBean) freq_card_hints.get(i);
				createCardComposite(card, chb, true);
			}
		}
		
		Label matchLabel = toolkit.createLabel(this.card_panel, 
			"The following cards are available for this site:");
		matchLabel.setData("need_to_be_disposed");
		RowData rd = new RowData();
		rd.width = this.card_panel.getBounds().width;
		matchLabel.setLayoutData(rd);
		for(int i=0; i<cards.size(); i++){
			ICard card = (ICard) cards.get(i);
			createCardComposite(card, null, true);
		}	
		
		Label unmatchLabel = toolkit.createLabel(this.card_panel, 
			"The following cards are not matched:");
		unmatchLabel.setData("need_to_be_disposed");
		rd = new RowData();
		rd.width = this.card_panel.getBounds().width;
		unmatchLabel.setLayoutData(rd);
		cards = (Vector) uiib.getUnmatchCards().clone();
		for(int i=0; i<cards.size(); i++){
			ICard card = (ICard) cards.get(i);
			createCardComposite(card, null, false);
		}
		
		card_panel.layout(true);
		
		Rectangle r = scrolled_card_panel.getClientArea();
		scrolled_card_panel.setMinSize(
			scrolled_card_panel.computeSize(r.width, SWT.DEFAULT));
		
		main_panel.layout(true);
	}
	
	private void clearCardPanel(){		
		for(int i=0; i<this.cardComps.size(); i++){
			Composite cardComp = (Composite) this.cardComps.get(i);
			cardComp.dispose();
		}		
		
		Control[] controls = this.card_panel.getChildren();
		for(int i=0; i<controls.length; i++){
			Object data = controls[i].getData();
			if(data!=null && (data instanceof String)){
				if("need_to_be_disposed".equals((String)data)){
					if(!(controls[i].isDisposed())){
						controls[i].dispose();
					}
				}
			}
		}
		
		this.cardComps.clear();
		this.currentWidget = null;
	}
	
    private void createCardComposite(ICard card, CardHistoryBean chb, boolean match) {
    	log.info("create card composite for card " + card.getName());
    	Group cardComp = new Group(card_panel, SWT.SHADOW_ETCHED_OUT);    	    	
		cardComp.setText(card.getType());
		if(card instanceof PersonalCard){
			int pinStatus = ((PersonalCard)card).getPinStatus();
			if(PersonalCard.LOCKED == pinStatus){
				cardComp.setText("LOCKED");
			}				
		}
    	
    	this.cardComps.add(cardComp);
    	cardComp.setBackground(Display.getCurrent()
    		.getSystemColor(SWT.COLOR_WHITE));
    	RowData rd = new RowData();
    	rd.width = cardSize.x;
    	rd.height = cardSize.y;
    	cardComp.setLayoutData(rd);
    	cardComp.setLayout(new RowLayout(SWT.VERTICAL));

    	Composite imageComp = new Composite(cardComp, SWT.NONE);
    	byte[] img_rawdata = card.getImage();
    	String imageType = card.getImageType();
    	Image image = null;

		if (img_rawdata == null) {
			log.warn("Error when loading ICard image of type [" 
					+ imageType + "] for Icard [" + card.getID() + "]");
			image = CardListView.INFOCARD;
		}
		if (image == null) {
			ByteArrayInputStream bis = new ByteArrayInputStream(img_rawdata);
			try {
				image = new Image(Display.getCurrent(), bis);
			} catch(org.eclipse.swt.SWTException e) {
				// in case of failure try to decode the stream as base64 and reconstruct the image
				log.warn("Could not construct the image, trying again after base64 decoding");
				img_rawdata = Base64.decodeBase64(img_rawdata);
				bis = new ByteArrayInputStream(img_rawdata);
				try {
					image = new Image(Display.getCurrent(), bis);
				} catch(org.eclipse.swt.SWTException e1) {
					log.warn("Could not construct the image using base64 decoding: we keep the default image");
				}
			}
		}
		
    	ImageData imd = image.getImageData().scaledTo(imageSize.x, imageSize.y);
    	Image newimg = new Image(Display.getCurrent(), imd);
    	imageComp.setBackgroundImage(newimg);
    	rd = new RowData();
    	rd.width = imageSize.x;
    	rd.height = imageSize.y;
    	imageComp.setLayoutData(rd);
    	imageComp.setData(CARD_IMAGE, newimg);
    	imd = image.getImageData().scaledTo(selectImageSize.x,
    		selectImageSize.y);
    	newimg = new Image(Display.getCurrent(), imd);
    	imageComp.setData(SELECT_CARD_IMAGE, newimg);

    	String name = card.getName();
    	Label label = new Label(cardComp, SWT.NONE);
    	label.setText(name);
    	label.setBackground(white);
    	label.setForeground(blue);

    	cardComp.setToolTipText(name);

    	if(match){
	    	CardMouseListener cml = new CardMouseListener(imageComp);
	    	imageComp.addMouseListener(cml);
	    	imageComp.setData(ICARD, card);
	    	if(chb!=null){
	    		imageComp.setData(ICARD_HISTORY, chb);
	    	}
	    	imageComp.setData(CARD_COMPOSITE, cardComp);
	    	imageComp.setData(CARD_LABEL, label);
    	}
    }

	public void setFocus() {
		this.card_panel.setFocus();
	}
	
    /**
     * @author bus
     * 
     */
    class CardMouseListener implements MouseListener {
		private Composite widget;
	
		public CardMouseListener(Composite widgetP) {
		    widget = widgetP;
		}
	
		public void mouseDoubleClick(MouseEvent arg0) {
		    log.info("mouseDoubleClick()");
		    ICard card = selectItem();
		    CardSelection cs = new CardSelection(CardSelection.TYPE_INFORM);
		    cs.setCard(card);
		    cs.setUiib(uiib);
		    fireSelectionChange(cs);
		}
	
		public void mouseDown(MouseEvent arg0) {
		    log.info("mouseDown()");
		    ICard card = selectItem();		    
		    CardSelection cs = new CardSelection(CardSelection.TYPE_INFORM);
		    cs.setCard(card);
		    cs.setUiib(uiib);
		    Object data = currentWidget.getData(ICARD_HISTORY);
		    if(data!=null){
		    	cs.setCardHistory((CardHistoryBean) data);
		    }
		    fireSelectionChange(cs);
		}
	
		public void mouseUp(MouseEvent arg0) {
		    log.info("mouseUp()");
		}
	
		private ICard selectItem() {
		    log.info("selectItem()");
		    Label cardLabel = null;
		    Composite cardComposite = null;
	
		    ICard currentSelectCard = (ICard) widget.getData(ICARD);
		    // currentSelectCard = (ICard) icards.get(index);
	
		    // un-select old graphics card
		    if (currentWidget != null) {
				cardComposite = (Composite) currentWidget.getData(CARD_COMPOSITE);
				cardComposite.setBackground(blue);
				cardComposite.setForeground(black);
				RowData rd = new RowData();
				rd.width = imageSize.x;
				rd.height = imageSize.y;
				currentWidget.setLayoutData(rd);
				cardComposite.setBackground(white);
				cardLabel = (Label) currentWidget.getData(CARD_LABEL);
				cardLabel.setBackground(white);
				cardLabel.setForeground(blue);
				cardLabel.setFont(sysFont);
				cardComposite.setSize(cardSize);
				Image image = (Image) currentWidget.getData(CARD_IMAGE);
				log.info("select image=" + image);
				currentWidget.setBackgroundImage(image);
				currentWidget.setSize(imageSize);
				rd = new RowData();
				rd.width = cardSize.x;
				rd.height = cardSize.y;
				cardComposite.setLayoutData(rd);
		    }
	
		    // select the new graphics card
		    cardComposite = (Composite) widget.getData(CARD_COMPOSITE);
		    cardComposite.setForeground(white);
		    RowData rd = new RowData();
		    rd.width = selectImageSize.x;
		    rd.height = selectImageSize.y;
		    widget.setLayoutData(rd);
		    cardComposite.setBackground(blue);
		    cardLabel = (Label) widget.getData(CARD_LABEL);
		    cardLabel.setBackground(blue);
		    cardLabel.setForeground(white);
		    cardLabel.setFont(selectFont);
		    Image image = (Image) widget.getData(SELECT_CARD_IMAGE);
		    log.info("image=" + image);
		    widget.setBackgroundImage(image);
		    widget.setSize(selectImageSize);
		    cardComposite.setSize(selectCardSize);
		    currentWidget = widget;
		    rd = new RowData();
		    rd.width = selectCardSize.x;
		    rd.height = selectCardSize.y;
		    cardComposite.setLayoutData(rd);
	
		    cardComposite.getParent().pack();
	
		    return currentSelectCard;
		}
    }
    
	public void createNewCard() {
		Shell shell = this.getViewSite().getShell();
		PersonalCardCreationDialog pccd = new PersonalCardCreationDialog(shell, null);
		pccd.setBlockOnOpen(true);
		int ret = pccd.open();
		ICard icard = pccd.getPersonalCard();
		if (icard != null) {
			log.info("icard:\n" + icard.getID());
			try {
				PerspectiveCardStore.getSecureCardProvider().addCard(null, icard);
			} catch (Exception e) {
				log.error("Error when adding card  [" + icard.getName() + "]", e);
				MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
				messageBox.setMessage("Error when adding card:"
						+ icard.getName() + "\n\n" + e);
				messageBox.setText("Error while adding a card");
				messageBox.open();
			}
			insertNewCardToView(new ICard[] { icard });
		}
	}
	
	private void insertNewCardToView(ICard[] icards) {
		for (int i = 0; i < icards.length; i++) {
			ICard card = icards[i];
			createCardComposite(card, null, true);
		}

		this.card_panel.layout(true);
	}
	
	public void importCard() {
		Shell shell = this.getViewSite().getShell();// ApplicationWorkbenchWindowAdvisor.getShell();
		FileDialog fileDialog = new FileDialog(shell, SWT.OPEN | SWT.MULTI);
		fileDialog.setFilterExtensions(new String[] { "*.crd", "*" });
		String ret = fileDialog.open();
		if (ret == null)
			return;
		String[] files1 = fileDialog.getFileNames();
		String files[] = new String[files1.length];
		for (int i = 0; i < files1.length; i++) {
			if (fileDialog.getFilterPath().endsWith(File.separator))
				files[i] = fileDialog.getFilterPath() + files1[i];
			else
				files[i] = fileDialog.getFilterPath() + File.separator
						+ files1[i];
		}
		importCards(files);
	}

	public void importCards(String[] files) {
		String filename;
		for (int i = 0; i < files.length; i++) {
			filename = files[i];
			if (filename.toLowerCase().endsWith(".crd")) {
				log.info("insertNewCardToView(" + filename + ")");
				ICard icard = null;

				try {
					icard=PerspectiveCardStore.getSecureCardProvider()
						.importCard(null, filename);
				} catch (Exception e) {
					log.error("Error when import file [" + filename + "]", e);
					MessageBox messageBox = new MessageBox(this.getSite().getShell(), 
							SWT.ERROR_IO);
					messageBox.setMessage("Error when importing File:"
							+ filename + "\n\n" + e);
					messageBox.setText("Error while importing");
					messageBox.open();
				}
				if (icard != null) {
					log.info("icard:\n" + icard.getID());
					insertNewCardToView(new ICard[] { icard });
				}
			} else if (filename.toLowerCase().endsWith(".crds")) {
				importCardStore(filename);
			}
		}
	}
	
	public void importCardStore(String filename) {
		ImportCardstoreDialog dialog = new ImportCardstoreDialog(card_panel.getShell(),
				filename);
		int rc = dialog.open();
		List cardList = dialog.getCardsList();
		Iterator iter = cardList.iterator();
		try {
			while (iter.hasNext()) {
				// import selected cards
				ICard card = (ICard) iter.next();
				// copy to current cardstore
				PerspectiveCardStore.getSecureCardProvider().addCard(null, card);
				// add to the list of infocards "from" current store
				CardListView.getInstance().icards.add(card);
				// create a virtualization for the card and add the virtualization to the view of the current cardstore
				insertNewCardToView(new ICard[]{card} );
			}
		} catch (Exception e) {
			log.error("Error when importing file [" + filename + "]", e);
			MessageBox messageBox = new MessageBox(card_panel.getShell(), SWT.ERROR_IO);
			messageBox.setMessage("Error when importing cards from File:"
					+ filename + "\n\n" + e);
			messageBox.setText("Error while importing");
			messageBox.open();
		}
	}

	private void fireSelectionChange(CardSelection cs){
		for(int i = 0; i < listeners.size(); i++){
			ISelectionChangedListener listener = (ISelectionChangedListener)listeners.get(i);
			SelectionChangedEvent event = new SelectionChangedEvent(this, cs);
			listener.selectionChanged(event);
		}
	}
	
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		if(!listeners.contains(listener)){
			listeners.add(listener);
		}
	}

	public ISelection getSelection() {
		return null;
	}

	public void removeSelectionChangedListener(ISelectionChangedListener listener) {
		if(listeners.contains(listener)){
			listeners.remove(listener);
		}
	}

	public void setSelection(ISelection selection) {
		
	}
}
