/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Thomas Gross  (IBM Corporation) - initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.crpps.util;

import java.security.SecureRandom;
import java.util.BitSet;


public class UUIDHelper {
    public static final boolean USE_JDK_UUID = false;

    /**
     * 
     * return the UUID using axiom library. he first return is random, every further call will only return the previous + 1
     * 
     * @return UUID in form of an URN
     */
    public static String hostUUID() {
	    return org.apache.axiom.om.util.UUIDGenerator.getUUID();
    }

    /**
         * Generates a random 16-byte String in UUID format. The output is
         * compliant to RFC4122 (http://www.ietf.org/rfc/rfc4122.txt), version
         * 4, that is UUIDs with random- or pseudorandom basis.
         * 
         * @return UUID String in RFC format with "-" delimeters.
         */
    public static String randomUUID() {
/*	if (USE_JDK_UUID) {
	    return java.util.UUID.randomUUID().toString();
	} else */ {
	    /*
                 * FROM RFC 4122: UUID generation 4.4. Algorithms for Creating a
                 * UUID from Truly Random or Pseudo-Random Numbers
                 * 
                 * The version 4 UUID is meant for generating UUIDs from
                 * truly-random or pseudo-random numbers.
                 * 
                 * The algorithm is as follows:
                 * 
                 * o Set the two most significant bits (bits 6 and 7) of the
                 * clock_seq_hi_and_reserved to zero and one, respectively.
                 * 
                 * o Set the four most significant bits (bits 12 through 15) of
                 * the time_hi_and_version field to the 4-bit version number
                 * from Section 4.1.3.
                 * 
                 * o Set all the other bits to randomly (or pseudo-randomly)
                 * chosen values.
                 * 
                 * FORMAT: 0 1 2 3 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2
                 * 3 4 5 6 7 8 9 0 1
                 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+ |
                 * time_low |
                 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+ |
                 * time_mid | time_hi_and_version |
                 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
                 * |clk_seq_hi_res | clk_seq_low | node (0-1) |
                 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+ |
                 * node (2-5) |
                 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
                 */

	    // 1. generate random byte array for full UUID length
	    SecureRandom random = new SecureRandom();
	    byte bytes[] = new byte[16];
	    do {
		random.nextBytes(bytes);
	    } while(bytes[0] == 0);

	    // 2. replace single bits as necessary for RFC compliance.
	    // BitSet as operational basis
	    BitSet bits = getBitsFromByteArray(bytes);

	    // Standard RFC compliance.
	    // Set bits 6 and 7 of clock_seq_hi_res = 64 + 6/7 to '01'
	    bits.clear(64 + 6);
	    bits.set(64 + 7);

	    // Set time_hi_and_version to '0100': 48 + 12-15
	    bits.clear(48 + 12);
	    bits.set(48 + 13);
	    bits.clear(48 + 14);
	    bits.clear(48 + 15);

	    bytes = getByteArrayFromBits(bits);
	    // 3. create RFC formed output.
	    return formatUUID(bytes);
	}
    }

    /**
         * Transforms a BitSet into a byte[].
         * 
         * @param bits
         * @return byte array corresponding to the BitSet.
         */
    private static byte[] getByteArrayFromBits(BitSet bits) {
	byte[] bytes = new byte[(bits.length() +7) / 8 ];

	int i = 0;
	for (; i < bits.length(); i++) {
	    if (bits.get(i)) {
		bytes[bytes.length - i / 8 - 1] |= 1 << (i % 8);
	    }
	}

	return bytes;
    }

    /**
         * Transforms a byte[] to a BitSet.
         * 
         * @param bytes
         * @return BitSet with with corresponding boolean assignment
         */
    private static BitSet getBitsFromByteArray(byte[] bytes) {
	BitSet bits = new BitSet(bytes.length * 8);
	int i = 0;
	int bitlength = bytes.length * 8;
	for (; i < bitlength; i++) {
	    if (isBitSet(bytes[bytes.length - i / 8 - 1], i % 8)) {
		bits.set(i);
	    }
	}
	return bits;
    }

    private static boolean isBitSet(byte b, int index) {
	return (b & (1 << index)) > 0;
    }

    /**
         * Creates an RFC4122 conform output: UUID = time-low "-" time-mid "-"
         * time-high-and-version "-" clock-seq-and-reserved clock-seq-low "-"
         * node
         * 
         * @param bytes
         * @return RFC UUID formated String.
         * @throws IndexOutOfBoundsException
         *                 if the bytes array is smaller than 16.
         */
    private static String formatUUID(byte[] bytes) {
	if (bytes.length < 16)
	    throw new IndexOutOfBoundsException(
		    "Byte basis for UUID formatting must be of 16 bytes length (found: "+bytes.length+").");
	StringBuffer buffer = new StringBuffer(20);
	int i = 0;
	for (; i < 4; i++) {
	    appendHexStringFromByte(buffer, bytes[i]);
	}
	buffer.append("-");
	for (; i < 6; i++) {
	    appendHexStringFromByte(buffer, bytes[i]);
	}
	buffer.append("-");
	for (; i < 8; i++) {
	    appendHexStringFromByte(buffer, bytes[i]);
	}
	buffer.append("-");
	for (; i < 10; i++) {
	    appendHexStringFromByte(buffer, bytes[i]);
	}
	buffer.append("-");
	for (; i < bytes.length; i++) {
	    appendHexStringFromByte(buffer, bytes[i]);
	}
	return buffer.toString();
    }

    /**
         * Appends the hex value of a byte to a given StringBuffer.
         * 
         * @param buffer
         *                overall StringBuffer
         * @param b
         *                byte to transform to hex.
         */
    private static void appendHexStringFromByte(StringBuffer buffer, byte b) {
	buffer.append(Character.forDigit((b & 0xF0) >> 4, 16));
	buffer.append(Character.forDigit((b & 0x0F), 16));
    }
}
