/*******************************************************************************
 * Copyright (c) 2007-2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tie Li  - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.crpps.service;

import java.security.cert.X509Certificate;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Vector;

import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.IPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCard;
import org.eclipse.higgins.iss.CredentialContainer;
import org.eclipse.higgins.iss.UCTelm;
import org.eclipse.higgins.iss.UserChoiceTree;

public class UserInterfaceInfoBean {
	
	//Relying party info
	private String _name;
	private String _orgnization;
	private String _country;
	private String _verifier;
	
	//Cards info
	private IPolicy policy = null;
	private Vector match_cards = new Vector();  // type: InfoCard
	private Vector unmatch_cards = new Vector();
	
	public UserInterfaceInfoBean() {
		_name = _orgnization = _country = _verifier = "unknown";
	}
	
	public void setUCL(UserChoiceTree ucl) {
		this.policy = null;
		this.match_cards.clear();
		this.unmatch_cards.clear();
		
		parseUserChoiceTree(ucl);
	}
	
	public void addUnmatched_card(ICard card){
		boolean sign = false;
		for(int i=0; i<match_cards.size(); i++){
			ICard match_card = (ICard) match_cards.get(i);
			CUID match_card_cuid = match_card.getCUID();
			CUID curr_card_cuid = card.getCUID();
			if(match_card_cuid.equals(curr_card_cuid)){
				sign = true;
				break;
			}
		}
		
		if(!sign){
			this.unmatch_cards.add(card);
		}
	}
	
	public Vector getMatchCards(){
		return this.match_cards;
	}
	
	public Vector getUnmatchCards(){
		return this.unmatch_cards;
	}
	
	public IPolicy getPolicy(){
		return policy;
	}
			
	private void parseUserChoiceTree(UserChoiceTree ucl) {
		if(ucl==null) return;
				
		//there should be only one node for the "and" tree.		
		//there may be many or trees, each branch represents a I-Card provider.
		ListIterator orNodes = ucl.getListIterator();
		while(orNodes!=null && orNodes.hasNext()){
			Object obj = orNodes.next();
			if(obj instanceof UserChoiceTree){
				UserChoiceTree orTree = (UserChoiceTree)obj;
				ListIterator termNodes = orTree.getListIterator();
				while(termNodes!=null && termNodes.hasNext()){
					Object sub_obj = termNodes.next();
					if(sub_obj instanceof UCTelm){
						UCTelm ucTelm = (UCTelm) sub_obj;						
						policy = ucTelm.getAtom();						
						Iterator iter_cards = ucTelm.getCredsIterator();
						while(iter_cards!=null && iter_cards.hasNext()){
							CredentialContainer cc = (CredentialContainer) iter_cards.next();
							ICard icard = cc.getCredential();
							this.match_cards.add(icard);
						}
						
					}					
				}
			}			
		}
		
	}
	
	public void setCertificate(X509Certificate sslCert) {
		if (sslCert==null) { 
			return; 
		}
		
		parseX500Names(sslCert);		
	}
	
	private void parseX500Names(X509Certificate sslCert){
		String rp_x500name = sslCert.getSubjectX500Principal().getName();
		Hashtable t = getHashFromX509Name(rp_x500name);
		
		_name = clean(t.get("CN"));
		_orgnization = clean(t.get("O"));
		String st = clean(t.get("ST"));
		String co = clean(t.get("C"));
		_country = (st.length()>0 ? st + " " : "") + co;

		t = getHashFromX509Name(sslCert.getIssuerX500Principal().getName());		
		_verifier = clean(t.get("CN"));
		String vc = clean(t.get("C"));
		if (vc.length() > 0) _verifier +=  " in " + vc;
	}

	public String clean(Object o) { 
		if (o==null) { 
			return ""; 
		} else { 
			return o.toString(); 
		} 
	}
	
	// name looks like 1.2.840.113549.1.9.1=#1612616273407a75726963682e69626d2e636f6d,CN=9.4.21.94,OU=abhi test,O=IBM ZRL,L=Zurich,ST=Zurich,C=CH
	private Hashtable getHashFromX509Name(String name) {
		Hashtable h = new Hashtable();
		String s[]  = name.split(",");
		if (s!=null) {
			for(int j=0; j<s.length;j++) {
				String keyval[] = s[j].split("=");
				if (keyval!=null && keyval.length>1) {
					h.put(keyval[0],keyval[1]);
				}
			}
		}
		return h;
	}

	public String get_name() {
		return _name;
	}

	public void set_name(String _name) {
		this._name = _name;
	}

	public String get_orgnization() {
		return _orgnization;
	}

	public void set_orgnization(String _orgnization) {
		this._orgnization = _orgnization;
	}

	public String get_country() {
		return _country;
	}

	public void set_country(String _country) {
		this._country = _country;
	}

	public String get_verifier() {
		return _verifier;
	}

	public void set_verifier(String _verifier) {
		this._verifier = _verifier;
	}
}
