/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Tie Li  - Initial API and implementation
 *     Anthony bussani - change to show card properties
 *     Bruce Rich - change to allow multiple cardstores
 *******************************************************************************/

package org.eclipse.higgins.crpps.ui;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.higgins.crpps.Activator;
import org.eclipse.higgins.crpps.app.Application;
import org.eclipse.higgins.crpps.app.ApplicationWorkbenchWindowAdvisor;
import org.eclipse.higgins.crpps.ui.dialogs.ChangePasswordDialog;
import org.eclipse.higgins.crpps.ui.dialogs.ConfirmAbandonEdit;
import org.eclipse.higgins.crpps.ui.dialogs.ImportCardstoreDialog;
import org.eclipse.higgins.crpps.ui.dialogs.NewCardStoreDialog;
import org.eclipse.higgins.crpps.util.UUIDHelper;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.CardStoreException;
import org.eclipse.higgins.icard.CardStoreStrategy;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCard;
import org.eclipse.higgins.icard.provider.securestorage.CardStorePersonalCard;
import org.eclipse.higgins.icard.provider.securestorage.ProviderConfiguration;
import org.eclipse.higgins.icard.provider.securestorage.SecureStorageICardProvider;
import org.eclipse.higgins.icard.provider.securestorage.SecureStorageStrategy;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.registry.IConfiguration;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.part.ViewPart;

/**
 * Visualize a set of cards in a cardstore
 * @author litie at cn dot ibm dot com
 * 
 */
public class CardListView extends ViewPart implements ISelectionChangedListener, ISelectionProvider {

	private static Log log = LogFactory.getLog(CardListView.class);

	public static final String SECURE_STORAGE_ID = "org.eclipse.higgins.icard.provider.securestorage";
	public static final String ICARD = "ICARD";
	public static final String CARD_COMPOSITE = "CARD_COMPOSITE";
	public static final String CARD_LABEL = "CARD_LABEL";
	public static final String CARD_IMAGE = "CARD_IMAGE";
	public static final String SELECT_CARD_IMAGE = "SELECT_CARD_IMAGE";
	public static final String EXPORT_FORMAT_SUFFIX = ".crds";
	private static final String EMPTY_S = "";

	protected ICardRegistry registry = ICardRegistry.getInstance();

	// protected Composite cardComposite = null;

	List icards = new ArrayList(); // type: ICard
	List icardComposites = new ArrayList(); // type: Composite (Group)

	// private ICard currentSelectCard = null;

	final Point imageSize = new Point(100, 60);

	final Point selectImageSize = new Point(120, 80);

	final Point cardSize = new Point(110, 90);

	final Point selectCardSize = new Point(130, 110);

	private Composite mainComp = null;

	private Composite cardPanelComp = null;

	public static CardEditView observerPart = null;

	protected Composite currentWidget = null; // when just one cardimage selected, this is it
	
	protected HashMap currentWidgets = new HashMap(); // need collection of selected cards 

	static Image INFOCARD;

	public static Image MAUS;
	
	static Image PERSONALCARD;
	static byte[] personalCardBytes;
	
	private Map myImages = new HashMap(3);

	static CardListView instance = null;

	Shell shell;

	final Display display = Display.getCurrent();

	private String currentSecureStorage = null;

	public Font sysFont;

	public Font selectFont;

	final Color blue = Display.getCurrent().getSystemColor(SWT.COLOR_DARK_BLUE);
	final Color normalText = blue;
	final Color selectedImageBackground = blue;

	final Color white = Display.getCurrent().getSystemColor(SWT.COLOR_WHITE);
	final Color reverseVideoText = white;
	final Color normalImageBackground = white;

	Color black = Display.getCurrent().getSystemColor(SWT.COLOR_BLACK);
	/**
	 * Indicate if card images are two sizes (big and small) or not
	 */
	final boolean bigAndSmall = false;
	
    // ISelection info
	// when the current cardstore changes, provide a notification mechanism
	private ISelection currentSelection;	
	private List listeners = new ArrayList();
	
	private IPreferenceStore prefs = Activator.getDefault().getPreferenceStore();
	private int originalWidth;
	private float originalPercentage = prefs.getInt(Activator.LEFT_PANE_RATIO);

	public CardListView() {
		log.info("-> CardlistView()");

		// this._credDir = System.getProperty("work.dir") + "/test_user";
		org.apache.xml.security.Init.init();
		String name = null;
		if (INFOCARD == null) {
			name = "/image/infocard.png";
			INFOCARD = Utils.loadImage(name);
			myImages.put(name, INFOCARD);
		}					
		if (MAUS == null) {
			name = "/image/maus.gif";
			MAUS = Utils.loadImage(name);
			myImages.put(name, MAUS);
		}
		if (PERSONALCARD == null) {
			name = "/image/personalCard.jpg";
			PERSONALCARD = Utils.loadImage(name);
			myImages.put(name, PERSONALCARD);
			personalCardBytes = Utils.rawBytes(name);
		}
		selectFont = new Font(Display.getCurrent(), "Ariel", 8, SWT.BOLD);
		sysFont = new Font(Display.getCurrent(), "Ariel", 7, SWT.NONE);
		instance = this;
		
		//reset();
		getAllCardSpaceCards();
		log.info("<- CardlistView()");
	}

	public static CardListView getInstance() {
		return instance;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent) {
		this.originalWidth = parent.getBounds().width;
		//parent.setLayout(new FillLayout());
		FormToolkit toolkit = new FormToolkit(parent.getDisplay());
		Form form = toolkit.createForm(parent);		
		mainComp = form.getBody();
		mainComp.setLayout(new FormLayout());
	
		form.setText("Manage the i-cards in the current card store");
		Path path = new Path("/image/form_banner.gif");
		URL url = FileLocator.find(Platform.getBundle("org.eclipse.higgins.crpps"), 
			path, null);
		ImageDescriptor imageDesc = ImageDescriptor.createFromURL(url);
		form.setBackgroundImage(imageDesc.createImage());
						
		Composite controlComp = new Composite(mainComp, SWT.NONE);
		controlComp.setBackground(new Color(Display.getCurrent(), 255, 255, 255));
		FormData fd = new FormData();
		fd.top = new FormAttachment(0, 0);
		fd.left = new FormAttachment(0, 0);
		fd.right = new FormAttachment(100, 0);
		// fd.bottom = new FormAttachment(10, 0);
		controlComp.setLayoutData(fd);
		controlComp.setLayout(new RowLayout());

		Button createButton = new Button(controlComp, SWT.NONE);
		createButton.setText("New Card");
		createButton.setEnabled(true);
		createButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				log.info("create card");
				createNewCard();
			}
		});

		Button modifyButton = new Button(controlComp, SWT.NONE);
		modifyButton.setText("Modify Card");
		modifyButton.setEnabled(true);
		modifyButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				log.info("modify card");
				modifyCard();
//				resetCurrentWidgets();
			}
		});

		Button importButton = new Button(controlComp, SWT.NONE);
		importButton.setText("Import Card");
		importButton.setEnabled(true);
		importButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				log.info("importing card");
				importCard();
			}
		});

		Button deleteButton = new Button(controlComp, SWT.NONE);
		deleteButton.setText("Delete Card");
		deleteButton.setEnabled(true);
		deleteButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				log.info("delete card");
				deleteCard();
			}
		});

		Button exportButton = new Button(controlComp, SWT.NONE);
		exportButton.setText("Export Card");
		exportButton.setToolTipText("Export selected cards to a different cardstore");
		exportButton.setEnabled(true);
		exportButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				log.info("export card");
				exportCard();
			}
		});

		Label separator = new Label(mainComp, SWT.SEPARATOR | SWT.HORIZONTAL);
		fd = new FormData();
		fd.top = new FormAttachment(controlComp, 5);
		fd.left = new FormAttachment(0, 0);
		fd.right = new FormAttachment(100, 0);
		separator.setLayoutData(fd);

		// add scrolled panel here
		final ScrolledComposite scrolledPanelComp = new ScrolledComposite(mainComp, SWT.V_SCROLL);
//		final ScrolledComposite scrolledPanelComp = new ScrolledComposite(mainComp,
//				SWT.BORDER | SWT.V_SCROLL);
//		ScrolledComposite scrolledPanelComp = new ScrolledComposite(mainComp,
//				SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		fd = new FormData();
//		fd.top = new FormAttachment(controlComp, 0);
		fd.top = new FormAttachment(separator, 0);
		fd.left = new FormAttachment(0, 0);
		fd.right = new FormAttachment(100, 0);
		fd.bottom = new FormAttachment(100, 0);
		scrolledPanelComp.setLayoutData(fd);

		cardPanelComp = new Composite(scrolledPanelComp, SWT.NONE);
		cardPanelComp.setBackground(Display.getCurrent().getSystemColor(
				SWT.COLOR_WHITE));
		
		// add the composites for the cards
		updateCardComposites();
		
		RowLayout rl = new RowLayout(SWT.HORIZONTAL);
		rl.wrap = true;
		cardPanelComp.setLayout(rl);

		scrolledPanelComp.setContent(cardPanelComp);
		scrolledPanelComp.setExpandVertical(true);
		scrolledPanelComp.setExpandHorizontal(true);
		scrolledPanelComp.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent e) {
				Rectangle r = scrolledPanelComp.getClientArea();
				scrolledPanelComp.setMinSize(cardPanelComp.computeSize(r.width, SWT.DEFAULT));
			}
		});
//		int x = parent.computeSize(SWT.DEFAULT, SWT.DEFAULT).x;
//		int y = parent.computeSize(SWT.DEFAULT, SWT.DEFAULT).y;
//		scrolledPanelComp.setMinWidth(500);
//		scrolledPanelComp.setMinHeight(400);

//		RowLayout rowlo = new RowLayout();
//		rowlo.marginTop = 12;
//		rowlo.marginLeft = 12;
//		rowlo.spacing = 12;
//		cardPanelComp.setLayout(rowlo);
//		fd = new FormData();
//		fd.top = new FormAttachment(0, 5);
//		fd.left = new FormAttachment(0, 0);
//		fd.right = new FormAttachment(100, 0);
//		fd.bottom = new FormAttachment(100, -45);
//		cardPanelComp.setLayoutData(fd);


//		int x = 500;
//		int y = cardPanelComp.computeSize(500, SWT.DEFAULT).y;
//		
//		cardPanelComp.setSize(x, y);
		cardPanelComp.addMouseListener(new FormMouseListener());

		this.addDropTarget();
		
//		this.cardPanelComp.layout();
//		cardPanelComp.pack();
//		cardPanelComp.layout();
//		parent.getParent().layout(true, true);

		
		// set a listener to detect card changes
		//getSite().getPage().addSelectionListener(this);
		addSelectionChangedListener(this);
		
		// monitor the size of the parent to be able to adjust the layout next time around
		parent.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent ce) {
				Composite par = (Composite)ce.getSource();
				int newWidth = par.getBounds().width;
				int shellWidth = prefs.getInt(Activator.SHELL_WIDTH);
				float myShare = ((float)newWidth)/((float)shellWidth);
				float otherShare = 1.0f - myShare;
				prefs.setValue(Activator.LEFT_PANE_RATIO, myShare);
				prefs.setValue(Activator.RIGHT_PANE_RATIO, otherShare);
			}
		});
	}

	protected void createSecureStorage(String cardstoreName, String password) {
		CardStoreStrategy toStrategy = new SecureStorageStrategy();
		SecureStorageICardProvider toCardProvider = new SecureStorageICardProvider(toStrategy);
		PasswordCallback passwordCallback = new PasswordCallback("---",
				false);
		try {
			passwordCallback.setPassword(password.toCharArray());
			toStrategy.initialize(null, toCardProvider, cardstoreName,
					passwordCallback, null);
			((SecureStorageStrategy)toStrategy).newStorage();
		} catch (Exception e) {
			log.error(
					"Error when creating new storage card [" + cardstoreName + "]",
					e);
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
			messageBox.setMessage("Error when opening cardstore: " + cardstoreName + "\n\n"
					+ e);
			messageBox.setText("Error while creating");
			messageBox.open();
			return;
		}
	}

	public void deleteCard() {
		Shell shell = this.getViewSite().getShell();
		if (currentWidgets.size() < 1) {
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_NULL_ARGUMENT);
			messageBox.setMessage("You must select at least one card to delete");
			messageBox.setText("No card selected");
			messageBox.open();
			return;
		}
		log.info(currentWidgets.size() + " cards selected");
		ICard card = null;

		MessageBox confirm = new MessageBox(shell, SWT.ICON_WARNING | SWT.OK | SWT.CANCEL);
		confirm.setText("Card deletion confirmation");
		confirm.setMessage("Are you sure you want to delete the selected card(s)?");
		int resp = confirm.open();
		if (resp != SWT.OK)
			return;
		
		Iterator iter = currentWidgets.entrySet().iterator();
		while (iter.hasNext()) {
			// go through all the currently-selected cards
			//Composite currentWidget = (Composite)currentWidgets.get((String)iter.next());
			Composite currentWidget = (Composite)((Map.Entry)iter.next()).getValue();
			try {
				card = (ICard) currentWidget.getData(ICARD);
				Composite cardComposite = (Composite) currentWidget
				.getData(CARD_COMPOSITE);
				//if (card.getID() != null)
				// deal with the cardstore
				if (!"0".equals(card.getVersion()))
					PerspectiveCardStore.getSecureCardProvider().deleteCard(null, card);
				// deal with the card visualization
				disposeComposite(cardComposite);
				cardComposite = null;
				cardPanelComp.pack();
				// deal with the icard list
				int i = 0;
				int j = icards.size();
				for (; i<j; i++) {
					if (((ICard)icards.get(i)).getCUID().equals(card.getCUID())) {
						icards.remove(i);
						((Composite)icardComposites.get(i)).dispose();
						icardComposites.remove(i);
						break;
					}
				}
				if (i >= j) {
					log.warn("didn't find the card in the list");
				}
				this.currentWidget = null;
			} catch (Exception e) {
				log.error("Error when deleting card ["
						+ (card == null ? " ???" : card.getID()) + "]", e);
				MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
				messageBox.setMessage("Error when deleting card:" + card + "\n\n"
						+ e);
				messageBox.setText("Error while deleting");
				messageBox.open();
			}
		}
		currentWidgets = new HashMap();
		if (observerPart != null)
			setSelection(new CardSelection(CardSelection.TYPE_RESET)); // clear any residual information from the edit window
	}
	
	public void dispose() {
		for (Iterator iter = myImages.entrySet().iterator(); iter.hasNext();) {
			Map.Entry me = (Entry) iter.next();
			((Image)me.getValue()).dispose();
		}
	}
	
	/**
	 * Before disposing a composite, see if there's an image that needs to be disposed of
	 * @param composite
	 */
	public void disposeComposite(Composite composite) {
		log.info("disposeComposite");
		Image image = (Image) composite.getData(CARD_IMAGE);
		if ((image != null)  && !isMyImage(image))
			image.dispose();
		composite.dispose();
	}

	public void createNewCard() {
//		Shell shell = this.getViewSite().getShell();
		// CardStorePersonalCard personalCard = new CardStorePersonalCard
//		PersonalCardCreationDialog pccd = new PersonalCardCreationDialog(shell,
//				null);
//		pccd.setBlockOnOpen(true);
//		int ret = pccd.open();
//		ICard icard = pccd.getPersonalCard();
//		if (icard != null) {
//			log.info("icard:\n" + icard.getID());
//			try {
//				PerspectiveCardStore.getSecureCardProvider().addCard(null,
//						icard);
//			} catch (Exception e) {
//				log.error("Error when adding card  [" + icard.getName() + "]",
//						e);
//				MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
//				messageBox.setMessage("Error when adding card:"
//						+ icard.getName() + "\n\n" + e);
//				messageBox.setText("Error while adding a card");
//				messageBox.open();
//			}
//			insertNewCardToView(new ICard[] { icard });
//		}		
		
		//todo add check for dirty card
		CardEditView ev = (CardEditView)observerPart;
		int resp = IDialogConstants.OK_ID;
		if (ev.isCardDirty()) {	
			ConfirmAbandonEdit cae = new ConfirmAbandonEdit(shell);
			cae.setBlockOnOpen(true);
			resp = cae.open();
		}
		if (resp == IDialogConstants.OK_ID) {
			// Unselect everything
			resetCurrentWidgets(false);
			//setSelection(new CardSelection(CardSelection.TYPE_RESET)); // clear any residual information from the edit window
			// Pick an unused card name
			String cardNameS = pickNewCardName(icards, null);
			ICard icard = null;
			try {
				icard = new CardStorePersonalCard();
				((CardStorePersonalCard)icard).setClaimList(new ArrayList());
//			    InputStream personalCardImageIns = PersonalCardCreationDialog.class
//			    .getResourceAsStream("personalCard.jpg");
//			    int length;
//		    	length = personalCardImageIns.available();
//		    	byte[] personalCardImage = new byte[length];
//		    	personalCardImageIns.read(personalCardImage);
				byte[] personalCardImage = new byte[personalCardBytes.length];
				System.arraycopy(personalCardBytes, 0, personalCardImage, 0, personalCardBytes.length);
		    	icard.setImage(personalCardImage, "image/jpeg");
		    } catch (Exception e) {
		    	log.error("Error when loading default personal image", e);
				MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
//				messageBox.setMessage("Error when picking this name:"
//						+ icard.getName() + "\n\n" + e );
		        StringWriter sw = new StringWriter();
		        PrintWriter pw = new PrintWriter(sw, true);
		        e.printStackTrace(pw);
		        pw.flush();
		        sw.flush();
				messageBox.setMessage("Error when loading default personal image\n"+sw.toString());
				messageBox.setText("Error while adding a card");
				messageBox.open();
		    }
			try {
				icard.setName(cardNameS);
				((CardStorePersonalCard)icard).setVersion("0");
				((CardStorePersonalCard)icard).setIId(new URI("urn:uuid:" + UUIDHelper.randomUUID()));
				insertNewCardToView(new ICard[] { icard });
				CardSelection cs = new CardSelection(CardSelection.TYPE_INFORM);
				cs.setCard(icard);
				setSelection(cs);
//				ev.notifySelectionChanged(icard);
				// add the card to the list that CardListView knows about
				// we expect to be notified when the edit is complete (see selectionChanged)
				// we expect the CUID to still be the same in the modified card,
				// which is how we make the connection
				this.icards.add(icard);
				// simulate user double-clicked on a card, go to edit
				ev.modify();
				//resetCurrentWidgets(true);
				//currentWidget.setData(ICARD, icard);
			} catch (Exception e) {
				log.error("Error when adding card  [" + icard.getName() + "]",
						e);
				MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
		        StringWriter sw = new StringWriter();
		        PrintWriter pw = new PrintWriter(sw, true);
		        e.printStackTrace(pw);
		        pw.flush();
		        sw.flush();
				messageBox.setMessage("Error when picking this name:"
				+ icard.getName() + "\n\n" + sw.toString());
				messageBox.setText("Error while adding a card");
				messageBox.open();
			}
		}
	}

	public void modifyCard() {
		Shell shell = this.getViewSite().getShell();
		if (currentWidget == null) {
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
			messageBox.setMessage("You must select a card to modify");
			messageBox.setText("No card selected");
			messageBox.open();
			return;
		}

		//user already selected the card...observerPart.notifySelectionChanged(card);
		observerPart.modify();
		resetCurrentWidgets(false);

	}

	public void exportCard() {
		Shell shell = this.getViewSite().getShell();// ApplicationWorkbenchWindowAdvisor.getShell();
		if (currentWidgets.size() < 1) {
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_NULL_ARGUMENT);
			messageBox.setMessage("You must select at least one card to export");
			messageBox.setText("No card selected");
			messageBox.open();
			return;
		}
		log.info(currentWidgets.size() + " cards selected");
		String filename = "";
		
		NewCardStoreDialog pccd = new NewCardStoreDialog(shell, filename);
		pccd.setBlockOnOpen(true);
		int ret = pccd.open();
		if (ret != 0)
			return;
		String password = pccd.getPasswd();
		filename = pccd.getFilename();
		if (filename != null) {
			if (!filename.toLowerCase().endsWith(EXPORT_FORMAT_SUFFIX))
				filename += EXPORT_FORMAT_SUFFIX;
		}
		exportCard(filename, password, false);
	}
	
	private void exportCard(String cardstoreName, String password, boolean move) {
		SecureStorageICardProvider fromCardProvider = PerspectiveCardStore.getSecureCardProvider();
		
		CardStoreStrategy toStrategy = new SecureStorageStrategy();
		SecureStorageICardProvider toCardProvider = new SecureStorageICardProvider(toStrategy);
		PasswordCallback passwordCallback = new PasswordCallback("---",
				false);
		try {
			passwordCallback.setPassword(password.toCharArray());
			toStrategy.initialize(null, toCardProvider, cardstoreName,
					passwordCallback, null);
			if (new File(cardstoreName).exists())
				toCardProvider.synchFromStorage(null); // read the current contents (if any)
			else
				((SecureStorageStrategy)toStrategy).newStorage();
		} catch (Exception e) {
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
			messageBox.setMessage("Error when opening cardstore: " + cardstoreName + "\n\n"
					+ e);
			messageBox.setText("Error while exporting");
			messageBox.open();
			return;
		}
		ICard card = null;

		Iterator iter = currentWidgets.entrySet().iterator();
		while (iter.hasNext()) {
			// go through all the currently-selected cards
			Composite currentWidget = (Composite)((Map.Entry)iter.next()).getValue();
			card = (ICard) currentWidget.getData(ICARD);
			Composite cardComposite = (Composite)currentWidget.getData(CARD_COMPOSITE);
			if (!"0".equals(card.getVersion())) {
				// if version=0, we presume it's not really in the cardstore
				try {
					// first copy the card to the target cardstore
					toCardProvider.addCard(null, card);
				} catch (Exception e) {
					log.error("Error when adding card ["
							+ (card == null ? " ???" : card.getID()) + "]", e);
					MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
					messageBox.setMessage("Error when adding card:" + card + "\n\n"
							+ e);
					messageBox.setText("Error while exporting");
					messageBox.open();
					return;
				}
				try {
					// after successful copy, then delete the card from the source
					if (move)
						fromCardProvider.deleteCard(null, card);
				} catch (Exception e) {
					log.error("Error when deleting card ["
							+ (card == null ? " ???" : card.getID()) + "]", e);
					MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
					messageBox.setMessage("Error when deleting card:" + card + "\n\n"
							+ e);
					messageBox.setText("Error while exporting");
					messageBox.open();
				}
			}
//				unselectCard(cardnameS, cardComposite);
			if (move) {
				cardComposite.dispose();
				cardComposite = null;
				currentWidget.dispose();
				cardPanelComp.pack();
			} else
				resetCurrentWidgets(true);
		}
		
		try {
			toCardProvider.synchToStorage(null); // make sure actually written out
		} catch (CardStoreException e) {
			log.error("Error when adding card ["
					+ (card == null ? " ???" : card.getID()) + "]", e);
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
			messageBox.setMessage("Error when adding card:" + card + "\n\n"
					+ e);
			messageBox.setText("Error while exporting");
			messageBox.open();
		}
		
		this.currentWidget = null;
		currentWidgets = new HashMap();
		setSelection(new CardSelection(CardSelection.TYPE_RESET)); // clear any residual information from the edit window
	}

	public void importCard() {
		Shell shell = this.getViewSite().getShell();// ApplicationWorkbenchWindowAdvisor.getShell();
		FileDialog fileDialog = new FileDialog(shell, SWT.OPEN | SWT.MULTI);
		fileDialog.setFilterExtensions(new String[] { "*.crd", "*" });
		String ret = fileDialog.open();
		if (ret == null)
			return;
		String[] files1 = fileDialog.getFileNames();
		String files[] = new String[files1.length];
		for (int i = 0; i < files1.length; i++) {
			if (fileDialog.getFilterPath().endsWith(File.separator))
				files[i] = fileDialog.getFilterPath() + files1[i];
			else
				files[i] = fileDialog.getFilterPath() + File.separator
						+ files1[i];
		}
		importCards(files);
	}

	public void importCards(String[] files) {
		String filename;
		for (int i = 0; i < files.length; i++) {
			filename = files[i];
			if (filename.toLowerCase().endsWith(".crd")) {
				log.info("insertNewCardToView(" + filename + ")");
				ICard icard = null;

				try {
					icard = PerspectiveCardStore.getSecureCardProvider()
							.importCard(null, filename);
				} catch (Exception e) {
					log.error("Error when import file [" + filename + "]", e);
					MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
					messageBox.setMessage("Error when importing File:"
							+ filename + "\n\n" + e);
					messageBox.setText("Error while importing");
					messageBox.open();
				}
				if (icard != null) {
					log.info("icard:\n" + icard.getID());
					icards.add(icard);
					insertNewCardToView(new ICard[] { icard });
				}
			} else if (filename.toLowerCase().endsWith(".crds")) {
				importCardStore(filename);
			}
		}
	}

	public void importCardStore(String filename) {
		ImportCardstoreDialog dialog = new ImportCardstoreDialog(shell,
				filename);
		int rc = dialog.open();
		List cardList = dialog.getCardsList();
		Iterator iter = cardList.iterator();
		try {
			while (iter.hasNext()) {
				// import selected cards
				ICard card = (ICard) iter.next();
				// copy to current cardstore
				PerspectiveCardStore.getSecureCardProvider().addCard(null, card);
				// add to the list of infocards "from" current store
				icards.add(card);
				// create a virtualization for the card and add the virtualization to the view of the current cardstore
				insertNewCardToView(new ICard[]{card} );
			}
		} catch (Exception e) {
			log.error("Error when importing file [" + filename + "]", e);
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
			messageBox.setMessage("Error when importing cards from File:"
					+ filename + "\n\n" + e);
			messageBox.setText("Error while importing");
			messageBox.open();
		}
	}
	
	/**
	 * Reset state dealing with particular cardstores
	 */
	private void reset() {
		icards.clear();
		freshWidgets();
		getAllCardSpaceCards();
		if (this.cardPanelComp != null)
			this.cardPanelComp.layout();
	}
	
	/**
	 * Go through the set of currently-selected cards and dispose of them,
	 * then prepare for a new group
	 */
	private void freshWidgets() {
		if (currentWidgets != null) {
			Iterator iter = currentWidgets.entrySet().iterator();
			while (iter.hasNext()) {
				Composite cardComposite = (Composite) ((Map.Entry)iter.next()).getValue();
				if (cardComposite != null)
					disposeComposite(cardComposite);
			}
		}
		currentWidgets = new HashMap(); // we allocate a new one to allow space reclamation
		this.currentWidget = null;
	}

	/**
	 * @param card
	 */
	private Composite createCardComposite(ICard card) {
		Group cardComp = new Group(cardPanelComp, SWT.SHADOW_ETCHED_OUT);
		cardComp.setBackground(Display.getCurrent().getSystemColor(
				SWT.COLOR_WHITE));
		RowData rd = new RowData();
		rd.width = cardSize.x;
		rd.height = cardSize.y;
		cardComp.setLayoutData(rd);
		cardComp.setLayout(new RowLayout(SWT.VERTICAL));
		
		cardComp.setText(card.getType());
		if(card instanceof PersonalCard){
			int pinStatus = ((PersonalCard)card).getPinStatus();
			if(PersonalCard.LOCKED == pinStatus){
				cardComp.setText("LOCKED");
			}				
		}

		Composite imageComp = buildImages(card, cardComp);

		String name = card.getName();
		Label label = new Label(cardComp, SWT.NONE | SWT.WRAP);
		label.setText(name);
		label.setBackground(white);
		label.setForeground(blue);

		cardComp.setToolTipText(name);

		CardMouseListener cml = new CardMouseListener(imageComp);
		imageComp.addMouseListener(cml);
		imageComp.setData(ICARD, card);
		imageComp.setData(CARD_COMPOSITE, cardComp);
		imageComp.setData(CARD_LABEL, label);
		return cardComp;
	}

	private Composite buildImages(ICard card, Composite parent) {
		RowData rd;
		Composite imageComp = new Composite(parent, SWT.NONE);

		String imageType = card.getImageType();
		// build image from card
		Image image = buildUnselectedImage(card, imageType);
		// adjust scale to what we want
		ImageData imd = image.getImageData().scaledTo(imageSize.x, imageSize.y);
		// re-gen the image to go into the composite
		Image newimg = new Image(Display.getCurrent(), imd);
		//  set it as background...will this get cleaned up on dispose?  doubt it...
		imageComp.setBackgroundImage(newimg);
		rd = new RowData();
		rd.width = imageSize.x;
		rd.height = imageSize.y;
		imageComp.setLayoutData(rd);
		imageComp.setData(CARD_IMAGE, newimg);
		if (bigAndSmall) {
			// adjust image for "selected" mode
			imd = image.getImageData().scaledTo(selectImageSize.x,
					selectImageSize.y);
			newimg = new Image(Display.getCurrent(), imd);
			imageComp.setData(SELECT_CARD_IMAGE, newimg);
		}
		if (!isMyImage(image))
			image.dispose();  // otherwise image leak
		return imageComp;
	}
	
	private Image buildUnselectedImage(ICard card, String imageType) {
		byte[] img_rawdata = card.getImage();
		log.info("buildUnselectedImage...imagesize = "+img_rawdata.length);
		Image image = null;
		if (img_rawdata == null) {
			log.warn("Error when loading ICard image of type [" + imageType
					+ "] for Icard [" + card.getID() + "]");
			image = CardListView.INFOCARD;
		}
		if (image == null) {
			ByteArrayInputStream bis = new ByteArrayInputStream(img_rawdata);
			try {
				image = new Image(Display.getCurrent(), bis);
			} catch(org.eclipse.swt.SWTException e) {
				// in case of failure try to decode the stream as base64 and reconstruct the image
				log.warn("Could not construct the image, trying again after base64 decoding");
				try {
					img_rawdata = Base64.decodeBase64(img_rawdata);
					bis = new ByteArrayInputStream(img_rawdata);
					image = new Image(Display.getCurrent(), bis);
				} catch(org.eclipse.swt.SWTException e1) {
					log.warn("Could not construct the image using base64 decoding: we keep the default image");
				}
			}
		}
		if (image != null) {
			Image scaledImage = new Image(Display.getCurrent(), image.getImageData().scaledTo(selectImageSize.x,
				selectImageSize.y));
			if (!isMyImage(image))
				image.dispose();
			image = scaledImage;
		}
		return image;
	}
	
	private void resetCardComposite(ICard newcard) {
//		// find the composite that matches the changed card
		if (newcard == null) {
			log.info("resetCardComposite called with null card");
			return;
		}
		Composite currentWidget = null;
		Control childrens[] = cardPanelComp.getChildren();
		for (int i = 0; i < icards.size(); i++) {
			ICard card = (ICard) icards.get(i);
			if (card.getID().equals(newcard.getID())) {
				if (i < childrens.length)
					currentWidget = (Composite) childrens[i];
				// ok, we found the appropriate widget, better check that the card we're dealing with
				//    doesn't have a version that indicates someone canceled out of Edit
				// if it is version 0, we should not just reset the card visualization, but get rid of it
				if ("0".equals(newcard.getVersion())) {
					currentWidget.dispose();
					cardPanelComp.pack();
					cardPanelComp.layout();
					return;
				}
				// bad karma...image leak at best
				icards.set(i, newcard);
				break;
			}
		}
		if (currentWidget == null) {
			log.warn("Did not find a match for the card to be reset");
			return;
		}
		
		// ok, now we need to update in place so as to not increase the number of cardPanelComp children
		
		// select the new graphics card
		Group cardComposite = (Group) currentWidget; //.getData(CARD_COMPOSITE);
		if(newcard instanceof PersonalCard){
			PersonalCard pcard = (PersonalCard)newcard;
			if(pcard.getPinStatus()==PersonalCard.LOCKED){
				cardComposite.setText("LOCKED");
			}
		}
		
		// see if we already have an image composite
		Control[] chilluns = cardComposite.getChildren();
		Composite imageComp = null;
		boolean needImage = false;
		if ((chilluns == null) || (chilluns.length < 1)) {
			imageComp = new Composite(cardComposite, SWT.NONE);
			needImage = true;
		} else
			imageComp = (Composite)chilluns[0];
			
		cardComposite.setBackground(Display.getCurrent().getSystemColor(
				SWT.COLOR_WHITE));
//		cardComposite.setForeground(white);
		
		String type = newcard.getType();
		cardComposite.setText(type);
		
		String name = newcard.getName();
		Label cardLabel = (Label) imageComp.getData(CARD_LABEL);
		if (cardLabel == null)
			cardLabel = new Label(imageComp, SWT.NONE);
		cardLabel.setBackground(white);
		cardLabel.setForeground(blue);
//		cardLabel.setFont(sysFont);
		cardLabel.setText(name);
		cardLabel.pack();
		
		cardComposite.setToolTipText(name);

		imageComp.setData(ICARD, newcard);

		// clean up the previous images (background is same as CARD-IMAGE)
		Image image = (Image) imageComp.getData(SELECT_CARD_IMAGE);
		if ((image != null) && !isMyImage(image))
			image.dispose();  // otherwise image leak
		image = imageComp.getBackgroundImage();
		if ((image != null) && !isMyImage(image))
			image.dispose();  // otherwise image leak

		image = buildUnselectedImage(newcard, newcard.getImageType());

		log.info("image=" + image);

		imageComp.setData(CARD_COMPOSITE, cardComposite);
		imageComp.setData(CARD_IMAGE, image);
		imageComp.setBackgroundImage(image);

		if (bigAndSmall) {
			RowData rd = new RowData();
			rd.width = cardSize.x;
			rd.height = cardSize.y;
			currentWidget.setLayoutData(rd);

			imageComp.setData(SELECT_CARD_IMAGE, new Image(Display.getCurrent(), image.getImageData().scaledTo(selectImageSize.x,
					selectImageSize.y)));
		}
//		cardComposite.setBackground(blue);

//		cardLabel.setBackground(blue);
//		cardLabel.setForeground(white);
//		cardLabel.setFont(selectFont);
		
		// and we're gambling that none of our layout sizes changed in the interim
		
		cardComposite.getParent().pack();
		this.cardPanelComp.layout(); //.redraw();
	}

	/**
	 * 
	 */
	private void updateCardComposites() {
		for (int i = 0; i < icards.size(); i++) {
			ICard card = (ICard) icards.get(i);
			icardComposites.add(createCardComposite(card));
		}
		shell = this.getViewSite().getShell();
	}

	/**
	 * 
	 */
	public void changePwdRepository() {
		ChangePasswordDialog chgPasswordDialog = new ChangePasswordDialog(
				shell, currentSecureStorage);
		int ret = chgPasswordDialog.open();
		if (ret != 0)
			return;
		String oldPasswd = chgPasswordDialog.getOldPassword();
		String newPasswd = chgPasswordDialog.getNewPassword();
		SecureStorageICardProvider cardProvider = PerspectiveCardStore
				.getSecureCardProvider();
		SecureStorageStrategy secureStorageStrategy = (SecureStorageStrategy) cardProvider
				.getStrategy();
		try {
			secureStorageStrategy.changePassword(oldPasswd.toCharArray(),
					newPasswd.toCharArray());
		} catch (CardStoreException e) {
			log.error("Error when changing password", e);
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
			messageBox.setMessage("Error:\n" + e.getMessage()
					+ "\nwhen changing password");
			messageBox.setText("Error when changing password");
			messageBox.open();
		}
	}

	/**
	 * @param filename
	 */
	public void changeRepository(String filename) {
		String password = null;
		if (filename == null) 
			filename = prefs.getString(Activator.MOST_RECENT_CARDSTORE);
		if ("".equals(filename)) {
			filename = PerspectiveCardStore.getSecureCardProvider().getFilename();
		}
		// NCSD understands that if a file doesn't exist, the confirm pw needs to
		//   match pw
		NewCardStoreDialog pccd = new NewCardStoreDialog(shell, filename);
		pccd.setBlockOnOpen(true);
		int ret = pccd.open();
		password = pccd.getPasswd();
		if (ret != 0)
			return;
		filename = pccd.getFilename();
		if (filename == null || ("".equals(filename))) {
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
			messageBox.setMessage("A cardstore name was NOT supplied");
			messageBox.setText("Error when opening a cardstore");
			messageBox.open();
			return;
		}
		File file = new File(filename);
		if (file.exists() && !file.canRead()) {
			MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
			messageBox.setMessage("The store [" + filename
					+ "] is not accessible\nand/or readable");
			messageBox.setText("Error when opening a cardstore");
			messageBox.open();
			return;
		}
		log.trace("passwd=" + password);
		log.trace("filename=" + filename);
		if (file.exists()) {
			openStorage(filename, password);
		} else {
			createSecureStorage(filename, password);
			openStorage(filename, password);
		}
	}

	/**
	 * Open a new "current cardstore" and let others know this happened.
	 * This code is similar to that in PerspectiveCardStore.
	 * @param filename
	 * @param password
	 */
	private void openStorage(String filename, String password) {
		SecureStorageStrategy oldSss = null;
		SecureStorageICardProvider cardProvider = null;
		IConfiguration conf = null;
		boolean noTurningBack = false;
		try {
			PasswordCallback passwordCallback = new PasswordCallback("---",
					false);
			passwordCallback.setPassword(password.toCharArray());
			// pick out current provider
			cardProvider = PerspectiveCardStore.getSecureCardProvider();
			// get the config info
			conf = cardProvider.getConfiguration();
			oldSss = (SecureStorageStrategy) cardProvider.getStrategy();
			SecureStorageStrategy sss = new SecureStorageStrategy();
			sss.initialize(null, cardProvider, filename, passwordCallback, null);
			PerspectiveCardStore.reInitCurrentSecureCardProvider(sss);
			// reset our view of things
//			noTurningBack = true;
//			reset();
			clearListCard();
			setSelection(new CardSelection(CardSelection.TYPE_RESET)); // clear any residual information from the edit window
//			if (observerPart != null)
//				observerPart.reset(); // clear any residual information from the edit window
			// go get info from new store
			getAllCardSpaceCards();
    		// change the configuration info for the card provider
    		((ProviderConfiguration)conf).setPassword(password);
    		((ProviderConfiguration)conf).setCardStoreFile(filename);
    		noTurningBack = true;
			updateCardComposites();
//			cardPanelComp.pack();
			cardPanelComp.layout();
			ApplicationWorkbenchWindowAdvisor.getInstance().setTitle(
					"[" + filename + "]");
			currentSecureStorage = filename;
			Application.savedLastRepository = filename;
			if (!Application.savedHistoryRepository.contains(filename))
				Application.savedHistoryRepository.add(filename);
			prefs.setDefault(Activator.MOST_RECENT_CARDSTORE, filename);
		} catch (Exception e) {
			if (noTurningBack) {				
			} else {
				if (oldSss != null) {
					// try to put things back the way they were
					PerspectiveCardStore.reInitCurrentSecureCardProvider(oldSss);
				}
			}
			openSecureStore(shell, e, filename);
		}
	}

	/**
	 * 
	 */
	public String newRepository() {
		Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getShell();
		SecureStorageICardProvider cardProvider = PerspectiveCardStore
				.getSecureCardProvider();
		String filename = cardProvider.getFilename();
		NewCardStoreDialog pccd = new NewCardStoreDialog(shell, filename); //, true);
		pccd.setBlockOnOpen(true);
		int ret = pccd.open();
		if (ret == 0) {
			filename = pccd.getFilename();
			log.trace("passwd=" + pccd.getPasswd());
			log.trace("filename=" + filename);
			createSecureStorage(pccd.getFilename(), pccd.getPasswd());
		}
		return filename;
	}

	/**
	 * Reset our view of current widgets, clear out the icard list,
	 * and dispose of the current visualized card images
	 */
	private void clearListCard() {
		currentWidget = null;
		currentWidgets = new HashMap(); // forget about previous widgets
		icards.clear();
		Iterator iter = icardComposites.iterator();
		while (iter.hasNext())
			((Composite)iter.next()).dispose();
		icardComposites.clear();
	}

	/**
	 * @author bus
	 * @author Bruce Rich (IBM Corporation) - Add double-click to edit, preserve dirty data, select multiple cards
	 * 
	 */
	class CardMouseListener implements MouseListener {
		
		private Composite widget;

		public CardMouseListener(Composite widgetP) {
			widget = widgetP;
		}

		public void mouseDoubleClick(MouseEvent arg0) {
			// we expect that the mouseDown event already checked for dirty edit in progress
			// we also expect the mouseDown event would select the item
			log.info("mouseDoubleClick()");
			CardEditView ev = (CardEditView)observerPart;
			// user double-clicked on a card, go to edit
			ev.modify();
		}

		public void mouseDown(MouseEvent mouseEvent) {
			log.info("mouseDown(button = "+mouseEvent.button+" stateMask = "+mouseEvent.stateMask+" )");
			/*if (mouseEvent.count > 1)
				return;*/
			//ICard icard = selectItem();
			CardEditView ev = (CardEditView)observerPart; 
			int resp = IDialogConstants.OK_ID;
			if (ev.isCardDirty()) {
				ConfirmAbandonEdit cae = new ConfirmAbandonEdit(shell);
				cae.setBlockOnOpen(true);
				resp = cae.open();
				if (resp == IDialogConstants.OK_ID) {
//					ev.reset();
					setSelection(new CardSelection(CardSelection.TYPE_RESET)); // clear any residual information from the edit window
				}
				widget.getParent().redraw();
			}
			if (resp == IDialogConstants.OK_ID) {
				// any edit session is now over
				//icard = (ICard)widget.getData(ICARD);
				ICard icard = null;
				CardSelection cs = null;
				switch (mouseEvent.stateMask) {
				case SWT.SHIFT :
					if (currentWidgets.size() > 0) {
						// handle shift-click to select all between
						shiftClick();
						break;
					} // else fall into control selection
				case SWT.CONTROL :
					// add this card to the selected collection
					icard = selectItem(false);
					//currentWidgets.put(icard.getCUID(), widget);
					cs = new CardSelection(CardSelection.TYPE_MULTISELECT);
					cs.setCard(icard);
					setSelection(cs);
					observerPart.browselist();
					break;
				default :
					// not a collection-style selection
					resetCurrentWidgets(true);
					//currentWidgets.put(icard.getName(), widget);
					icard = selectItem(true);
					cs = new CardSelection(CardSelection.TYPE_INFORM);
					cs.setCard(icard);
					setSelection(cs);
					ev.browse();
				}
			}
		}

		public void mouseUp(MouseEvent arg0) {
			log.info("mouseUp()");
		}

		private ICard selectItem(boolean justOne) {
			log.info("selectItem()");

			ICard currentSelectCard = (ICard) widget.getData(ICARD);
			// currentSelectCard = (ICard) icards.get(index);

			// un-select old graphics card
			if (justOne && currentWidget != null) {
				unselectCard(widget, true, false);
//				cardComposite = (Composite) currentWidget
//						.getData(CARD_COMPOSITE);
//				cardComposite.setBackground(blue);
//				cardComposite.setForeground(black);
//				RowData rd = new RowData();
//				rd.width = imageSize.x;
//				rd.height = imageSize.y;
//				currentWidget.setLayoutData(rd);
//				cardComposite.setBackground(white);
//				cardLabel = (Label) currentWidget.getData(CARD_LABEL);
//				cardLabel.setBackground(white);
//				cardLabel.setForeground(blue);
//				cardLabel.setFont(sysFont);
//				cardComposite.setSize(cardSize);
//				Image image = (Image) currentWidget.getData(CARD_IMAGE);
//				log.info("select image=" + image);
//				currentWidget.setBackgroundImage(image);
//				currentWidget.setSize(imageSize);
//				rd = new RowData();
//				rd.width = cardSize.x;
//				rd.height = cardSize.y;
//				cardComposite.setLayoutData(rd);
			}
			
			selectCard(currentSelectCard, widget, true);
//			selected = true;
//
//			// select the new graphics card
//			cardComposite = (Composite) widget.getData(CARD_COMPOSITE);
//			cardComposite.setForeground(white);
//			RowData rd = new RowData();
//			rd.width = selectImageSize.x;
//			rd.height = selectImageSize.y;
//			widget.setLayoutData(rd);
//			cardComposite.setBackground(blue);
//			cardLabel = (Label) widget.getData(CARD_LABEL);
//			cardLabel.setBackground(blue);
//			cardLabel.setForeground(white);
//			cardLabel.setFont(selectFont);
//			Image image = (Image) widget.getData(SELECT_CARD_IMAGE);
//			log.info("image=" + image);
//			widget.setBackgroundImage(image);
//			widget.setSize(selectImageSize);
//			cardComposite.setSize(selectCardSize);
//			currentWidget = widget;
//			rd = new RowData();
//			rd.width = selectCardSize.x;
//			rd.height = selectCardSize.y;
//			cardComposite.setLayoutData(rd);
//
//			cardComposite.getParent().pack();
//
			return currentSelectCard;
		}
		
		/**
		 * Select everything between last-selected and just-selected
		 */
		private void shiftClick() {
			log.trace("shiftClick");
			ICard justSelectedCard = (ICard) widget.getData(ICARD);
			ICard lastSelectedCard = null;
			List cards = new ArrayList();
			if (currentWidget != null) {
				lastSelectedCard = (ICard) currentWidget.getData(ICARD);
			}
			if (lastSelectedCard == null) {
				// just have one card selected
				selectCard(justSelectedCard, widget, true);
			} else {
				// find the range of cards to be considered selected
				List lookFor = new ArrayList(2);
				lookFor.add(justSelectedCard);
				lookFor.add(lastSelectedCard);
				int first = -1, last = -1, justSelectedIndex = -1;
				boolean foundOne = false;
				outer: for (int i=0; i<icards.size(); i++) {
					for (int j=0; j<lookFor.size(); j++) {
						if (icards.get(i) == lookFor.get(j)) {
							// found one
							if (icards.get(i) == justSelectedCard)
								justSelectedIndex = i; // remember where the just-selected one is
							if (foundOne) {
								last = i;
								break outer; // found both, so done
							} else {
								foundOne = true;
								first = i;
								lookFor.remove(j); // no longer need to look for this one
								continue outer;
							}					
						}
					}
				}
				log.info("first = "+first+", last = "+last);
				// ok, should have found the range
				// unselect everything currently selected (sigh)
				resetCurrentWidgets(true);
				// select everything from first to last
				// remember which one was just selected so we can reset currentWidget when we get done
				// (we could run the loop backwards, but then the listsummary would be backwards too
				Composite rememberCurrent = null;
				for (int i=first; i<=last; i++) {
					// gotta find the child of the Group composite
					Composite firstLevel = (Composite)icardComposites.get(i);
					Control[] chilluns = firstLevel.getChildren();
					ICard thisIcard = (ICard)icards.get(i);
					selectCard(thisIcard, (Composite)chilluns[0], false);
					if (i == justSelectedIndex)
						rememberCurrent = (Composite)chilluns[0];
					cards.add(thisIcard);
				}
				// make sure currentWidget indicates the "just-selected"
				currentWidget = rememberCurrent;
			}
			CardSelection cs = new CardSelection(CardSelection.TYPE_MULTISELECT);
			cs.setCardList(cards);
			setSelection(cs);
			observerPart.browselist();
		}
	}
	
	// ISelectionProvider APIs

	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		if(!listeners.contains(listener))
			listeners.add(listener);
	}

	public ISelection getSelection() {
		return this.currentSelection;
	}

	public void removeSelectionChangedListener(
			ISelectionChangedListener listener) {
		if(listeners.contains(listener))
			listeners.remove(listener);
	}

	public void setSelection(ISelection selection) {
		// could not disable the public API so that only locals can trigger selections,
		//	as editor may wish to indicate a card was changed
		if (!(selection instanceof CardSelection)) {
			throw new IllegalArgumentException("Need CardSelection");
		}
		this.currentSelection = selection;
		fireSelectionChange();
	}
	
	private void fireSelectionChange(){
		for(int i = 0; i < listeners.size(); i++){
			ISelectionChangedListener listener = (ISelectionChangedListener)listeners.get(i);
			SelectionChangedEvent event = new SelectionChangedEvent(this, 
					this.currentSelection);
			listener.selectionChanged(event);
		}
	}
	
	// ISelectionChangedListener API
	
	public void selectionChanged(SelectionChangedEvent event) {
		if (event.getSelection() instanceof CardSelection) {
			CardSelection cs = (CardSelection)event.getSelection();
			switch (cs.getSelType()) {
			case CardSelection.TYPE_MODIFIED :
				System.out.println("iid = "+((CardSelection)event.getSelection()).getCard().getID());
				ICard card = ((CardSelection)event.getSelection()).getCard();
				resetCardComposite(card);
				resetCurrentWidgets(false);
				break;
			case CardSelection.TYPE_MULTISELECT :
				log.info("Ignoring multiselect");
				break;
			case CardSelection.TYPE_RESET :
				ICard icard = cs.getCard();
				if (icard == null)
					log.info("Ignoring general reset");
				else {
					log.info("Card-specific reset");
					resetCardComposite(icard);
				}
				break;
			case CardSelection.TYPE_INFORM :
				log.info("Ignoring inform");
				break;
			}
		}
	}

	
	// image management
	public boolean isMyImage(Image image) {
		boolean answer = false;
		for (Iterator iter = myImages.entrySet().iterator(); iter.hasNext();) {
			Map.Entry me = (Entry) iter.next();
			if (image == me.getValue()) {
				answer = true;
				break;
			}
		}
		return answer;
	}

	
	/**
	 * Indicate no cards are selected
	 * @author brich
	 *
	 */
	class FormMouseListener implements MouseListener {

		public void mouseDoubleClick(MouseEvent e) {
		}

		public void mouseDown(MouseEvent e) {
			log.info("FormMouseListener.mouseDown");
			// reset all "selected" cards
			resetCurrentWidgets(true); // which will cause the reset to the clv
//			// reset the edit window
//			observerPart.reset();
			cardPanelComp.layout(); //.redraw();
		}

		public void mouseUp(MouseEvent e) {
		}
		
	}

	public void setFocus() {
		log.info("setFocus()");

	}

	/**
	 * 
	 */
	private void getAllCardSpaceCards() {
		CallbackHandler handler = new CallbackHandler() {
			public void handle(Callback[] callbacks) throws IOException,
					UnsupportedCallbackException {
				for (int i = 0; i < callbacks.length; i++) {
					if (callbacks[i] instanceof NameCallback) {
						NameCallback n = (NameCallback) callbacks[i];
						n.setName("test_user");
					} else if (callbacks[i] instanceof PasswordCallback) {
						PasswordCallback p = (PasswordCallback) callbacks[i];
						p.setPassword("password".toCharArray());
					} else {
						throw new UnsupportedCallbackException(callbacks[i]);
					}
				}
			}
		};

		for (Iterator itr = registry.getICardProviders(); itr.hasNext();) {
			ICardProvider p = (ICardProvider) itr.next();
			log.trace("cardProvider[" + p.getID() + "]=[" + p.getName() + "|"
					+ p.getDescription() + "]");
			Class[] supportedTypes = p.getSupportedTypes();
			for (int i = 0; i < supportedTypes.length; i++) {
				Class supportedType = supportedTypes[i];

				try {
					for (Iterator cards = p.getICards(handler); cards
							.hasNext();) {
						ICard card = (ICard) cards.next();
						this.icards.add(card);
					}
				} catch (Exception e) {
					log.error("Error when reading the card provider"+p.getDescription(), e);
					if (shell != null) {
						MessageBox messageBox = new MessageBox(shell,
								SWT.ERROR_IO);
						messageBox.setMessage("Error:\n" + e.getMessage()
								+ "\nlook at logs for details.");
						messageBox.setText("Error when reading the card provider:\n"+p.getDescription());
						messageBox.open();
					}
				}

// if (supportedType.isAssignableFrom(IInformationCard.class)) {
//					try {
//						for (Iterator cards = p.getICards(handler, null); cards
//								.hasNext();) {
//							ICard card = (ICard) cards.next();
//							this.icards.add(card);
//						}
//					} catch (Exception e) {
//						log.error("Error when reading the cardstore", e);
//						if (shell != null) {
//							MessageBox messageBox = new MessageBox(shell,
//									SWT.ERROR_IO);
//							messageBox.setMessage("Error:\n" + e.getMessage()
//									+ "\nlook at logs for details.");
//							messageBox
//									.setText("Error when reading the cardstore");
//							messageBox.open();
//						}
//					}
//				} else if(supportedType.isAssignableFrom(IdemixCard.class)) {
//					try {
//						for (Iterator cards = p.getICards(handler, null); cards
//								.hasNext();) {
//							ICard card = (ICard) cards.next();
//							this.icards.add(card);
//						}
//					} catch (Exception e) {
//						log.fatal("Error when reading z-card storage", e);
//						if (shell != null) {
//							MessageBox messageBox = new MessageBox(shell,
//									SWT.ERROR_IO);
//							messageBox.setMessage("Error:\n" + e.getMessage()
//									+ "\nlook at logs for details.");
//							messageBox
//									.setText("Error when reading z-card storage");
//							messageBox.open();
//						}
//					}
//				}
			}
		}
	}
	
	//private void unselectCard(ICard card, Composite currentWidget) {
	private void unselectCard(Composite currentWidget, boolean removeWidget, boolean packParent) {
		log.info("unselectCard removeWidget="+removeWidget);
		Composite cardComposite = (Composite) currentWidget.getData(CARD_COMPOSITE);
		cardComposite.setBackground(blue);
		cardComposite.setForeground(black);
		RowData rd = new RowData();
		rd.width = imageSize.x;
		rd.height = imageSize.y;
		currentWidget.setLayoutData(rd);
		cardComposite.setBackground(white);
		Label cardLabel = (Label) currentWidget.getData(CARD_LABEL);
		cardLabel.setBackground(white);
		cardLabel.setForeground(blue);
		//cardLabel.setFont(sysFont);
		cardComposite.setSize(cardSize);
		Image image = (Image) currentWidget.getData(CARD_IMAGE);
//		log.info("unselect image=" + image);
		currentWidget.setBackgroundImage(image);
		currentWidget.setSize(imageSize);
		rd = new RowData();
		rd.width = cardSize.x;
		rd.height = cardSize.y;
		cardComposite.setLayoutData(rd);
		//if (this.currentWidget == cardComposite)
		if (this.currentWidget == currentWidget)
			this.currentWidget = null;
		if (removeWidget)
			this.currentWidgets.remove(((ICard)currentWidget.getData(ICARD)).getCUID());
		if (packParent)
			cardComposite.getParent().pack();
	}
	
	private void selectCard(ICard card, Composite currentWidget, boolean packParent) {
		// select the new graphics card
		Composite cardComposite = (Composite) currentWidget.getData(CARD_COMPOSITE);
		cardComposite.setForeground(white);
		cardComposite.setBackground(blue);
		Label cardLabel = (Label) currentWidget.getData(CARD_LABEL);
		cardLabel.setBackground(blue);
		cardLabel.setForeground(white);
		cardLabel.setFont(selectFont);
		if (bigAndSmall) {
			RowData rd = new RowData();
			rd.width = selectImageSize.x;
			rd.height = selectImageSize.y;
			currentWidget.setLayoutData(rd);
			Image image = (Image) currentWidget.getData(SELECT_CARD_IMAGE);
			log.info("image=" + image);
			currentWidget.setBackgroundImage(image);
			currentWidget.setSize(selectImageSize);
			cardComposite.setSize(selectCardSize);
			rd = new RowData();
			rd.width = selectCardSize.x;
			rd.height = selectCardSize.y;
			cardComposite.setLayoutData(rd);
		}
		if (packParent)
			cardComposite.getParent().pack();
		this.currentWidget = currentWidget; // remember that this card image is now current
		//this.currentWidget = cardComposite; // remember that this card image is now current
		//this.currentWidgets.put(card.getName(), currentWidget);
		this.currentWidgets.put(card.getCUID(), currentWidget);
		//this.currentWidgets.put(card.getName(), cardComposite);
	}


	private void addDropTarget() {
		// drag and drop
		// DropTarget target = new DropTarget(this.cardPanelComp, DND.DROP_COPY
		// | DND.DROP_DEFAULT);
		DropTarget target = new DropTarget(this.cardPanelComp, DND.DROP_COPY
				| DND.DROP_DEFAULT | DND.DROP_MOVE);
		final TextTransfer textTransfer = TextTransfer.getInstance();
		final FileTransfer fileTransfer = FileTransfer.getInstance();
		Transfer[] types = new Transfer[] { fileTransfer, textTransfer };
		target.setTransfer(types);

		target.addDropListener(new DropTargetListener() {
			public void dragEnter(DropTargetEvent event) {
				log.trace("DropTargetEvent:" + event);
				if (event.detail == DND.DROP_DEFAULT) {
					if ((event.operations & DND.DROP_COPY) != 0) {
						event.detail = DND.DROP_COPY;
					} else {
						event.detail = DND.DROP_NONE;
					}
				}
				for (int i = 0; i < event.dataTypes.length; i++) {
					if (fileTransfer.isSupportedType(event.dataTypes[i])) {
						event.currentDataType = event.dataTypes[i];
						// only copy
						if (event.detail != DND.DROP_COPY) {
							event.detail = DND.DROP_NONE;
						}
						break;
					}
				}
			}

			public void dragOver(DropTargetEvent event) {
				log.trace("dragOver");
				event.feedback = DND.FEEDBACK_SELECT | DND.FEEDBACK_SCROLL;
				if (textTransfer.isSupportedType(event.currentDataType)) {
					// NOTE: on unsupported platforms this will return null
					Object o = textTransfer.nativeToJava(event.currentDataType);
					log.trace("dragOver:" + o);
					String t = (String) o;
					if (t != null)
						System.out.println(t);
				}
			}

			public void dragOperationChanged(DropTargetEvent event) {
				log.trace("dragOperationChanged");
				if (event.detail == DND.DROP_DEFAULT) {
					if ((event.operations & DND.DROP_COPY) != 0) {
						event.detail = DND.DROP_COPY;
					} else {
						event.detail = DND.DROP_NONE;
					}
				}
			}

			public void dragLeave(DropTargetEvent event) {
			}

			public void dropAccept(DropTargetEvent event) {
				log.trace("dropAccept");
			}

			public void drop(DropTargetEvent event) {
				log.trace("drop");
				if (textTransfer.isSupportedType(event.currentDataType)) {
					log.trace("text supported");
				}
				if (fileTransfer.isSupportedType(event.currentDataType)) {
					String[] files = (String[]) event.data;
					importCards(files);
				}
			}

			// public void drop(DropTargetEvent event) {
			// log.trace("drop");
			// if (textTransfer.isSupportedType(event.currentDataType)) {
			// String text = (String) event.data;
			// // instr.setText(text);
			// log.trace("text supported");
			// }
			// if (fileTransfer.isSupportedType(event.currentDataType)) {
			// String[] files = (String[]) event.data;
			// // open file and read contents
			// File f = new File(files[0]);
			// if (!f.isFile()) {
			// // instr.setText("Drag a single .crds file only:");
			// return;
			// } else if (files[0].endsWith(".crd")) {
			// // directly place this file into the cardstore dir,
			// // rename it to prevent clash
			// StringBuffer buf = new StringBuffer();
			// Date d = new Date();
			// SimpleDateFormat sdf = new SimpleDateFormat(
			// "yyyy-MM-dd-HH-mm-ss");
			// String dest = _credDir + File.separator + f.getName()
			// + sdf.format(d) + ".crd";
			// log.trace("saving to " + dest);
			// try {
			// File fdest = new File(dest);
			// if (fdest != null && f.renameTo(fdest)) {
			// String xmlCrd = AxiomUtility.readTextFile(dest);
			// log.trace("insertNewCardToView(" + dest + ")");
			// ICard icard = PerspectiveCardStore
			// .getSecureCardProvider().importCard(
			// null, dest);
			// insertNewCardToView(new ICard[] { icard });
			// }
			// } catch (Exception e) {
			// log.error("drop", e);
			// }
			// } else {
			// String filename = files[0];
			//
			// Shell shell = PlatformUI.getWorkbench()
			// .getActiveWorkbenchWindow().getShell();
			// PasswordDialog pd = new PasswordDialog(shell, filename);
			// pd.open();
			// }
			// }
			// }

		});
	}

	/**
	 * @param icards
	 */
	private void insertNewCardToView(ICard[] icards) {
		for (int i = 0; i < icards.length; i++) {
			ICard card = icards[i];
			icardComposites.add(createCardComposite(card));
		}
		this.cardPanelComp.layout(true);
	}
	
	private void resetCurrentWidgets(boolean notifyAll) {
		log.info("resetCurrentWidgets");
		// go through the set of currently-selected cards and de-select them
		Iterator iter = currentWidgets.entrySet().iterator();
		boolean hadSome = iter.hasNext();
		while (iter.hasNext()) {
			Composite cardComposite = (Composite) ((Map.Entry)iter.next()).getValue();
			if (cardComposite != null)
				unselectCard(cardComposite, false, true);
		}
		currentWidgets = new HashMap();
		this.currentWidget = null;
		if (hadSome && notifyAll) {
			log.info("had some already selected");
			// notify listeners (hello CardListComposite?) that things are unset
			setSelection(new CardSelection(CardSelection.TYPE_RESET)); // clear any residual information from the edit window
		}
	}
	
	/**
	 * Read the names from the input list of icards and pick one not currently there
	 * @param icardsL the list of icards whose names we don't want to duplicate
	 * @param stemname the base name we should use to start with
	 * @return a name not currently in the list of icards
	 */
	private String pickNewCardName(List icardsL, String stemname) {
		String ans = EMPTY_S;
		if (icardsL == null)
			icardsL = new ArrayList();
		if ((stemname == null) || (EMPTY_S.equals(stemname)))
			stemname = "New Card ";
		// remember all the names currently in use
		HashMap map = new HashMap(icardsL.size());
		int i = 0;
		for (; i<icardsL.size(); i++) {
			map.put(((ICard)(icardsL.get(i))).getName(), EMPTY_S);
		}
		String tempS = null;
		// generate potential new names until we find one not in use
		while (true) {
			tempS = stemname + Integer.toString(i);
			if (map.get(tempS) == null) {
				ans = tempS;
				break;
			}
			i++;
		}
		return ans;
	}

	public void widgetSelected(String pass) {
		log.info("widgetSelected");
	}

//	private class CrdFileFilter implements FileFilter {
//
//		public boolean accept(File pathname) {
//			if (pathname.exists() && pathname.isFile() && pathname.canRead()
//					&& (pathname.getName().endsWith(".crds"))) {
//				return true;
//			}
//			return false;
//		}
//	}
//
//	private void watchDirectory(final String dir,
//			DirectorySurvey directorySurvey) {
//		File file = new File(dir);
//		this.swtSync_directorySurvey = directorySurvey;
//		if (file.exists() && file.isDirectory()) {
//			final File[] files = file.listFiles(new CrdFileFilter());
//			// check for files that disappear
//			if (files.length > 0) { // && Display.getCurrent()!=null) {
//				display.syncExec(new Runnable() {
//					public void run() {
//						openAsyncStorage(files, dir);
//					}
//				});
//			}
//		} else if (swtSync_directorySurvey.cardstore.size() > 0) {
//			// we need to remove all children as they are not accessible
//			Display.getCurrent().syncExec(new Runnable() {
//				public void run() {
//					display.syncExec(new Runnable() {
//						public void run() {
//							clearListCard();
//						}
//					});
//				}
//			});
//		}
//	}
//
//	/**
//	 * 
//	 */
//	public void run() {
//		log
//				.info("Starting Directory Survey Thread: waiting for the initialization of the Workbench ");
//		// wait for the whole Workbench to be initialized
//		boolean init = false;
//		do {
//			try {
//				PlatformUI.getWorkbench(); // .getActiveWorkbenchWindow().getShell();
//				init = true;
//			} catch (NullPointerException e) {
//				log.info("DirectorySurvey: waiting for Workbench init: " + e);
//				try {
//					Thread.sleep(1000);
//				} catch (InterruptedException ie) {
//				}
//			}
//		} while (init == false);
//		log.info("Starting Directory Survey Thread");
//		while (stop == false) {
//			String delaySz = Activator.getDefault().getPreferenceStore()
//					.getString("dirSurvey.delay");
//			int timeout = 0;
//			try {
//				timeout = Integer.valueOf(delaySz).intValue();
//			} catch (NumberFormatException e) {
//			}
//			if (timeout == 0) {
//				try {
//					Thread.sleep(1000);
//				} catch (InterruptedException e) {
//				}
//				continue;
//			}
//			String surveyPath = Activator.getDefault().getPreferenceStore()
//					.getString("dirSurvey.path");
//			DirectorySurvey directorySurvey = new DirectorySurvey();
//			// log.trace("surveyPath=[" + surveyPath + "] delay=[" + delaySz +
//			// "]");
//			try {
//				Thread.sleep(timeout);
//				watchDirectory(surveyPath, directorySurvey);
//			} catch (InterruptedException e) {
//				log.error("Error in the SurveyThread", e);
//			}
//		}
//		log.info("Stopping Directory Survey Thread");
//	}
//
//	/**
//	 * @param files
//	 * @return
//	 */
//	void openAsyncStorage(File[] files, String dir) {
//		int totalHash = 0;
//		for (int i = 0; i < files.length; i++) {
//			totalHash += files[i].getPath().hashCode();
//		}
//		Integer hash = new Integer(totalHash);
//		if (dirSurveyHistory.containsKey(hash))
//			return;
//		dirSurveyHistory.put(hash, hash);
//		DirSurveyDialog dialog = new DirSurveyDialog(shell, files, dir);
//		int ret = dialog.open();
//		if (ret == 0) {
//			String filename = dialog.getFilename();
//			String password = dialog.getPassword();
//			openStorage(filename, password);
//		}
//
//	}

	public static void openSecureStore(Shell shell, Exception e, String filename) {
		Class storeDecryptionExceptionClass = null;
		try {
			storeDecryptionExceptionClass = Class
					.forName("org.eclipse.higgins.cardstore.exceptions.StoreDecryptionException");
		} catch (ClassNotFoundException e1) {
			log
					.info("Warning: can not load class [org.eclipse.higgins.cardstore.exceptions.StoreDecryptionException]");
		}
		String message = "Error:\n" + e.getMessage()
				+ "\nwhen opening cardstore [" + filename + "]";
		if (e.getCause().getClass() == storeDecryptionExceptionClass) {
			message = "Decryption error for store [" + filename
					+ "]\nbad password ?";
		}
		log.error("Error when opening cardstore", e);
		MessageBox messageBox = new MessageBox(shell, SWT.ERROR_IO);
		messageBox.setMessage(message);
		messageBox.setText("Error when opening a cardstore");
		messageBox.open();
	}

//	public void resetSurveyDir() {
//		dirSurveyHistory.clear();
//	}
//
//	class DirectorySurvey {
//	    String dir;
//	    HashMap cardstore = new HashMap();
//	    TreeItem treeItem;
//	}

//	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
//		System.out.println("SelectionChanged called");
//		if (part != this && selection instanceof ICardSelection) {
//			System.out.println("iid = "+((ICardSelection)selection).getCard());
//			resetCurrentWidgets();
//			ICard card = ((ICardSelection)selection).getCard();
//			resetCardComposite(card);
////			if (this.observerPart != null)
////				observerPart.notifySelectionChanged(card);
//		}
//	}
	
    private static boolean isExtant(String filename) {
    	boolean ans = false;
		File icardFile = new File(filename);
		if ((icardFile.exists()) && (icardFile.isFile()))
			ans = true;
		return ans;
    }
 
	static {
		try {
			INFOCARD = Utils.loadImage("/image/infocard.png");
		} catch (Exception e) {}
		try {
			MAUS = Utils.loadImage("/image/maus.gif");
		} catch (Exception e) {}
	}

}
