/*******************************************************************************
 * Copyright (c) 2006-2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Tie Li (IBM Corporation) - initial API and implementation
 *******************************************************************************/ 

package org.eclipse.higgins.crpps.service;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.IJobChangeListener;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.higgins.configuration.xml.ConfigurationHandler;
import org.eclipse.higgins.crpps.ui.CardSelectorView;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.auth.ISelfIssuedCredential;
import org.eclipse.higgins.icard.auth.ITokenServiceCredential;
import org.eclipse.higgins.icard.provider.cardspace.common.CardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.PPIDCardSpacePolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.PersonalCard;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.iss.IICardSelector;
import org.eclipse.higgins.iss.SelectionANDofORs;
import org.eclipse.higgins.iss.SelectionANDofORsElm;
import org.eclipse.higgins.iss.UserChoiceTree_ANDofORs;
import org.eclipse.higgins.iss.cardspace.CardSpaceSelector;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;

public class InfoCardController{	
	private Log log = LogFactory.getLog(InfoCardController.class);
	
	private CardSpacePolicy policy;	
	private List rp_cert_chain = new ArrayList();
	private String rp_url = "";
	private String rp_object_name = "";
	
	private InfoCardModel model = new InfoCardModel();
	private RPPSService2 tokenListener;
		
	private CardSelectorView selector_view;
	private SecurityTokenBean cachedToken = null;
	
	private static InfoCardController _inst = null;
	
	private Job idpJob;
	private UserInterfaceInfoBean uiib;
	
	public static InfoCardController getInst(){
		if(_inst==null){
			_inst = new InfoCardController();
		}
		return _inst;
	}
		
	public void setSelectorView(CardSelectorView v){
		this.selector_view = v;
	}
			
	public void startup(SOAPMessage policyMsg, RPPSService2 tokenListener) throws SOAPException{
		this.tokenListener = tokenListener;		
		this.policy = new CardSpacePolicy();
		this.cachedToken = null;
		this.rp_cert_chain.clear();
				
		SOAPBody soapBody = policyMsg.getSOAPBody();
		Iterator iter = soapBody.getChildElements();
		while(iter.hasNext()){
			SOAPBodyElement elem = (SOAPBodyElement)iter.next();
			if("hbx_request".equals(elem.getNodeName())){
				Iterator contents = elem.getChildElements();
				while(contents.hasNext()){
					javax.xml.soap.Node contNode = (javax.xml.soap.Node) contents.next();
					String nodeName = contNode.getNodeName();
					if("object_name".equals(nodeName)) {
						this.rp_object_name = contNode.getValue();
					}
					if("document_URL".equals(nodeName)) {
						String encoded_url = contNode.getValue();
						try {
							this.rp_url = URLDecoder.decode(encoded_url, "UTF-8");
						} catch (UnsupportedEncodingException e) {
							this.rp_url = encoded_url;
						}
					}
					if("parameters".equals(nodeName)){
						Iterator parameters = ((javax.xml.soap.SOAPElement)contNode).getChildElements();
						while(parameters.hasNext()){
							javax.xml.soap.SOAPElement paramElement = (javax.xml.soap.SOAPElement)parameters.next();
							String name = paramElement.getAttribute("name");
							String value = paramElement.getAttribute("value");
							if("TokenType".equals(name)) {
								this.policy.setTokenType(value);
							}
							if("requiredClaims".equals(name)) {
								List requiredClaims = separateTokens(value);
								this.policy.setRequiredClaims(requiredClaims);
							}
							if("optionalClaims".equals(name)) {
								List optionalClaims = separateTokens(value);
								this.policy.setOptionalClaims(optionalClaims);
							}
							if("issuer".equals(name)){
								this.policy.setIssuer(value);
							}
							if("issuerPolicy".equals(name)){
								this.policy.setIssuerPolicy(value);
							}
							if("privacyUrl".equals(name)){
								this.policy.setPrivacyUrl(value);
							}
							if("privacyVersion".equals(name)){
								this.policy.setPrivacyVersion(value);
							}
						}
					}
					if ("certificate_chain".equals(nodeName)){
						Iterator parameters = ((javax.xml.soap.SOAPElement)contNode).getChildElements();
						while(parameters.hasNext()){
							javax.xml.soap.SOAPElement paramElement = (javax.xml.soap.SOAPElement)parameters.next();
							String value = paramElement.getValue();
							this.rp_cert_chain.add(value);
						}
					}
				}
			}
		}
								
		idpJob = new Job("Accessing I-Card selector service."){
			protected IStatus run(IProgressMonitor arg0) {
				uiib = model.handleUIRequest(rp_url, policy, "cardspace", "");								
				return Status.OK_STATUS;
			}
		};
		idpJob.addJobChangeListener(new IJobChangeListener(){
			public void aboutToRun(IJobChangeEvent event) {}
			public void awake(IJobChangeEvent event) {}			
			public void running(IJobChangeEvent event) {}
			public void scheduled(IJobChangeEvent event) {}
			public void sleeping(IJobChangeEvent event) {}
			public void done(IJobChangeEvent event) {
				Display.getDefault().asyncExec(new Runnable(){
					public synchronized void run() {
						selector_view.notifyUIInfo(rp_url, uiib);
					}
				});
			}
		});		
		Display.getDefault().asyncExec(new Runnable(){
			public void run() {
				IWorkbenchWindow iww = PlatformUI.getWorkbench()
					.getActiveWorkbenchWindow();
				try {
					Shell shell = iww.getShell();
					shell.setVisible(true);
					shell.setMinimized(false);
					PlatformUI.getWorkbench().showPerspective(
						"org.eclipse.higgins.crpps.perspective_selector", iww);										
					selector_view.reset();
					idpJob.schedule();
				} catch (WorkbenchException e) {
					log.error("Workbench cannot be launched.", e);
				}
			}
		});
	}
	
	private List separateTokens(String line){
		List list = new ArrayList();		
		String[] claims_split = line.split("\\s+");
		for (int i = 0; i < claims_split.length; i++) {
			if (claims_split[i].length()>0) {
				list.add(claims_split[i]);
			}
		}
		
		return list;
	}
			
	//serves for "preview" button
	public SecurityTokenBean previewToken(Vector optionalClaims, ICard card, 
			String pinCode) throws Exception{
		log.info("preview the display token");
		log.info("input optional claims are:");
		for(int i=0; i<optionalClaims.size(); i++){
			log.info(" " + i + optionalClaims.get(i));
		}
			
		policy.setOptionalClaims(optionalClaims);			
		cachedToken = model.handleGetTokenRequest(card, rp_url, policy, 
			"cardspace", pinCode);
		
		return cachedToken;
	}
	
	//serves for "submit" button
	public void sendToken(Vector optionalClaims, final ICard card, String pinCode) 
			throws Exception{		
		if(this.cachedToken == null){
			previewToken(optionalClaims, card, pinCode);
		}
		this.tokenListener.notifySecurityToken(cachedToken.getSecurityToken());
		
		CardHistoryBean chb = new CardHistoryBean();
		chb.setCard_id(card.getID());
		chb.setDoc_url(this.rp_url);
		CardHistoryManager.getInst().writeCardHistory(chb);
	}
	
	//serves for "cancel" button
	public void cancelPressed(){
		this.tokenListener.notifyCanceled();
	}
	
	public List get_rp_list(){
		return this.rp_cert_chain;
	}
	
	public PersonalCard findPersonalCardForManagedCard(ICard card)
			throws Exception{
		ConfigurationHandler ch = getConfigurationHandler(false);
		Map mapGlobalSettings = ch.getSettings();
		
		//Get the client side certificate store from configuration
		final KeyStore ksTrusted = (KeyStore)mapGlobalSettings.get("TrustedKeyStore");
		java.util.Collection collCertificates = new java.util.ArrayList();
		java.util.Enumeration enumCertificates = ksTrusted.aliases();
		while (enumCertificates.hasMoreElements()){
			String strAlias = (String)enumCertificates.nextElement();
			java.security.cert.Certificate cert = ksTrusted.getCertificate(strAlias);
			collCertificates.add(cert);
		}
		java.security.cert.CollectionCertStoreParameters collCSP = 
			new java.security.cert.CollectionCertStoreParameters(collCertificates);
		java.security.cert.CertStore certStoreTrust = 
			java.security.cert.CertStore.getInstance("Collection", collCSP);
				
		ICredentialDescriptor[] credDescr = card.getRequiredCredentials();
		if(credDescr != null){
			String endpointURI = null;
			X509Certificate endpointCert = null;
			byte[] bppid = null;
			
			ICredential cred = credDescr[0].getCredential();
			if (cred instanceof ITokenServiceCredential) {
				ITokenServiceCredential tsc = (ITokenServiceCredential) cred;
				endpointCert = tsc.getCertificate();
				endpointURI = tsc.getAddress().toString();
			}			
			if(cred instanceof ISelfIssuedCredential){
				ISelfIssuedCredential sic = (ISelfIssuedCredential) cred;
				String ppid = sic.getPPID();
				bppid = org.apache.commons.codec.binary.Base64.decodeBase64(ppid.getBytes());		
			}
			if(endpointCert != null && bppid != null){
				PPIDCardSpacePolicy ppp = new PPIDCardSpacePolicy(
					new X509Certificate[] {endpointCert}, 
					certStoreTrust, bppid, endpointURI);
				String stsConfigBase = System.getProperty("org.eclipse.higgins.sts.conf");
				IICardSelector selector = new CardSpaceSelector(stsConfigBase, null, ppp);
				UserChoiceTree_ANDofORs uct = (UserChoiceTree_ANDofORs) selector.getUserChoice();
				SelectionANDofORs selection = uct.getDefaultSelection();
				SelectionANDofORsElm elm = (SelectionANDofORsElm) selection.getElements().get(0);
				String card_uid = elm.getUUID();
				ICardRegistry registry = ICardRegistry.getInstance();
				ICard personalCard = registry.getICardByCUID(null, card_uid);
				if(personalCard instanceof PersonalCard){
					return (PersonalCard) personalCard;
				}
			}
		}
		
		return null;
	}
	
	public ConfigurationHandler getConfigurationHandler(boolean selfIssued) throws Exception {
		String stsConfigurationBase = System.getProperty("org.eclipse.higgins.sts.conf");		
		if (null == stsConfigurationBase) {
			throw new Exception("CardspaceSelector: org.eclipse.higgins.sts.conf System property not found");
		}
		String stsConfigurationFile = null;
		String stsConfigurationPersonalFile = System.getProperty("org.eclipse.higgins.sts.conf.personal.file");
		String stsConfigurationManagedFile = System.getProperty("org.eclipse.higgins.sts.conf.managed.client.file");
		if (selfIssued) {
			if (stsConfigurationPersonalFile == null)
				stsConfigurationFile = "PersonalConfiguration.xml";
			else
				stsConfigurationFile = stsConfigurationPersonalFile;
		}
		else {
			if (stsConfigurationManagedFile == null)
				stsConfigurationFile = "ClientConfiguration.xml";
			else
				stsConfigurationFile = stsConfigurationManagedFile;
		}	
		ConfigurationHandler configurationHandler = new ConfigurationHandler();
		configurationHandler.setConfigurationBase(stsConfigurationBase);
		
		configurationHandler.setFileName(stsConfigurationFile);

		if (!configurationHandler.configure(null)) {
			log.error("Not Initialized!");
			throw new Exception("CardspaceSelector: cannot initialize configHandler");
		}

		return configurationHandler;
	}
}
