/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.net.URI;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.higgins.as.util.Constants;
import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.as.util.IdASHinUtil;
import org.eclipse.higgins.as.util.MapUtil;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.MessagingTarget;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class AttributeSubjectPredicateInnerGraphResourceHandler extends AbstractResourceHandler {

	private XRI3Segment hin;
	private URI entityType;
	private URI attrId;
	private int deep;

	private IContext context;
	private IAttribute parentAttribute;
	private IEntity entity;
	private IAttribute attribute;

	AttributeSubjectPredicateInnerGraphResourceHandler(Message message, Subject subject, Predicate predicate, Graph innerGraph, int deep, IContext context, IAttribute parentAttribute) {

		super(message, subject, predicate, innerGraph);

		this.context = context;
		this.parentAttribute = parentAttribute;

		this.hin = subject.getSubjectXri();
		this.attrId = IdASUtil.xriToAttrID(predicate.getPredicateXri());
		this.deep = deep;

		// read additional information from the message

		Predicate typePredicate = subject.getPredicate(Constants.XRI_RDFTYPE);
		if (typePredicate != null && typePredicate.containsReferences()) {

			this.entityType = URI.create(IdASUtil.xriToEntityID(typePredicate.getReference().getReferenceXri()));
		}
	}

	@Override
	public boolean executeGet(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext)executionContext);
			if (this.entity == null) throw new RuntimeException("Entity value " + this.hin.toString() + " not found.");
			if (this.attribute == null) throw new RuntimeException("Entity value " + this.hin + " does not have attribute " + this.attrId.toString() + ".");

			// map value and attribute value

			Graph graph = messageResult.getGraph();

			Subject subject = MapUtil.mapMultiEntityValue(this.entity, graph, 0);
			Predicate predicate = MapUtil.mapAttribute(this.attribute, subject, 0);
			MapUtil.mapMultiValue(null, predicate, this.deep);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public boolean executeAdd(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext)executionContext);

			// create entity and attribute

			if (this.entity == null) {
				
				this.entity = this.context.addEntity(this.entityType, null);
				this.parentAttribute.addValue(this.entity);
			}
			if (this.attribute == null) this.attribute = this.entity.addAttribute(this.attrId);
			this.context.applyUpdates();
			((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
			((IdASContextExecutionContext) executionContext).getHinCache().put(this.hin, this.entity);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public boolean executeMod(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext)executionContext);
			if (this.entity == null) throw new RuntimeException("Entity value " + this.hin.toString() + " not found.");
			if (this.attribute == null) throw new RuntimeException("Entity value " + this.hin + " does not have attribute " + this.attrId.toString() + ".");
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public boolean executeDel(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext)executionContext);
			if (this.entity == null) throw new RuntimeException("Entity value " + this.hin.toString() + " not found.");
			if (this.attribute == null) throw new RuntimeException("Entity value " + this.hin + " does not have attribute " + this.attrId.toString() + ".");
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public MessagingTarget getInnerMessagingTarget(Object executionContext) throws MessagingException {

		try {

			// return MessagingTarget that represents this attribute of the entity value

			return(new AttributeMessagingTarget(this.deep - 1, this.context, this.attribute, executionContext));
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
	}

	@SuppressWarnings("unchecked")
	private void lookup(IdASContextExecutionContext executionContext) throws Exception  {

		this.entity = null;
		this.attribute = null;

		// find entity value and attribute of entity value

		Map<XRI3Segment, IValue> hinCache = executionContext.getHinCache();

		if (hinCache.containsKey(this.hin)) {

			this.entity = (IEntity) hinCache.get(this.hin);
		} else {

			String algorithm = IdASHinUtil.segmentToAlgorithm(this.hin);
			if (algorithm == null) throw new Exception("Invalid HIN: " + this.hin);

			Iterator i = this.parentAttribute.getValues();

			if (i != null) while (i.hasNext()) {

				IValue value = (IValue) i.next();
				if (! (value instanceof IEntity)) continue;

				this.entity = (IEntity) value;

				if (this.hin.equals(IdASHinUtil.makeHin(this.entity, algorithm))) break; else this.entity = null;
			}

			if (this.entity != null) hinCache.put(this.hin, this.entity);
		}

		if (this.entity == null) return;

		this.attribute = this.entity.getAttribute(this.attrId);
		if (this.attribute == null) return;
	}
}
