/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.util.Iterator;
import java.util.Map;

import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.as.util.IdASHinUtil;
import org.eclipse.higgins.as.util.MapUtil;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class AttributeSubjectResourceHandler extends AbstractResourceHandler {

	private XRI3Segment hin;
	private int deep;

	private IContext context;
	private IAttribute parentAttribute;
	private IValue value;

	public AttributeSubjectResourceHandler(Message message, Subject subject, int deep, IContext context, IAttribute parentAttribute) {

		super(message, subject);

		this.hin = subject.getSubjectXri();
		this.deep = deep;
		
		this.context = context;
		this.parentAttribute = parentAttribute;
	}

	@Override
	public boolean executeGet(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext) executionContext);
			if (this.value == null) throw new RuntimeException("Value " + this.hin.toString() + " not found.");

			// map value

			Predicate predicate = messageResult.getGraph().getPredicate();

			MapUtil.mapMultiValue(this.value, predicate, this.deep);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public boolean executeDel(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext) executionContext);
			if (this.value == null) throw new RuntimeException("Value " + this.hin.toString() + " not found.");

			// delete value

			this.value.remove();
			this.context.applyUpdates();
			((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@SuppressWarnings("unchecked")
	private void lookup(IdASContextExecutionContext executionContext) throws Exception {

		this.value = null;

		// find value

		Map<XRI3Segment, IValue> hinCache = executionContext.getHinCache();

		if (hinCache.containsKey(this.hin)) {

			this.value = hinCache.get(this.hin);
		} else {

			String algorithm = IdASHinUtil.segmentToAlgorithm(this.hin);
			if (algorithm == null) throw new Exception("Invalid HIN: " + this.hin);

			Iterator i = this.parentAttribute.getValues();

			if (i != null) while (i.hasNext()) {

				this.value = (IValue) i.next();

				if (this.hin.equals(IdASHinUtil.makeHin(this.value, algorithm))) break; else this.value = null;
			}

			if (this.value != null) hinCache.put(this.hin, this.value);
		}

		if (this.value == null) return;
	}
}
