/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.util.Map;

import javax.servlet.ServletContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.xml.ConfigurationHandler;
import org.eclipse.higgins.idas.api.IAuthNMaterials;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.AuthNMaterialsSerializer;
import org.eclipse.higgins.idas.udi.IContextUDI;
import org.eclipse.higgins.idas.udi.UDIResolver;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Literal;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Reference;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.server.EndpointRegistry;
import org.eclipse.higgins.xdi4j.messaging.server.impl.ResourceHandler;
import org.eclipse.higgins.xdi4j.messaging.server.impl.ResourceMessagingTarget;

public class IdASContextMessagingTarget extends ResourceMessagingTarget {

	private static final Log log = LogFactory.getLog(IdASContextMessagingTarget.class);

	private IContext context;
	private String higginsConfigurationFile;
	private String contextUdi;
	private String authNMaterialsType;
	private String authNMaterials;
	private String sessionKey;

	private long lastActivityMillis;

	public IdASContextMessagingTarget() {

		super(false);

		this.context = null;
		this.higginsConfigurationFile = "./WEB-INF/higgins.xml";
		this.contextUdi = null;
		this.authNMaterialsType = null;
		this.authNMaterials = null;
		this.sessionKey = null;
	}

	public void init(EndpointRegistry endpointRegistry) throws Exception {

		super.init(endpointRegistry);

		if (this.context == null) {

			// load the Higgins UDIResolver

			ServletContext servletContext = endpointRegistry.getServletContext();
			String higginsConfigurationFileRealPath = servletContext.getRealPath(this.higginsConfigurationFile);

			log.debug("Trying to load Higgins configuration file " + higginsConfigurationFileRealPath + ".");

			ConfigurationHandler configurationHandler;
			Map<?, ?> configurationSettings;

			configurationHandler = new ConfigurationHandler();
			configurationHandler.setConfigurationBase("");
			configurationHandler.setFileName(higginsConfigurationFileRealPath);
			configurationHandler.configure(null);
			configurationSettings = configurationHandler.getSettings();

			UDIResolver udiResolver = (UDIResolver) configurationSettings.get("UDIResolver");
			if (udiResolver == null) throw new RuntimeException("Cannot find UDIResolver component in Higgins configuration.");

			log.info("Loaded Higgins configuration.");

			// open the context

			IContextUDI contextUDI = udiResolver.parseContextUDI(this.contextUdi);
			if (contextUDI == null) throw new NullPointerException("contextUDI is null");

			String[] contextTypes = contextUDI.getContextUDIMetadata().getTypes();

			this.context = udiResolver.resolve(contextUDI);
			if (this.context == null) throw new NullPointerException("context is null");

			IAuthNMaterials authNMaterials = AuthNMaterialsSerializer.deserialize(this.authNMaterialsType, this.authNMaterials);

			this.context.open(authNMaterials);

			log.info("Successfully resolved and opened context. Context UDI: " + this.contextUdi + ", Context Types: " + contextTypes);
		}

		// init synchronization

		if (this.contextUdi != null) {

			this.messageEnvelopeInterceptors.add(new SynchronizingMessageEnvelopeInterceptor(this.contextUdi));
		}

		// init activity timer

		this.lastActivityMillis = System.currentTimeMillis();
	}

	public void shutdown() throws Exception {

		super.shutdown();

		try {

			if (this.context != null) this.context.close();
		} catch (IdASException ex) {

			log.error("Problem while shutting down Higgins context: " + ex.getMessage(), ex);
		}
	}

	@Override
	public Object getExecutionContext() {

		return(new IdASContextExecutionContext());
	}

	@Override
	public ResourceHandler getResource(Message message, Graph operationGraph) {

		this.lastActivityMillis = System.currentTimeMillis();

		return(new GraphResourceHandler(message, operationGraph, this.context));
	}

	@Override
	public ResourceHandler getResource(Message message, Subject subject) {

		this.lastActivityMillis = System.currentTimeMillis();

		return(new SubjectResourceHandler(message, subject, this.context));
	}

	@Override
	public ResourceHandler getResource(Message message, Subject subject, Predicate predicate) {

		this.lastActivityMillis = System.currentTimeMillis();

		return(new SubjectPredicateResourceHandler(message, subject, predicate, this.context));
	}

	@Override
	public ResourceHandler getResource(Message message, Subject subject, Predicate predicate, Reference reference) {

		this.lastActivityMillis = System.currentTimeMillis();

		return(new SubjectPredicateReferenceResourceHandler(message, subject, predicate, reference, this.context));
	}

	@Override
	public ResourceHandler getResource(Message message, Subject subject, Predicate predicate, Literal literal) {

		this.lastActivityMillis = System.currentTimeMillis();

		return(new SubjectPredicateLiteralResourceHandler(message, subject, predicate, literal, this.context));
	}

	@Override
	public ResourceHandler getResource(Message message, Subject subject, Predicate predicate, Graph innerGraph) {

		this.lastActivityMillis = System.currentTimeMillis();

		return(new SubjectPredicateInnerGraphResourceHandler(message, subject, predicate, innerGraph, this.context));
	}

	public IContext getContext() {

		return this.context;
	}

	public void setContext(IContext context) {

		this.context = context;
	}

	public String getHigginsConfigurationFile() {

		return this.higginsConfigurationFile;
	}

	public void setHigginsConfigurationFile(String higginsConfigurationFile) {

		this.higginsConfigurationFile = higginsConfigurationFile;
	}

	public String getContextUdi() {

		return this.contextUdi;
	}

	public void setContextUdi(String contextUdi) {

		this.contextUdi = contextUdi;
	}

	public String getAuthNMaterialsType() {

		return this.authNMaterialsType;
	}

	public void setAuthNMaterialsType(String authNMaterialsType) {

		this.authNMaterialsType = authNMaterialsType;
	}

	public String getAuthNMaterials() {

		return this.authNMaterials;
	}

	public void setAuthNMaterials(String authNMaterialsStr) {

		this.authNMaterials = authNMaterialsStr;
	}

	public String getSessionKey() {

		return this.sessionKey;
	}

	public void setSessionKey(String sessionKey) {

		this.sessionKey = sessionKey;
	}

	public long getLastActivityMillis() {

		return this.lastActivityMillis;
	}
}
