/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.as.util.Constants;
import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.xdi4j.Literal;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;

public class SubjectPredicateLiteralResourceHandler extends AbstractResourceHandler {

	private String entityId;
	private URI entityType;
	private URI attrId;
	private String simpleValueData;

	private IContext context;
	private IEntity entity;
	private IAttribute attribute;
	private ISimpleValue simpleValue;

	SubjectPredicateLiteralResourceHandler(Message message, Subject subject, Predicate predicate, Literal literal, IContext context) {

		super(message, subject, predicate, literal);

		this.context = context;

		this.entityId = IdASUtil.xriToEntityID(subject.getSubjectXri());
		this.attrId = IdASUtil.xriToAttrID(predicate.getPredicateXri());
		this.simpleValueData = literal.getData();

		// read additional information from the message

		Predicate typePredicate = subject.getPredicate(Constants.XRI_RDFTYPE);
		if (typePredicate != null && typePredicate.containsReferences()) {

			this.entityType = URI.create(IdASUtil.xriToEntityID(typePredicate.getReference().getReferenceXri()));
		}
	}

	@Override
	public boolean executeAdd(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup();

			// create entity and attribute and simple value

			if (this.entity == null) this.entity = this.context.addEntity(this.entityType, this.entityId);
			if (this.attribute == null) this.attribute = this.entity.addAttribute(this.attrId);
			this.simpleValue = this.attribute.addSimpleValue(ITypedValue.STRING_TYPE_URI, this.simpleValueData);
			this.context.applyUpdates();
			((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@Override
	public boolean executeMod(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup();
			if (this.entity == null) throw new RuntimeException("Entity " + this.entityId + " not found.");
			if (this.attribute == null) throw new RuntimeException("Entity " + this.entityId + " does not have attribute " + this.attrId.toString() + ".");
			if (! (this.attribute.getValues().next() instanceof ISimpleValue)) throw new RuntimeException("Entity " + this.entityId + " has attribute " + this.attrId.toString() + ", but no simple value to modify.");

			// modify simple value

			this.simpleValue = (ISimpleValue) this.attribute.getValues().next();
			this.simpleValue.setData(this.simpleValueData);
			this.context.applyUpdates();
			((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@SuppressWarnings("unchecked")
	private void lookup() throws Exception {

		this.entity = null;
		this.attribute = null;
		this.simpleValue = null;

		// find entity and attribute and simple value

		try { this.entity = this.context.getEntity(this.entityId); } catch (NoSuchEntityException ex) { this.entity = null; }
		if (this.entity != null && this.entity.isProxy()) this.entity = null;
		if (this.entity == null) return;

		this.attribute = this.entity.getAttribute(this.attrId);
		if (this.attribute == null) return;

		Iterator i = this.attribute.getValues();

		if (i != null) while (i.hasNext()) {

			IValue value = (IValue) i.next();
			if (! (value instanceof ISimpleValue)) continue;

			if (this.simpleValueData.equals(((ISimpleValue) value).getLexical())) {
			
				this.simpleValue = (ISimpleValue) value;
				return;
			}
		}
	}
}
