/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.as.util.Constants;
import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.as.util.MapUtil;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Reference;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;

public class SubjectPredicateReferenceResourceHandler extends AbstractResourceHandler {

	private String entityId;
	private URI entityType;
	private URI attrId;
	private String entityValueId;

	private IContext context;
	private IEntity entity;
	private IAttribute attribute;
	private IEntity entityValue;

	SubjectPredicateReferenceResourceHandler(Message message, Subject subject, Predicate predicate, Reference reference, IContext context) {

		super(message, subject, predicate, reference);

		this.context = context;

		this.entityId = IdASUtil.xriToEntityID(subject.getSubjectXri());
		this.attrId = IdASUtil.xriToAttrID(predicate.getPredicateXri());
		this.entityValueId = IdASUtil.xriToEntityID(reference.getReferenceXri());

		// read additional information from the message

		Predicate typePredicate = subject.getPredicate(Constants.XRI_RDFTYPE);
		if (typePredicate != null && typePredicate.containsReferences()) {

			this.entityType = URI.create(IdASUtil.xriToEntityID(typePredicate.getReference().getReferenceXri()));
		}
	}

	@Override
	public boolean executeGet(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup();
			if (this.entity == null) throw new RuntimeException("Entity " + this.entityId + " not found.");
			if (this.attribute == null) throw new RuntimeException("Attribute " + this.attrId.toString() + " on entity " + this.entityId + " not found.");
			if (this.entityValue == null) throw new RuntimeException("Entity Value " + this.entityValueId + " on attribute " + this.attrId.toString() + " on entity " + this.entityId + " not found.");

			// map entity and attribute and entity value

			Graph graph = messageResult.getGraph();
			Subject subject = MapUtil.mapEntity(this.entity, graph, 0);
			Predicate predicate = MapUtil.mapAttribute(this.attribute, subject, 0);
			MapUtil.mapEntityValue(this.entityValue, predicate, 0);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@Override
	public boolean executeAdd(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup();

			// create entity and attribute and entity value

			if (this.entity == null) this.entity = this.context.addEntity(this.entityType, this.entityId);
			if (this.operationPredicate.getPredicateXri().equals(Constants.XRI_RDFTYPE)) {
				
				this.context.applyUpdates();
				((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
				return(true);
			}

			if (this.attribute == null) this.attribute = this.entity.addAttribute(this.attrId);
			this.attribute.addComplexValue(this.entityValueId);
			this.context.applyUpdates();
			((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

/*	@Override
	public boolean executeDel(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		if (this.predicate.getPredicateXri().equals(Constants.XRI_RDFTYPE)) return(true);

		try {

			this.lookup();
			if (this.entity == null) throw new RuntimeException("Entity " + this.entityId + " not found.");
			if (this.attribute == null) throw new RuntimeException("Attribute " + this.attrId.toString() + " on entity " + entityId + " not found.");
			if (this.entityValueId == null) throw new RuntimeException("Entity Value " + this.entityValueId + " not found.");

			// delete value of attribute

			this.entity2.r
			this.context.applyUpdates();
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}*/

	@SuppressWarnings("unchecked")
	private void lookup() throws Exception {

		this.entity = null;
		this.attribute = null;
		this.entityValue = null;

		// find entity and attribute and entity value

		try { this.entity = this.context.getEntity(this.entityId); } catch (NoSuchEntityException ex) { this.entity = null; }
		if (this.entity != null && this.entity.isProxy()) this.entity = null;
		if (this.entity == null) return;

		this.attribute = this.entity.getAttribute(this.attrId);
		if (this.attribute == null) return;

		Iterator i = this.attribute.getValues();

		if (i != null) while (i.hasNext()) {

			IValue value = (IValue) i.next();
			if (! (value instanceof IEntity)) continue;

			if (this.entityValueId.equals(((IEntity) value).getEntityID())) {
				
				this.entityValue = (IEntity) value;
				return;
			}
		}
	}
}
