/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.net.URI;

import org.eclipse.higgins.as.util.Constants;
import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.as.util.MapUtil;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;

public class SubjectResourceHandler extends AbstractResourceHandler {

	private String entityId;
	private URI entityType;
	private int deep;

	private IContext context;
	private IEntity entity;

	SubjectResourceHandler(Message message, Subject subject, IContext context) {

		super(message, subject);

		this.context = context;
		
		this.entityId = IdASUtil.xriToEntityID(subject.getSubjectXri());

		// read additional information from the message

		Predicate typePredicate = subject.getPredicate(Constants.XRI_RDFTYPE);
		if (typePredicate != null && typePredicate.containsReferences()) {

			this.entityType = URI.create(IdASUtil.xriToEntityID(typePredicate.getReference().getReferenceXri()));
		}

		Predicate deepPredicate = message.getSubject().getPredicate(Constants.XRI_DEEP);
		if (deepPredicate != null && deepPredicate.containsLiteral()) {

			this.deep = Integer.parseInt(deepPredicate.getLiteral().getData());
		}
	}

	@Override
	public boolean executeGet(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup();
			if (this.entity == null) throw new RuntimeException("Entity " + this.entityId + " not found.");

			// map entity

			Graph graph = messageResult.getGraph();

			MapUtil.mapEntity(this.entity, graph, this.deep);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@Override
	public boolean executeAdd(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			// create entity

			this.entity = this.context.addEntity(this.entityType, this.entityId);
			this.context.applyUpdates();
			((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@Override
	public boolean executeDel(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup();
			if (this.entity == null) throw new RuntimeException("Entity " + this.entityId + " not found.");

			// delete entity

			this.entity.remove();
			this.context.applyUpdates();
			((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	private void lookup() throws Exception {

		this.entity = null;

		// find entity

		try { this.entity = this.context.getEntity(this.entityId); } catch (NoSuchEntityException ex) { this.entity = null; }
		if (this.entity != null && this.entity.isProxy()) this.entity = null;
		if (this.entity == null) return;
	}
}
