package org.eclipse.higgins.as;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.as.util.SynchronizingUtil;
import org.eclipse.higgins.xdi4j.addressing.Addressing;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.messaging.MessageEnvelope;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.server.interceptor.MessageEnvelopeInterceptor;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class SynchronizingMessageEnvelopeInterceptor implements MessageEnvelopeInterceptor {

	private static final Log log = LogFactory.getLog(SynchronizingMessageEnvelopeInterceptor.class);

	private static final XRI3Segment XRI_TRUE = new XRI3Segment("$true");
	private static final XRI3Segment XRI_FALSE = new XRI3Segment("$false");
	private static final XRI3Segment XRI_SELF = new XRI3Segment("$");
	private static final XRI3Segment XRI_VERSION = new XRI3Segment("$v");

	private String contextUdi;

	public SynchronizingMessageEnvelopeInterceptor(String contextUdi) {

		this.contextUdi = contextUdi;
	}

	public boolean before(MessageEnvelope messageEnvelope, MessageResult messageResult, Object executionContext) throws MessagingException {

		// check if the XDI message specifies a version number

		String expectedVersionStr = Addressing.findLiteralData(messageEnvelope.getGraph(), new XRI3("$/$v"));

		if (expectedVersionStr != null) {

			int currentVersion;
			int expectedVersion;

			// read the XDI message's required version number

			expectedVersion = Integer.parseInt(expectedVersionStr);

			// read context's current version

			try {

				currentVersion = SynchronizingUtil.getCurrentVersion(this.contextUdi);
			} catch (IOException ex) {

				throw new MessagingException("Cannot read context's version number: " + ex.getMessage(), ex);
			}

			// only proceed normally if the expected version is the same as the current version

			if (currentVersion != expectedVersion) {

				log.debug("Client's context version (" + expectedVersion + ") is different from ours (" + currentVersion + "). Refusing to process message.");

				messageResult.getGraph().createStatement(XRI_FALSE);
				messageResult.getGraph().createStatement(XRI_SELF, XRI_VERSION, Integer.toString(currentVersion));
				return true;
			} else {

				log.debug("Client's context version is equal to ours (" + currentVersion + "). Continuing to process message.");
			}
		}

		// done

		return false;
	}

	public boolean after(MessageEnvelope messageEnvelope, MessageResult messageResult, Object executionContext) throws MessagingException {

		// check if the XDI message specifies a version number

		String expectedVersionStr = Addressing.findLiteralData(messageEnvelope.getGraph(), new XRI3("$/$v"));

		if (expectedVersionStr != null) {

			// update or read context's current version

			int currentVersion;

			if (((IdASContextExecutionContext) executionContext).getIncreaseVersion()) {

				try {

					currentVersion = SynchronizingUtil.incCurrentVersion(this.contextUdi);

					log.debug("Updated context version (" + currentVersion + ").");
				} catch (IOException ex) {

					throw new MessagingException("Cannot update context's version number: " + ex.getMessage(), ex);
				}
			} else {

				try {

					currentVersion = SynchronizingUtil.getCurrentVersion(this.contextUdi);
				} catch (IOException ex) {

					throw new MessagingException("Cannot read context's version number: " + ex.getMessage(), ex);
				}
			}

			// output current version

			messageResult.getGraph().createStatement(XRI_TRUE);
			messageResult.getGraph().createStatement(XRI_SELF, XRI_VERSION, Integer.toString(currentVersion));
		}

		// done

		return false;
	}

	public void exception(MessageEnvelope messageEnvelope, MessageResult messageResult, Object executionContext, Exception ex) {
		
	}
}
