/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as.util;

import java.net.URI;
import java.util.Iterator;

import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.idas.common.SimpleValueSerializer;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Reference;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.constants.TypeConstants;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

@SuppressWarnings("unchecked")
public class MapUtil {

	private MapUtil() { }

	public static Subject mapEntity(IEntity entity, Graph graph, int deep) throws IdASException {

		// get or create subject

		XRI3Segment subjectXri = IdASUtil.entityIDToXri(entity.getEntityID());
		Subject subject = graph.getSubject(subjectXri);
		if (subject == null) subject = graph.createSubject(subjectXri);

		// map type

		URI entityType = entity.getType();
		if (entityType != null) {

			XRI3Segment typeXri = IdASUtil.entityIDToXri(entityType.toString());
			subject.createStatement(Constants.XRI_RDFTYPE, typeXri);
		}

		// map attributes?

		if (deep > 0) {

			Iterator attributes = entity.getAttributes();

			while (attributes.hasNext()) {

				MapUtil.mapAttribute((IAttribute) attributes.next(), subject, deep - 1);
			}
		}

		return(subject);
	}

	public static Predicate mapAttribute(IAttribute attribute, Subject subject, int deep) throws IdASException {

		// get or create predicate

		XRI3Segment predicateXri = IdASUtil.attrIDToXri(attribute.getType());
		Predicate predicate = subject.getPredicate(predicateXri);
		if (predicate == null) predicate = subject.createPredicate(predicateXri);

		// map values?

		if (deep > 0) {

			Iterator values = attribute.getValues();

			if (hasOnlyEntityValues(attribute)) {

				if (values != null) while (values.hasNext()) {

					mapEntityValue((IEntity) values.next(), predicate, deep - 1);
				}
			} else if (hasSingleSimpleValueWithStringType(attribute)) {

				if (values != null) {

					mapSingleSimpleStringValue((ISimpleValue) values.next(), predicate);
				}
			} else {

				if (values != null) while (values.hasNext()) {

					mapMultiValue((IValue) values.next(), predicate, deep - 1);
				}
			}
		}

		return(predicate);
	}

	public static void mapEntityValue(IEntity value, Predicate predicate, int deep) throws IdASException {

		if (value.getEntityID() != null) {

			XRI3Segment referenceXri = IdASUtil.entityIDToXri(value.getEntityID());
			if (! predicate.containsReference(referenceXri)) predicate.createReference(referenceXri);
		} else {

			// get or create inner graph subject

			XRI3Segment referenceXri = IdASHinUtil.makeHin(value, IdASHinUtil.ALGORITHM_DEFAULT);
			Reference reference = predicate.getReference(referenceXri);
			if (reference == null) reference = predicate.createReference(referenceXri);

			// get or create entity subject

			Subject subject = predicate.getTopLevelGraph().getSubject(referenceXri);

			if (subject == null) {

				subject = predicate.getTopLevelGraph().createSubject(referenceXri);

				// map type

				URI entityType = value.getType();
				if (entityType != null) {

					XRI3Segment typeXri = IdASUtil.entityIDToXri(entityType.toString());
					subject.createStatement(Constants.XRI_RDFTYPE, typeXri);
				}

				// map attributes?

				if (deep > 0) {

					Iterator attributes = value.getAttributes();

					while (attributes.hasNext()) {

						MapUtil.mapAttribute((IAttribute) attributes.next(), subject, deep - 1);
					}
				}
			}
		}
	}

	public static void mapSingleSimpleStringValue(ISimpleValue value, Predicate predicate) throws IdASException {

		predicate.createLiteral(value.getLexical());
	}

	public static void mapMultiValue(IValue value, Predicate predicate, int deep) throws IdASException {

		// get or create inner graph

		Graph innerGraph = predicate.getInnerGraph();
		if (innerGraph == null) innerGraph = predicate.createInnerGraph(null);

		// map value

		if (value instanceof ISimpleValue) {

			mapMultiSimpleValue((ISimpleValue) value, innerGraph);
		} else if (value instanceof IEntity) {

			mapMultiEntityValue((IEntity) value, innerGraph, deep);
		}
	}

	public static void mapMultiSimpleValue(ISimpleValue simpleValue, Graph graph) throws IdASException {

		// get or create inner graph subject

		XRI3Segment subjectXri = IdASHinUtil.makeHin(simpleValue, IdASHinUtil.ALGORITHM_DEFAULT);
		Subject innerGraphSubject = graph.getSubject(subjectXri);
		if (innerGraphSubject == null) innerGraphSubject = graph.createSubject(subjectXri);

		// map type

		URI type = simpleValue.getType();
		if (type != null) {

			innerGraphSubject.createStatement(Constants.XRI_DATATYPE, IdASUtil.entityIDToXri(type.toString()));
		}

		// map data

		String literalData = SimpleValueSerializer.serialize(simpleValue.getData());
		innerGraphSubject.createStatement(TypeConstants.XRI_VALUE, literalData);
	}

	public static Subject mapMultiEntityValue(IEntity entity, Graph graph, int deep) throws IdASException {

		if (entity.getEntityID() != null) {

			// get or create inner graph subject

			XRI3Segment subjectXri = IdASUtil.entityIDToXri(entity.getEntityID());
			Subject innerGraphSubject = graph.getSubject(subjectXri);
			if (innerGraphSubject == null) innerGraphSubject = graph.createSubject(subjectXri);
			
			// get or create entity subject
	
			Subject subject = graph.getTopLevelGraph().getSubject(subjectXri);
	
			if (subject == null) {
	
				subject = graph.getTopLevelGraph().createSubject(subjectXri);
	
				// map type
	
				URI entityType = entity.getType();
				if (entityType != null) {
	
					XRI3Segment typeXri = IdASUtil.entityIDToXri(entityType.toString());
					subject.createStatement(Constants.XRI_RDFTYPE, typeXri);
				}
			}
			
			return subject;
		} else {

			// get or create inner graph subject

			XRI3Segment subjectXri = IdASHinUtil.makeHin(entity, IdASHinUtil.ALGORITHM_DEFAULT);
			Subject innerGraphSubject = graph.getSubject(subjectXri);
			if (innerGraphSubject == null) innerGraphSubject = graph.createSubject(subjectXri);
	
			// get or create entity subject
	
			Subject subject = graph.getTopLevelGraph().getSubject(subjectXri);
	
			if (subject == null) {
	
				subject = graph.getTopLevelGraph().createSubject(subjectXri);
	
				// map type
	
				URI entityType = entity.getType();
				if (entityType != null) {
	
					XRI3Segment typeXri = IdASUtil.entityIDToXri(entityType.toString());
					subject.createStatement(Constants.XRI_RDFTYPE, typeXri);
				}
	
				// map attributes?
	
				if (deep > 0) {
	
					Iterator attributes = entity.getAttributes();
	
					while (attributes.hasNext()) {
	
						MapUtil.mapAttribute((IAttribute) attributes.next(), subject, deep - 1);
					}
				}
			}
			
			return subject;
		}
	}

	private static boolean hasSingleSimpleValueWithStringType(IAttribute attribute) throws IdASException {

		Iterator values = attribute.getValues();

		IValue value = (IValue) values.next();
		if (! (value instanceof ISimpleValue)) return(false);
		if (values.hasNext()) return(false);
		if (! (value.getType().equals(ITypedValue.STRING_TYPE_URI))) return(false);

		return(true);
	}

	private static boolean hasOnlyEntityValues(IAttribute attribute) throws IdASException {

		Iterator values = attribute.getValues();

		while (values.hasNext()) {

			IValue value = (IValue) values.next();

			if (! (value instanceof IEntity)) return(false);
		}

		return(true);
	}
}
