/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.net.URI;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.higgins.as.util.Constants;
import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.as.util.IdASHinUtil;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.idas.api.ITypedValue;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.xdi4j.Literal;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.constants.TypeConstants;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class AttributeSubjectPredicateLiteralResourceHandler extends AbstractResourceHandler {

	private XRI3Segment hin;
	private URI entityType;
	private URI dataType;
	private URI attrId;
	private String simpleValueData;

	private IContext context;
	private IAttribute parentAttribute;
	private IValue value;
	private IAttribute attribute;
	private ISimpleValue simpleValue;

	public AttributeSubjectPredicateLiteralResourceHandler(Message message, Subject subject, Predicate predicate, Literal literal, int deep, IContext context, IAttribute parentAttribute) {

		super(message, subject, predicate, literal);

		this.context = context;
		this.parentAttribute = parentAttribute;

		this.hin = subject.getSubjectXri();
		this.attrId = IdASUtil.xriToAttrID(predicate.getPredicateXri());
		this.simpleValueData = literal.getData();

		// read additional information from the message

		Predicate typePredicate = subject.getPredicate(Constants.XRI_RDFTYPE);
		if (typePredicate != null && typePredicate.containsReferences()) {

			this.entityType = URI.create(IdASUtil.xriToEntityID(typePredicate.getReference().getReferenceXri()));
		}

		Predicate dataTypePredicate = subject.getPredicate(Constants.XRI_DATATYPE);
		if (dataTypePredicate != null && dataTypePredicate.containsReferences()) {

			this.dataType = URI.create(IdASUtil.xriToEntityID(dataTypePredicate.getReference().getReferenceXri()));
		}
	}

	@Override
	public boolean executeAdd(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext) executionContext);

			// if we are a simple value: create simple value
			// if we are an entity value: create entity and attribute and simple value

			if (this.operationPredicate.getPredicateXri().equals(TypeConstants.XRI_VALUE.toString())) {

				if (this.dataType != null)
					this.value = this.parentAttribute.addSimpleValue(this.dataType, this.simpleValueData);
				else
					this.value = this.parentAttribute.addSimpleValue(ITypedValue.STRING_TYPE_URI, this.simpleValueData);
				this.context.applyUpdates();
				((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
				((IdASContextExecutionContext) executionContext).getHinCache().put(this.hin, this.value);
			} else {

				if (this.value == null) {

					this.value = this.context.addEntity(this.entityType, null);
					this.parentAttribute.addValue(this.value);
				}
				if (this.attribute == null) this.attribute = this.value.addAttribute(this.attrId);
				this.simpleValue = this.attribute.addSimpleValue(ITypedValue.STRING_TYPE_URI, this.simpleValueData);
				this.context.applyUpdates();
				((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
				((IdASContextExecutionContext) executionContext).getHinCache().put(this.hin, this.value);
			}
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@Override
	public boolean executeMod(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext) executionContext);
			if (this.value == null) throw new RuntimeException("Value " + this.hin.toString() + " not found.");
			if (this.value instanceof ISimpleValue && ! this.operationPredicate.getPredicateXri().equals(Constants.XRI_VALUE)) throw new RuntimeException("Simple value " + this.hin + " can not have attributes.");
			if (this.value instanceof IEntity && this.attribute == null) throw new RuntimeException("Entity value " + this.hin + " does not have attribute " + this.attrId.toString() + ".");
			if (this.value instanceof IEntity && ! (this.attribute.getValues().next() instanceof ISimpleValue)) throw new RuntimeException("Entity value " + this.hin + " has attribute " + this.attrId.toString() + ", but no simple value to modify.");

			// if it's a simple value: modify simple value
			// if it's an entity value: modify entity value and attribute and entity value attribute

			if (this.value instanceof ISimpleValue) {

				((ISimpleValue) this.value).setData(this.simpleValueData);
				this.context.applyUpdates();
				((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
			} else if (this.value instanceof IEntity) {

				this.simpleValue = (ISimpleValue) this.attribute.getValues().next();
				this.simpleValue.setData(this.simpleValueData);
				this.context.applyUpdates();
				((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
			}
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@SuppressWarnings("unchecked")
	private void lookup(IdASContextExecutionContext executionContext) throws Exception {

		this.value = null;
		this.attribute = null;
		this.simpleValue = null;

		// if we are a simple value: find simple value
		// if we are an entity value: find entity value and attribute and simple value

		Map<XRI3Segment, IValue> hinCache = executionContext.getHinCache();

		if (hinCache.containsKey(this.hin)) {

			this.value = hinCache.get(this.hin);
		} else {

			String algorithm = IdASHinUtil.segmentToAlgorithm(this.hin);
			if (algorithm == null) throw new Exception("Invalid HIN: " + this.hin);

			Iterator i = this.parentAttribute.getValues();

			if (i != null) while (i.hasNext()) {

				this.value = (IValue) i.next();

				if (this.hin.equals(IdASHinUtil.makeHin(this.value, algorithm))) break; else this.value = null;
			}

			if (this.value != null) hinCache.put(this.hin, this.value);
		}

		if (this.value == null) return;

		if (this.value instanceof ISimpleValue) {

			this.attribute = null;
		} else if (this.value instanceof IEntity) {

			this.attribute = ((IEntity) this.value).getAttribute(this.attrId);
			if (this.attribute == null) return;

			Iterator i = this.attribute.getValues();

			if (i != null) while (i.hasNext()) {

				IValue value = (IValue) i.next();
				if (! (value instanceof ISimpleValue)) continue;

				if (this.simpleValueData.equals(((ISimpleValue) value).getLexical())) {

					this.simpleValue = (ISimpleValue) value;
					return;
				}
			}
		}
	}
}
