/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.net.URI;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.as.util.IdASHinUtil;
import org.eclipse.higgins.as.util.MapUtil;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IValue;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.ISimpleValue;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.constants.TypeConstants;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class AttributeSubjectPredicateResourceHandler extends AbstractResourceHandler {

	private XRI3Segment hin;
	private URI attrId;
	private int deep;

	private IContext context;
	private IAttribute parentAttribute;
	private IValue value;
	private IAttribute attribute;

	public AttributeSubjectPredicateResourceHandler(Message message, Subject subject, Predicate predicate, int deep, IContext context, IAttribute parentAttribute) {

		super(message, subject, predicate);

		this.hin = subject.getSubjectXri();
		this.attrId = IdASUtil.xriToAttrID(predicate.getPredicateXri());
		this.deep = deep;

		this.context = context;
		this.parentAttribute = parentAttribute;
	}

	@Override
	public boolean executeGet(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext)executionContext);
			if (this.value == null) throw new RuntimeException("Value " + this.hin.toString() + " not found.");
			if (this.value instanceof ISimpleValue && ! this.attrId.equals(URI.create(TypeConstants.XRI_VALUE.toString()))) throw new RuntimeException("Simple value " + this.hin + " does not have attribute " + this.attrId.toString() + ".");
			if (this.value instanceof IEntity && this.attribute == null) throw new RuntimeException("Entity value " + this.hin + " does not have attribute " + this.attrId.toString() + ".");

			// if we are a simple value: map simple value
			// if we are an entity value: map entity value and attribute

			Graph graph = messageResult.getGraph();

			if (this.value instanceof ISimpleValue) {

				MapUtil.mapMultiSimpleValue(((ISimpleValue) this.value), graph);
			} else if (this.value instanceof IEntity) {

				Subject subject = MapUtil.mapMultiEntityValue(((IEntity) this.value), graph, 0);
				MapUtil.mapAttribute(this.attribute, subject, this.deep);
			}
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public boolean executeDel(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup((IdASContextExecutionContext)executionContext);
			if (this.value == null) throw new RuntimeException("Value " + this.hin.toString() + " not found.");
			if (this.value instanceof ISimpleValue && ! this.attrId.equals(URI.create(TypeConstants.XRI_VALUE.toString()))) throw new RuntimeException("Simple value " + this.hin + " does not have attribute " + this.attrId.toString() + ".");
			if (this.value instanceof IEntity && this.attribute == null) throw new RuntimeException("Entity value " + this.hin + " does not have attribute " + this.attrId.toString() + ".");

			// if we are a simple value: delete simple value
			// if we are an entity value: delete attribute of entity value

			if (this.value instanceof ISimpleValue) {

				this.value.remove();
				this.context.applyUpdates();
				((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
			} else if (this.value instanceof IEntity) {

				this.attribute.remove();
				this.context.applyUpdates();
				((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
			}
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@SuppressWarnings("unchecked")
	private void lookup(IdASContextExecutionContext executionContext) throws Exception {

		this.value = null;
		this.attribute = null;

		// if we are a simple value: find simple value
		// if we are an entity value: find entity value and attribute

		Map<XRI3Segment, IValue> hinCache = executionContext.getHinCache();

		if (hinCache.containsKey(this.hin)) {

			this.value = hinCache.get(this.hin);
		} else {

			String algorithm = IdASHinUtil.segmentToAlgorithm(this.hin);
			if (algorithm == null) throw new Exception("Invalid HIN: " + this.hin);

			Iterator i = this.parentAttribute.getValues();

			if (i != null) while (i.hasNext()) {

				this.value = (IValue) i.next();

				if (this.hin.equals(IdASHinUtil.makeHin(this.value, algorithm))) break; else this.value = null;
			}

			if (this.value != null) hinCache.put(this.hin, this.value);
		}

		if (this.value == null) return;

		if (this.value instanceof ISimpleValue) {

			this.attribute = null;
		} else if (this.value instanceof IEntity) {

			this.attribute = ((IEntity) this.value).getAttribute(this.attrId);
		}

		if (this.attribute == null) return;
	}
}
