/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.util.Iterator;

import org.eclipse.higgins.as.util.Constants;
import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.as.util.MapUtil;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.IdASException;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;
import org.eclipse.higgins.xdi4j.xri3.impl.XRI3Segment;

public class GraphResourceHandler extends AbstractResourceHandler {

	private int deep;

	private IContext context;
	private Iterator<IEntity> entities;

	GraphResourceHandler(Message message, Graph operationGraph, IContext context) {

		super(message, operationGraph);
		
		this.context = context;

		// read additional information from the message

		Predicate deepPredicate = message.getSubject().getPredicate(Constants.XRI_DEEP);
		if (deepPredicate != null && deepPredicate.containsLiteral()) {

			this.deep = Integer.parseInt(deepPredicate.getLiteral().getData());
		}
	}

	@Override
	public boolean executeGet(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup();
			if (this.entities == null) throw new RuntimeException("Entities not found.");
			
			// map entities

			while (this.entities.hasNext()) {
				
				IEntity entity = (IEntity) this.entities.next();

				MapUtil.mapEntity(entity, messageResult.getGraph(), this.deep);
			}
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@SuppressWarnings("unchecked")
	@Override
	public boolean executeDel(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			// clear context

			for (Iterator i = this.context.getEntities((String) null); i.hasNext(); ) {

				IEntity entity = (IEntity) i.next();
				entity.remove();
			}

			this.context.applyUpdates();
		} catch (IdASException ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}

		return(true);
	}

	@Override
	public boolean executeDo(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		if (operation.getOperationXri().equals(new XRI3Segment("$do$close"))) {

			try {

				this.context.applyUpdates();
				this.context.close();
			} catch (Exception ex) {

				throw ExceptionUtil.convertIdASException(ex);
			}

			return(true);
		}

		return(false);
	}

	@SuppressWarnings("unchecked")
	private void lookup() throws Exception {

		this.entities = null;

		// list entities

		this.entities = (Iterator<IEntity>) this.context.getEntities((String) null);
		if (this.entities == null) return;
	}
}
