/*******************************************************************************
 * Copyright (c) 2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Markus Sabadello - Initial API and implementation
 *******************************************************************************/
package org.eclipse.higgins.as;

import java.net.URI;

import org.eclipse.higgins.as.util.Constants;
import org.eclipse.higgins.as.util.ExceptionUtil;
import org.eclipse.higgins.as.util.MapUtil;
import org.eclipse.higgins.idas.api.IAttribute;
import org.eclipse.higgins.idas.api.IContext;
import org.eclipse.higgins.idas.api.IEntity;
import org.eclipse.higgins.idas.api.NoSuchEntityException;
import org.eclipse.higgins.xdi4j.Graph;
import org.eclipse.higgins.xdi4j.Predicate;
import org.eclipse.higgins.xdi4j.Subject;
import org.eclipse.higgins.xdi4j.exceptions.MessagingException;
import org.eclipse.higgins.xdi4j.idas.IdASUtil;
import org.eclipse.higgins.xdi4j.messaging.Message;
import org.eclipse.higgins.xdi4j.messaging.MessageResult;
import org.eclipse.higgins.xdi4j.messaging.Operation;
import org.eclipse.higgins.xdi4j.messaging.server.MessagingTarget;
import org.eclipse.higgins.xdi4j.messaging.server.impl.AbstractResourceHandler;

public class SubjectPredicateInnerGraphResourceHandler extends AbstractResourceHandler {

	private String entityId;
	private URI entityType;
	private URI attrId;
	private int deep;

	private IContext context;
	private IEntity entity;
	private IAttribute attribute;

	SubjectPredicateInnerGraphResourceHandler(Message message, Subject subject, Predicate predicate, Graph innerGraph, IContext context) {

		super(message, subject, predicate, innerGraph);

		this.context = context;

		this.entityId = IdASUtil.xriToEntityID(subject.getSubjectXri());
		this.attrId = IdASUtil.xriToAttrID(predicate.getPredicateXri());

		// read additional information from the message

		Predicate typePredicate = subject.getPredicate(Constants.XRI_RDFTYPE);
		if (typePredicate != null && typePredicate.containsReferences()) {

			this.entityType = URI.create(IdASUtil.xriToEntityID(typePredicate.getReference().getReferenceXri()));
		}

		Predicate deepPredicate = message.getSubject().getPredicate(Constants.XRI_DEEP);
		if (deepPredicate != null && deepPredicate.containsLiteral()) {

			this.deep = Integer.parseInt(deepPredicate.getLiteral().getData());
		}
	}

	@Override
	public boolean executeGet(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		try {

			this.lookup();
			if (this.entity == null) throw new RuntimeException("Entity " + this.entityId + " not found.");
			if (this.attribute == null) throw new RuntimeException("Attribute " + this.attrId.toString() + " on entity " + entityId + " not found.");

			// map entity and attribute and values

			Graph graph = messageResult.getGraph();

			Subject subject = MapUtil.mapEntity(this.entity, graph, 0);
			Predicate predicate = MapUtil.mapAttribute(this.attribute, subject, 0);
			MapUtil.mapMultiValue(null, predicate, this.deep);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public boolean executeAdd(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		if (this.operationPredicate.getPredicateXri().equals(Constants.XRI_RDFTYPE)) return(true);

		try {

			this.lookup();

			// create entity and attribute

			if (this.entity == null) this.entity = this.context.addEntity(this.entityType, this.entityId);
			if (this.attribute == null) this.attribute = this.entity.addAttribute(this.attrId);
			this.context.applyUpdates();
			((IdASContextExecutionContext) executionContext).setIncreaseVersion(true);
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public boolean executeMod(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		if (this.operationPredicate.getPredicateXri().equals(Constants.XRI_RDFTYPE)) return(true);

		try {

			this.lookup();
			if (this.entity == null) throw new RuntimeException("Entity " + this.entityId + " not found.");
			if (this.attribute == null) throw new RuntimeException("Attribute " + this.attrId.toString() + " on entity " + entityId + " not found.");
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public boolean executeDel(Operation operation, MessageResult messageResult, Object executionContext) throws MessagingException {

		if (this.operationPredicate.getPredicateXri().equals(Constants.XRI_RDFTYPE)) return(true);

		try {

			this.lookup();
			if (this.entity == null) throw new RuntimeException("Entity " + this.entityId + " not found.");
			if (this.attribute == null) throw new RuntimeException("Attribute " + this.attrId.toString() + " on entity " + entityId + " not found.");
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
		
		return(true);
	}

	@Override
	public MessagingTarget getInnerMessagingTarget(Object executionContext) throws MessagingException {

		try {

			// return MessagingTarget that represents this attribute

			return(new AttributeMessagingTarget(this.deep - 1, this.context, this.attribute, executionContext));
		} catch (Exception ex) {

			throw ExceptionUtil.convertIdASException(ex);
		}
	}

	private void lookup() throws Exception {

		this.entity = null;
		this.attribute = null;

		// find entity and attribute

		try { this.entity = this.context.getEntity(this.entityId); } catch (NoSuchEntityException ex) { this.entity = null; }
		if (this.entity != null && this.entity.isProxy()) this.entity = null;
		if (this.entity == null) return;

		this.attribute = this.entity.getAttribute(this.attrId);
		if (this.attribute == null) return;
	}
}
