/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.common;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.InvalidValueCountException;
import org.eclipse.higgins.icard.InvalidValueException;
import org.eclipse.higgins.icard.ReadOnlyObjectException;

public class ClaimValue implements ISimpleClaim {
	protected ISimpleClaimType type_;

	protected ArrayList values_ = new ArrayList(1);
	
	protected ICard card_ = null;

	public ClaimValue(IClaimType type, List values, ICard card) throws CardException {
		if(!(type instanceof ISimpleClaimType)) {
			throw new CardException("Parameter \"type\": " + type);
		}
		ISimpleClaimType stype = (ISimpleClaimType) type;
		if (stype.getMaxCardinality() != 1) {
			throw new InvalidTypeException("Invalid cardinality: type=" + stype.getType() + " cardinality=" + stype.getMaxCardinality());
		}
		if (!stype.getDataType().isAssignableFrom(String.class)) {
			throw new InvalidTypeException("Invalid data type class: type=" + stype.getType() + " dataType=" + stype.getDataType());
		}
		type_ = stype;
		setValues(values);
		card_ = card;
	}
	
	public ClaimValue(ISimpleClaimType type, String value) throws CardException {
		this(type, value, null);
	}
	
	/**
	 * @param type
	 * @param value
	 * @throws CardException
	 */
	public ClaimValue(ISimpleClaimType type, String value, ICard card) throws CardException {
		if (type == null)
			throw new CardException("Parameter \"type\" is null.");
		if (type.getMaxCardinality() != 1) {
			throw new InvalidTypeException("Invalid cardinality: type=" + type.getType() + " cardinality=" + type.getMaxCardinality());
		}
		if (!type.getDataType().isAssignableFrom(String.class)) {
			throw new InvalidTypeException("Invalid data type class: type=" + type.getType() + " dataType=" + type.getDataType());
		}
		card_ = card;
		type_ = type;
//		if (value == null)
//			throw new CardException("Parameter \"value\" is null.");
//		value_ = value;
		setValue(value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.IClaim#getType()
	 */
	public IClaimType getType() {
		return type_;
	}

	public String getValue() {
		return (String) values_.get(0);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.IClaim#getTypeLocalName()
	 */
	public String getTypeLocalName() {
		return getType().getTypeLocalName();
	}

	protected void setValue(String value) {
		if (value == null) {
			value = "";
		}
		if (values_.size() == 0)
			values_.add(value);
		else
			values_.set(0, value);
	}

	public List getValues() {
		return Collections.unmodifiableList(values_);
	}

	public void setValues(List newValues) throws InvalidValueException, InvalidStateException, ReadOnlyObjectException, InvalidValueCountException {
		if (card_ != null) {
			if (!card_.isEditable()) {
				throw new ReadOnlyObjectException();
			}
			if (!card_.isEditMode()) {
				throw new InvalidStateException();
			}
		}
		if (newValues.size() != 1) {
			throw new InvalidValueCountException();
		}
		Object o = newValues.get(0);
		if (!(o instanceof String)) {
			throw new InvalidValueException("Invalid value: " + o);
		}
		setValue((String) o); 
		if (card_ != null) {
			try {
				card_.setClaim(this);
			}
			catch(Exception e) {
				throw new InvalidStateException(e);
			}
		}
	}

}
