/*******************************************************************************
 * Copyright (c) 2007-2008 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - Initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.common.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xml.serializer.DOMSerializer;
import org.apache.xml.serializer.OutputPropertiesFactory;
import org.apache.xml.serializer.Serializer;
import org.apache.xml.serializer.SerializerFactory;
import org.eclipse.higgins.icard.common.io.IOElement;
import org.eclipse.higgins.icard.io.IElement;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Used to help with common XML processing functions
 */
public class XMLHelper
{
	private static final Log log = LogFactory.getLog(XMLHelper.class);
	
	public static final String IC_NS = "http://schemas.xmlsoap.org/ws/2005/05/identity";

	public static final String TRUST_NS = "http://schemas.xmlsoap.org/ws/2005/02/trust";

	public static final String WSA_NS = "http://www.w3.org/2005/08/addressing";

	public static final String MEX_NS = "http://schemas.xmlsoap.org/ws/2004/09/mex";

	public static final String WSAI_NS = "http://schemas.xmlsoap.org/ws/2006/02/addressingidentity";

	public static final String DSIG_NS = "http://www.w3.org/2000/09/xmldsig#";
	
	public static final String XE_NS = "http://www.w3.org/2001/04/xmlenc#";
	
	public static final String HI_NS = "http://www.eclipse.org/higgins/icard";
	
	/**
	* Converts an OMElement to an equivalent IOElement
	*
	* @param omElement an OMElement to be converted into a IOElement
	* @return the converted IOElement
	*/
	public static IElement toElement(OMElement omElement) throws Exception
	{
		IElement element = new IOElement();
		element.set(omElement);
		return element;
	}

	/**
	* Converts an DOM Element to an equivalent Element
	*
	* @param domElement a DOM Element to be converted into a Element
	* @return the converted Element
	*/
	public static IElement toElement(Element domElement) throws Exception
	{
		IElement element = new IOElement();
		element.set(domElement);
		return element;
	}

	/**
	* Converts an OMElement to an equivalent DOM Element
	*
	* @param omElement an OMElement to be converted into a DOM Element
	* @return the converted DOM Element
	*/
	public static Element toDOM(OMElement omElement) throws Exception
	{
		if (null == omElement) {
			return null;
		}		
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		try
		{
			omElement.serialize(outputStream);
		}
		catch (Exception e)
		{
			log.error(e);
			return null;
		}
		if (0 == outputStream.size())
		{
			return null;
		}
		ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());
		DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
		documentBuilderFactory.setNamespaceAware(true);
		DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
		Document domDocument = documentBuilder.parse(inputStream);
		return domDocument.getDocumentElement();
	}
	
	/**
	* Converts an OMElement to an equivalent String
	*
	* @param omElement an OMElement to be converted into a String
	* @return the converted a String
	*/
	public static String toString(OMElement omElement) throws Exception
	{
		if (null == omElement) {
			return null;
		}
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		omElement.serialize(outputStream);
	    String strElement = null;
	    try
	    {
	    	strElement = outputStream.toString("UTF-8");
	    	strElement = XMLHelper.StripNamespaceDeclaration(strElement);
	    }
	    catch (UnsupportedEncodingException uee)
	    {
	    	log.error(uee);
			throw uee;
	    }
	    return strElement;
	}
	
	public static String StripNamespaceDeclaration(String strElement)
	{
		boolean bNamespaceDeclaration = strElement.startsWith("<?xml");
		if (!bNamespaceDeclaration) {
			return strElement;
		}
		String strResult = strElement.substring(1);
		int nBegin = strResult.indexOf("<");
		return strResult.substring(nBegin);
	}

	/**
	* Converts an DOM Element into an equivalent OMElement
	*
	* @param domElement an Element to be converted
	* @return the converted DOM Element
	*/
	public static OMElement toOM(Element domElement) throws Exception
	{
		if (null == domElement) {
			return null;
		}
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        Properties prop = OutputPropertiesFactory.getDefaultMethodProperties("xml");
        Serializer ser = org.apache.xml.serializer.SerializerFactory.getSerializer(prop);
        ser.setOutputFormat(prop);
        ser.setOutputStream(outputStream);
        DOMSerializer domSerializer = ser.asDOMSerializer();
        domSerializer.serialize(domElement);
		ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());
        XMLInputFactory xif = XMLInputFactory.newInstance();
        XMLStreamReader reader = xif.createXMLStreamReader(inputStream);
        StAXOMBuilder builder = new StAXOMBuilder(reader);
        return builder.getDocumentElement();
	}
	
	public static String toString(Element domElement) throws IOException
	{
		if (null == domElement) {
			return null;
		}
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        Properties prop = OutputPropertiesFactory.getDefaultMethodProperties("xml");
        Serializer ser = SerializerFactory.getSerializer(prop);
        ser.setOutputFormat(prop);
        ser.setOutputStream(outputStream);
        DOMSerializer domSerializer = ser.asDOMSerializer();
        domSerializer.serialize(domElement);
	    String strElement = null;
	    try
	    {
	    	strElement = outputStream.toString("UTF-8");
	    	strElement = XMLHelper.StripNamespaceDeclaration(strElement);
	    }
	    catch (UnsupportedEncodingException uee)
	    {
	    	log.error(uee);
			throw uee;
	    }
	    return strElement;
	}
	
	public static void logElement(Element domElement) throws IOException
	{
		String strElement = XMLHelper.toString(domElement);
		XMLHelper.log.trace(strElement);
	}
	
	public static Element toDOM(String strElement) throws IOException, Exception
	{
		if (null == strElement) {
			return null;
		}
	    ByteArrayInputStream inputStream = new ByteArrayInputStream(strElement.getBytes());
	    DocumentBuilder documentBuilder = null;
	    DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
	    documentBuilderFactory.setNamespaceAware(true);
	    try
	    {
	    	documentBuilder = documentBuilderFactory.newDocumentBuilder();
	    }
	    catch (final Exception e)
	    {
	    	log.error(e);
			throw e;
	    }
	    Document docElement = null;
	    try
	    {
	    	docElement = documentBuilder.parse(inputStream);
	    }
	    catch (final Exception e)
	    {
	    	log.error(e);
			throw e;
	    }
	    return docElement.getDocumentElement();
	}

	public static Element domFromStream(InputStream inputStream) throws FileNotFoundException, Exception
	{
	    DocumentBuilder documentBuilder = null;
	    DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
	    documentBuilderFactory.setNamespaceAware(true);
	    try
	    {
	    	documentBuilder = documentBuilderFactory.newDocumentBuilder();
	    }
	    catch (final Exception e)
	    {
	    	log.error(e);
			throw e;
	    }
	    Document docElement = null;
	    try
	    {
	    	docElement = documentBuilder.parse(inputStream);
	    }
	    catch (final Exception e)
	    {
	    	log.error(e);
			throw e;
	    }
	    return docElement.getDocumentElement();
	}
	
	public static Element domFromFile(String strFileName) throws FileNotFoundException, Exception
	{
	    FileInputStream inputStream = null;
	    try
	    {
	    	inputStream = new FileInputStream(strFileName);
	    }
	    catch (FileNotFoundException fnfe)
	    {
	    	log.error(fnfe);
			throw fnfe;
	    }
	    return XMLHelper.domFromStream(inputStream);
	}
	
	public static OMElement omFromStream(InputStream inputStream) throws FileNotFoundException, Exception
	{
        XMLInputFactory xif = XMLInputFactory.newInstance();
        XMLStreamReader reader = xif.createXMLStreamReader(inputStream);
        StAXOMBuilder builder = new StAXOMBuilder(reader);
        return builder.getDocumentElement();
	}
	
	public static OMElement omFromFile(String strFileName) throws FileNotFoundException, Exception
	{
	    FileInputStream inputStream = null;
	    try
	    {
	    	inputStream = new FileInputStream(strFileName);
	    }
	    catch (FileNotFoundException fnfe)
	    {
	    	log.error(fnfe);
			throw fnfe;
	    }
	    return XMLHelper.omFromStream(inputStream);
	}
	
	public static OMElement toOM(String strElement) throws Exception
	{
	    ByteArrayInputStream inputStream = new ByteArrayInputStream(strElement.getBytes());
	    XMLInputFactory xif = XMLInputFactory.newInstance();
	    XMLStreamReader reader = xif.createXMLStreamReader(inputStream);
	    StAXOMBuilder builder = new StAXOMBuilder(reader);
	    return builder.getDocumentElement();
	}
	
}
