/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.common;

import java.io.ByteArrayInputStream;
import java.io.StringReader;
import java.net.URI;

import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.IEndpointReference;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardContext;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

public class EndpointReference implements IEndpointReference {
	private Log log = LogFactory.getLog(EndpointReference.class);

	protected URI address_ = null;

	protected X509Certificate certificate_ = null;

	protected URI metadataAddress_ = null;

	protected Element metadata_ = null;

	protected Element identity_ = null;

	public EndpointReference(URI address, String metadata, String identity) throws CardException {
		if (address == null)
			throw new IllegalArgumentException("Parameter \"address\" is null.");
		address_ = address;
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setIgnoringComments(true);
		dbf.setIgnoringElementContentWhitespace(true);
		dbf.setNamespaceAware(true);
		dbf.setValidating(false);
		try {
			DocumentBuilder db = dbf.newDocumentBuilder();
			if (metadata != null) {
				StringReader sisMetadata = new StringReader(metadata);
				InputSource isMetadata = new InputSource(sisMetadata);
				Document mdDoc = db.parse(isMetadata);
				metadata_ = mdDoc.getDocumentElement();
			}
			if (identity != null) {
				StringReader sisIdentity = new StringReader(identity);
				InputSource isIdentity = new InputSource(sisIdentity);
				Document idntDoc = db.parse(isIdentity);
				identity_ = idntDoc.getDocumentElement();
			}
			parseMetadataAndCertificate();
		}
		catch(Exception e) {
			log.error(e);
			throw new CardException(e);
		}
	}

	private void parseMetadataAndCertificate() throws CardException {
		try {
			if (metadata_ != null) {
				String mdAddr = null;
				try {
					Element metadata = XMLUtils.getChildElement(metadata_, CardContext.MEX_NS, "Metadata");
					Element metadataSection = XMLUtils.getChildElement(metadata, CardContext.MEX_NS, "MetadataSection");
					Element metadataReference = XMLUtils.getChildElement(metadataSection, CardContext.MEX_NS, "MetadataReference");
					Element address = XMLUtils.getChildElement(metadataReference, CardContext.WSA_NS, "Address");
					mdAddr = address.getFirstChild().getNodeValue();
				}
				catch(Exception e) {
					log.error(e);
				}
				if (mdAddr != null) {
					metadataAddress_ = new URI(mdAddr);
				}
			}
			if (identity_ != null) {
				String certStr = null; 
				try {
					Element keyInfo = XMLUtils.getChildElement(identity_, CardContext.DSIG_NS, "KeyInfo");
					Element x509Data = XMLUtils.getChildElement(keyInfo, CardContext.DSIG_NS, "X509Data");
					Element x509Certificate = XMLUtils.getChildElement(x509Data, CardContext.DSIG_NS, "X509Certificate");
					certStr = x509Certificate.getFirstChild().getNodeValue();
				}
				catch(Exception e) {
					log.error(e);
				}
				if (certStr != null) {
					byte[] certBytes = CardCryptography.decodeBase64(certStr);
					ByteArrayInputStream bis = new ByteArrayInputStream(certBytes);
					CertificateFactory cf = CertificateFactory.getInstance("X.509");
					certificate_ = (X509Certificate) cf.generateCertificate(bis);
				}
			}
		}
		catch(Exception e) {
			log.error(e);
			throw new CardException(e);
		}
	}

	public EndpointReference(URI address, Element metadata, Element identity) throws CardException {
		if (address == null)
			throw new CardException("Parameter \"address\" is null.");
		address_ = address;
		metadata_ = metadata;
		identity_ = identity;
		parseMetadataAndCertificate();
	}

	public URI getAddress() {
		return address_;
	}

	public Element getMetadata() {
		return metadata_;

	}

	public Element getIdentity() {
		return identity_;
	}

	public X509Certificate getCertificate() {
		return certificate_;
	}

	public URI getMetadataAddress() {
		return metadataAddress_;
	}

}
