/*******************************************************************************
 * Copyright (c) 2007 Parity Communications, Inc.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Valery Kokhan - initial API and implementation
 *******************************************************************************/

package org.eclipse.higgins.icard.provider.cardspace.common;

import java.io.ByteArrayInputStream;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.security.auth.callback.CallbackHandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.AuthenticationException;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.IManagedInformationCard;
import org.eclipse.higgins.icard.ISimpleClaim;
import org.eclipse.higgins.icard.ISimpleClaimType;
import org.eclipse.higgins.icard.auth.ICredential;
import org.eclipse.higgins.icard.common.ClaimValue;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.iss.ICardSelectorService;
import org.eclipse.higgins.iss.IDisplayToken;
import org.eclipse.higgins.iss.IICardSelector;
import org.eclipse.higgins.iss.IIdentityToken;
import org.eclipse.higgins.iss.SelectionANDofORs;
import org.eclipse.higgins.iss.SelectionANDofORsElm;
import org.w3c.dom.Element;

public abstract class ManagedCardProvider implements ICardProvider {
	protected Log log = LogFactory.getLog(ManagedCardProvider.class);

	//	protected static String strCertificate = "MIIDHjCCAtwCBEcw9YAwCwYHKoZIzjgEAwUAMHUxCzAJBgNVBAYTAlVBMQswCQYDVQQIEwJETjEXMBUGA1UEBxMORG5lcHJvcGV0cm92c2sxDzANBgNVBAoTBlBhcml0eTEXMBUGA1UECxMOUGFyaXR5IFVrcmFpbmUxFjAUBgNVBAMTDVZhbGVyeSBLb2toYW4wHhcNMDcxMTA2MjMxNTEyWhcNMDgwMjA0MjMxNTEyWjB1MQswCQYDVQQGEwJVQTELMAkGA1UECBMCRE4xFzAVBgNVBAcTDkRuZXByb3BldHJvdnNrMQ8wDQYDVQQKEwZQYXJpdHkxFzAVBgNVBAsTDlBhcml0eSBVa3JhaW5lMRYwFAYDVQQDEw1WYWxlcnkgS29raGFuMIIBtzCCASwGByqGSM44BAEwggEfAoGBAP1/U4EddRIpUt9KnC7s5Of2EbdSPO9EAMMeP4C2USZpRV1AIlH7WT2NWPq/xfW6MPbLm1Vs14E7gB00b/JmYLdrmVClpJ+f6AR7ECLCT7up1/63xhv4O1fnxqimFQ8E+4P208UewwI1VBNaFpEy9nXzrith1yrv8iIDGZ3RSAHHAhUAl2BQjxUjC8yykrmCouuEC/BYHPUCgYEA9+GghdabPd7LvKtcNrhXuXmUr7v6OuqC+VdMCz0HgmdRWVeOutRZT+ZxBxCBgLRJFnEj6EwoFhO3zwkyjMim4TwWeotUfI0o4KOuHiuzpnWRbqN/C/ohNWLx+2J6ASQ7zKTxvqhRkImog9/hWuWfBpKLZl6Ae1UlZAFMO/7PSSoDgYQAAoGALQOE9rSSxFhktVM7whlol8WhgNpU1ZAKuJF0PYBK8fBHMH/+6RFpqrd44DyiPQ1vwtFzHaOLCNh0Q1u8aRJc3DwgX62v/4VU6U+1HHmU9JsLWxFGZgk2xCOcU3UJzE0HaQji3TvR3Punl2vdRbLjy/XWnxpcIF4yGb+wbeS2hoowCwYHKoZIzjgEAwUAAy8AMCwCFB9FoKbmWzoxjD4C0QmxbBAVY0FNAhQA3LbUiH6NU7FNuv1vhRI7OxQdkQ==";
	protected static String strCertificate = "MIICVDCCAb2gAwIBAgIERWUHrzANBgkqhkiG9w0BAQUFADBQMQswCQYDVQQGEwJVUzEYMBYGA1UEChMPSUJNIENvcnBvcmF0aW9uMRUwEwYDVQQLEwxJQk0gUmVzZWFyY2gxEDAOBgNVBAMTB1Jvb3QgQ0EwHhcNMDYxMTIzMDIyOTU3WhcNMjYxMTE4MDIyOTU3WjBFMQswCQYDVQQGEwJVUzEQMA4GA1UEChMHRWNsaXBzZTEQMA4GA1UECxMHSGlnZ2luczESMBAGA1UEAxMJbG9jYWxob3N0MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCUFDtM2/nLEhSJqF7zqayW8gQ183KCiKME53VkSjl6C7O9J2hEiBMWsI9a+qBm13VWKvmTc3h3EsLaR/dOreXyCLKEwDIpwHojW3+i037OOxL6U3Ij094rqfHBCJP2xEyLNpPXdIioEgP5H6MTRPXShsBsTQ7bEhZs0QKyXwTNHwIDAQABo0YwRDAdBgNVHQ4EFgQUTHM9O3KViNYsvuGeg4eANMCFHuMwEwYDVR0lBAwwCgYIKwYBBQUHAwEwDgYDVR0PAQH/BAQDAgSwMA0GCSqGSIb3DQEBBQUAA4GBADkZZ34T637gwxp/EmRlRd4ru1Y2ribGoLRpY1icJYlyrUB3552f1K7r69EQxuwnyUnFYVhB+4ZnzhralkKtNs8UVo7RZHSE8jKWOi4YU3RXSqEHKs10ttQLLcHPuH4o9t2OJLeaVtaijbdicbxUEwqhgcNnNrQHe6+f5fgwzSYM";
	protected X509Certificate certificate = null;

	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid, ICredential userCredential) throws AuthenticationException, CardException {
		ICard card = getICardByCUID(authHandler, cuid);
		if (card instanceof ManagedCard) {
			ManagedCard mcard = (ManagedCard) card;
			CardSpacePolicy p = new CardSpacePolicy();
			p.setRequiredClaims(mcard.getSupportedClaimTypesUris());
			try {
				ICardSelectorService iss = ICardSelectorService.getInstance();
				IICardSelector s = iss.getICardSelector(authHandler, p);
				SelectionANDofORs selection = new SelectionANDofORs();
				selection.add(new SelectionANDofORsElm(mcard.getCUID().toString(), 0, 0));
				selection.setCredential(userCredential);
				selection.sslCert = getCertificate();
				
				List claims = new LinkedList();
				
				IIdentityToken t = s.getIdentityToken(selection);
//				System.err.println(t.getAs(String.class));
				t.getAs(String.class);
				IDisplayToken dt = (IDisplayToken) t.getAs(IDisplayToken.class);
				System.err.println(dt.getAs(String.class));
				for (Iterator itr = mcard.getSupportedClaimTypes(); itr.hasNext(); ) {
					ISimpleClaimType ct = (ISimpleClaimType) itr.next();
					ISimpleClaim c = (ISimpleClaim) dt.getClaim(ct.getType());
					if (c == null) {
						c = new ClaimValue(ct, "");
					}
					claims.add(c);
				}
				mcard.setClaims(claims);
				return mcard;
			} catch (Exception e) {
				throw new CardException(e);
			}
			
		} else {
			throw new CardException("Unsupported card type: " + card);
		}
	}

	/**
	 * Managed card provider doesn't support card creation. So, just return <code>null</code>
	 */
	public ICardTemplate[] getCardCreationTemplates(CallbackHandler authHandler) {
		return null;
	}

	/**
	 * The only supported type is {@link IManagedInformationCard}
	 */
	public Class[] getSupportedTypes() {
		return new Class[] {IManagedInformationCard.class};
	}

	protected X509Certificate getCertificate() {
		if (certificate == null) {
			try {
				CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
				String strEncryptionCertificate = "-----BEGIN CERTIFICATE-----\n"
						+ strCertificate + "\n-----END CERTIFICATE-----\n";
				ByteArrayInputStream streamCertificate = new ByteArrayInputStream(strEncryptionCertificate.getBytes());
				certificate = (java.security.cert.X509Certificate) certificateFactory.generateCertificate(streamCertificate);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return certificate;
	}

	public boolean canImportICard(CallbackHandler authHandler, IElement element) {
		if (element == null)
			return false;
		try {
			Element card = (Element) element.getAs(Element.class);
			if (card == null) {
				return false;
			}
			Element crd = null;
			if ("Signature".equals(card.getLocalName()))
				crd = CardCryptography.getCardFromSignedEnvelop(card);
			else
				crd = card;
			DummyManagedCard c = new DummyManagedCard();
			c.initFromXML(crd);
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			log.error(e);
		}
		return false;
	}
	
	
}
