/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.icard.provider.cardspace.entity.mysql;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Map;
import java.util.StringTokenizer;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;

public class ConnectionFactory {
	private static Log log = LogFactory.getLog(ConnectionFactory.class);

	public final static String JNDI_CONTEXT = "jndi.context";

	public final static String SOURCE_NAME = "source.name";

	public final static String DRIVER_CLASS_NAME = "driver.classname";

	public final static String DB_USER_NAME = "db.username";

	public final static String DB_PASSWORD = "db.password";

	public final static String DB_URL = "db.url";

	public final static String USE_POOL = "db.pool";

	protected String dbUserName;

	protected String dbPassword;

	protected String dbURL;

	protected String jndiContext;
	
	protected boolean isPoolUsed;

	protected DataSource dataSource;

	protected String driverName;

	protected String userName;

	protected String password;

	protected String url;

	protected String contextName;

	protected String sourceName;

	/**
	 * Factory for direct (non-pooled) connections
	 * 
	 * @param driverName
	 * @param userName
	 * @param password
	 * @param url
	 * @throws CardException
	 * @throws ClassNotFoundException
	 */
	public ConnectionFactory(String driverName, String userName, String password, String url) throws CardException {
		log.debug("driverName = " + driverName + ", userName = " + userName + ", password = " + password + ", url = "
		        + url);
		this.isPoolUsed = false;
		this.driverName = driverName;
		this.userName = userName;
		this.password = password;
		this.url = url;
		try {
			Class.forName(driverName);
			initDatabase();
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} catch (ClassNotFoundException e) {
			log.error(e, e);
			throw new CardException(e);
		}
	}

	/**
	 * Factory for pooled connections
	 * 
	 * @param driverName
	 * @param userName
	 * @param password
	 * @param url
	 * @param contextName
	 * @param sourceName
	 * @throws NamingException
	 * @throws CardException
	 */
	public ConnectionFactory(String contextName, String sourceName) throws CardException {
		log.debug("context name = " + contextName + ",dataSource name " + sourceName);
		this.isPoolUsed = true;
		this.contextName = contextName;
		this.sourceName = sourceName;
		try {
			Context init = new InitialContext();
			Context ctx = (Context) init.lookup(contextName);
			if (ctx == null)
				throw new CardException("Can not find context by name = " + contextName);
			dataSource = (DataSource) ctx.lookup(sourceName);
			if (dataSource == null)
				throw new CardException("Can not find DataSource by name = " + sourceName);
			initDatabase();
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} catch (NamingException e) {
			log.error(e, e);
			throw new CardException(e);
		}
	}

	private synchronized void initDatabase() throws CardException, SQLException {
		Connection con = null;
		ResultSet rs = null;
		try {
			con = getConnection();
			String[] types = {"TABLE"};
			rs = con.getMetaData().getTables(null, null, "ManagedCard", types);
			if (!rs.next()) {
				createDbStructure(con);
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (con != null && !con.isClosed()) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private ArrayList getCreateStatements() throws IOException {
		InputStream is = null;
		StringBuffer sb = new StringBuffer();
		try {
			is = ClassLoader.getSystemResourceAsStream("mysql.sql");
			BufferedReader br = new BufferedReader(new InputStreamReader(is));
			String line = br.readLine();
			while (line != null) {
				sb.append(line);
				line = br.readLine();
			}
		}
		finally {
			if (is != null) {
				is.close();
			}
		}
		ArrayList statemens = new ArrayList();
		String all = sb.toString();
		StringTokenizer st = new StringTokenizer(all, ";");
		while(st.hasMoreTokens()) {
			statemens.add(st.nextToken());
		}
		return statemens;
	}

	private void createDbStructure(Connection con) throws SQLException, IOException {
		ArrayList sts = getCreateStatements();
		Statement st = null;
		try {
			st = con.createStatement();
			for (int i = 0, s = sts.size(); i < s; i++) {
				String sql = (String)sts.get(i);
				st.addBatch(sql);
			}
			st.executeBatch();
		} finally {
			try {
				if (st != null) {
					st.close();
					st = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	public Connection getConnection() throws SQLException {
		Connection con = null;
		if (isPoolUsed)
			con = dataSource.getConnection();
		else
			con = DriverManager.getConnection(url, userName, password);
		con.setAutoCommit(false);
		return con;
	}

	public static ConnectionFactory buildConnectionFactory(Map config) throws CardException {
		String useConnectionPool = config.containsKey(USE_POOL) ? (String) config.get(USE_POOL) : null;
		boolean isPoolUsed = ("yes".equalsIgnoreCase(useConnectionPool)) ? true : false;
		if (isPoolUsed) {
			String jndiContext = config.containsKey(JNDI_CONTEXT) ? (String) config.get(JNDI_CONTEXT) : null;
			String sourceName = config.containsKey(SOURCE_NAME) ? (String) config.get(SOURCE_NAME) : null;
			if (jndiContext == null)
				throw new CardException("Property \"" + JNDI_CONTEXT + "\" not found.");
			if (sourceName == null)
				throw new CardException("Property \"" + SOURCE_NAME + "\" not found.");
			return new ConnectionFactory(jndiContext, sourceName);
		} else {
			String driverName = config.containsKey(DRIVER_CLASS_NAME) ? (String) config.get(DRIVER_CLASS_NAME) : "com.mysql.jdbc.Driver";
			String dbUserName = config.containsKey(DB_USER_NAME) ? (String) config.get(DB_USER_NAME) : null;
			String dbPassword = config.containsKey(DB_PASSWORD) ? (String) config.get(DB_PASSWORD) : "";
			String dbURL = config.containsKey(DB_URL) ? (String) config.get(DB_URL) : null;
			if (dbUserName == null)
				throw new CardException("Property \"" + DB_USER_NAME + "\" not found.");
			if (dbURL == null)
				throw new CardException("Property \"" + DB_URL + "\" not found.");
			return new ConnectionFactory(driverName, dbUserName, dbPassword, dbURL);
		}
	}

	
}
