/**
 * Copyright (c) 2009 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.icard.provider.cardspace.entity.mysql;

import java.io.IOException;
import java.math.BigDecimal;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IEndpointReference;
import org.eclipse.higgins.icard.ITokenService;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.provider.cardspace.common.CredentialDescriptor;
import org.eclipse.higgins.icard.provider.cardspace.common.EndpointReference;
import org.eclipse.higgins.icard.provider.cardspace.common.STSPrivacyPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.TokenService;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.CardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.Extension;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.MCardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.XMLUtils;
import org.xml.sax.SAXException;

public class MCardHelper {
	private static Log log = LogFactory.getLog(MCardHelper.class);

	private static final long maxMySQLDate_ = 253402253999000l;

	public ArrayList getMCards(Connection con, String userId) throws CardException {
		ArrayList list = new ArrayList();
		String query = "SELECT * FROM ManagedCard WHERE userID_ = ? ORDER BY id_";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, userId);
			rs = ps.executeQuery();
			while (rs.next()) {
				MCardEntity mc = new MCardEntity();
				initMCard(mc, rs, con);
				list.add(mc);
			}
			return list;
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	public static void initMCard(MCardEntity card, ResultSet rs, Connection con) throws Exception {
		initCard(card, rs, con);
		int id = ((Integer) card.getId()).intValue();
		card.setSupportedTokenTypeList(initMCardSupportedTokenTypes(id, con));
		card.setSupportedClaimTypeList(initMCardClaimTypes(id, con));
		card.setExtensionList(initMCardExtensionList(id, con));
		BigDecimal rat = rs.getBigDecimal("requireAppliesTo_");
		if (rat != null) {
			card.setRequireAppliesTo((rat.intValue() == 0) ? Boolean.TRUE : Boolean.FALSE);
		} else {
			card.setRequireAppliesTo(null);
		}
		initStsPolicy(card, rs);
		card.setTokenServiceList(getTokenServiceList(id, con));
	}

	private static void initStsPolicy(MCardEntity card, ResultSet rs) throws URISyntaxException, SQLException, CardException {
		String policyStr = rs.getString("privacyNotice_");
		if (policyStr != null) {
			STSPrivacyPolicy sp = new STSPrivacyPolicy(policyStr);
			card.setPolicyURL(new URI(sp.getPrivacyUrl()));
			String version = sp.getPrivacyVersion();
			try {
				Integer versionInt = new Integer(version);
				card.setPolicyVersion(versionInt);
			} catch (NumberFormatException e) {
				log.error(e, e);
			}
		}
	}

	private static void initCard(CardEntity card, ResultSet rs, Connection con) throws SQLException, URISyntaxException {
		int rowId = rs.getInt("id_");
		card.setId(new Integer(rowId));
		String cardID = rs.getString("cardID_");
		card.setCardID(new URI(cardID));
		card.setLanguage(rs.getString("language_"));
		card.setName(rs.getString("name_"));
		card.setVersion(rs.getInt("version_"));
		card.setIssuer(new URI(rs.getString("issuer_")));
		card.setIssuerID(rs.getBytes("issuerID_"));
		card.setIssuerName(rs.getString("issuerName_"));
		card.setImage(rs.getBytes("image_"));
		card.setImageType(rs.getString("imageType_"));
		card.setTimeIssued(rs.getDate("timeIssued_"));
		card.setTimeExpires(rs.getDate("timeExpires_"));
		card.setTimeLastUpdated(rs.getDate("timeLastUpdated_"));
		card.setHashSalt(rs.getBytes("hashSalt_"));
		card.setMasterKey(rs.getBytes("masterKey_"));
	}

	private static ArrayList initMCardSupportedTokenTypes(int cardRowId, Connection con) throws SQLException, URISyntaxException {
		String query = "SELECT stt.id_, stt.tokenTypeID_, tt.type_ FROM SupportedTokenType AS stt, TokenType AS tt WHERE stt.cardID_ = ? AND stt.tokenTypeID_ = tt.id_ ORDER BY stt.id_";
		ArrayList supportedTokenTypeList = new ArrayList();
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			rs = ps.executeQuery();
			while (rs.next()) {
				supportedTokenTypeList.add(new URI(rs.getString("type_")));
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
		return supportedTokenTypeList;
	}

	private static ArrayList initMCardExtensionList(int cardRowId, Connection con) throws SQLException {
		String query = "SELECT * FROM MCardExtension WHERE cardID_ = ?";
		ArrayList extensionList = new ArrayList();
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			rs = ps.executeQuery();
			while (rs.next()) {
				Extension ext = new Extension();
				ext.setId(new Integer(rs.getInt("id_")));
				ext.setExtensionData(rs.getString("element_"));
				ext.setEnabled(rs.getBoolean("enabled_"));
				extensionList.add(ext);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
		return extensionList;
	}

	private static ArrayList initMCardClaimTypes(int cardRowId, Connection con) throws SQLException, CardException {
		String query = "SELECT sct.id_, sct.claimTypeID_, ct.type_ , ct.displayName_ , ct.description_ FROM SupportedClaimType AS sct, ClaimType AS ct WHERE sct.cardID_ = ? AND sct.claimTypeID_ = ct.id_ ORDER BY sct.id_";
		ArrayList claimTypes = new ArrayList();
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			rs = ps.executeQuery();
			while (rs.next()) {
				String type = rs.getString("type_");
				String description = rs.getString("description_");
				String displayName = rs.getString("displayName_");
				boolean isPPID = SelfIssuedCardClaims.PPID_TYPE.equals(type);
				ClaimType ct = new ClaimType(type, displayName, description, !isPPID, !isPPID);
				claimTypes.add(ct);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
		return claimTypes;
	}

	private static ArrayList getTokenServiceList(int cardRowId, Connection con) throws SQLException, CardException, URISyntaxException,
			IllegalArgumentException, ParserConfigurationException, SAXException, IOException {
		ArrayList tokenServiceList = new ArrayList();
		String query = "SELECT * FROM TokenService WHERE cardID_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			rs = ps.executeQuery();
			while (rs.next()) {
				String endpointAddress = rs.getString("endpointAddress_");
				String hint = rs.getString("hint_");
				String identity = rs.getString("identity_");
				String credential = rs.getString("credential_");
				String metadata = rs.getString("metadata_");
				URI epAddress = new URI(endpointAddress);
				EndpointReference endpoint = new EndpointReference(epAddress, metadata, identity);
				CredentialDescriptor credentialDescriptor = new CredentialDescriptor(hint, credential);
				TokenService ts = new TokenService(endpoint, credentialDescriptor);
				tokenServiceList.add(ts);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
		return tokenServiceList;
	}

	private static BigDecimal boolToInt(Boolean bool) {
		if (bool != null)
			return (bool.booleanValue()) ? new BigDecimal(0) : new BigDecimal(1);
		else
			return null;
	}

	private static String getPrivacyPolicy(MCardEntity card) throws Exception {
		URI uri = card.getPolicyURL();
		if (uri == null)
			return null;
		Integer ver = card.getPolicyVersion();
		String version = (ver != null) ? ver.toString() : null;
		STSPrivacyPolicy spp = new STSPrivacyPolicy(uri.toString(), version);
		return (String) spp.getPrivacyElement().getAs(String.class);
	}

	private static java.sql.Date convertDate(Date date) {
		if (date == null)
			return null;
		if (date.getTime() > maxMySQLDate_) {
			return new java.sql.Date(maxMySQLDate_);
		}
		if (date instanceof java.sql.Date) {
			return (java.sql.Date) date;
		}
		return new java.sql.Date(date.getTime());
	}

	public static void saveMCard(MCardEntity card, Connection con, String userId) throws Exception {
		Integer id = null;
		String query = "INSERT INTO ManagedCard (cardID_, language_, name_, version_, issuer_, issuerName_, image_, imageType_, timeIssued_, timeExpires_, timeLastUpdated_, hashSalt_, masterKey_, requireAppliesTo_, privacyNotice_, userID_, issuerID_) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setString(1, card.getCardID().toString());
			ps.setString(2, card.getLanguage());
			ps.setString(3, card.getName());
			ps.setInt(4, card.getVersion());
			URI issuer = card.getIssuer();
			ps.setString(5, (issuer != null) ? issuer.toString() : null);
			ps.setString(6, card.getIssuerName());
			ps.setBytes(7, card.getImage());
			ps.setString(8, card.getImageType());
			ps.setDate(9, convertDate(card.getTimeIssued()));
			ps.setDate(10, convertDate(card.getTimeExpires()));
			ps.setDate(11, convertDate(card.getTimeLastUpdated()));
			ps.setBytes(12, card.getHashSalt());
			ps.setBytes(13, card.getMasterKey());
			ps.setBigDecimal(14, boolToInt(card.getRequireAppliesTo()));
			ps.setString(15, getPrivacyPolicy(card));
			ps.setString(16, userId);
			ps.setBytes(17, card.getIssuerID());
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next()) {
				id = new Integer(rs.getInt(1));
			} else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
		saveMCardTokenServiceList(id.intValue(), con, card.getTokenServiceList());
		saveMCardSupportedTokenTypeList(id.intValue(), con, card.getSupportedTokenTypeList());
		saveMCardSupportedClaimTypeList(id.intValue(), con, card.getSupportedClaimTypeList());
		insertMCardExtensionList(id.intValue(), con, card.getExtensionList());
		card.setId(id);
	}

	public static void saveMCardTokenServiceList(int cardId, Connection con, List tokenServiceList) throws CardException,
			TransformerException {
		if (tokenServiceList == null || tokenServiceList.size() == 0) {
			return;
		}
		String query = "INSERT INTO TokenService (cardID_, endpointAddress_, hint_, identity_, credential_, metadata_) VALUES (?, ?, ?, ?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardId);
			for (int i = 0; i < tokenServiceList.size(); i++) {
				ITokenService ts = (ITokenService) tokenServiceList.get(i);
				IEndpointReference er = ts.getEndpointReference();

				ICredentialDescriptor cd = ts.getUserCredential();
				ps.setString(2, er.getAddress().toString());
				ps.setString(3, cd.getDisplayCredentialHint());
				ps.setString(4, XMLUtils.elementToString(er.getIdentity()));
				ps.setString(5, XMLUtils.elementToString(cd.asXML()));
				ps.setString(6, XMLUtils.elementToString(er.getMetadata()));
				ps.execute();
			}
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}

	}

	private static int getTokenType(Connection con, String tokenType) throws CardException {
		String query = "SELECT id_ FROM TokenType WHERE type_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, tokenType);
			rs = ps.executeQuery();
			if (rs.next()) {
				int id = rs.getInt("id_");
				return id;
			} else
				return -1;
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static int insertTokenType(Connection con, String tokenType) throws CardException {
		String query = "INSERT INTO TokenType (type_) VALUES (?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setString(1, tokenType);
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next()) {
				int id = rs.getInt(1);
				return id;
			} else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static int initTokenType(Connection con, String tokenType) throws CardException {
		int tokenTypeID = getTokenType(con, tokenType);
		if (tokenTypeID == -1)
			tokenTypeID = insertTokenType(con, tokenType);
		return tokenTypeID;
	}

	public static void saveMCardSupportedTokenTypeList(int cardId, Connection con, List tokenList) throws CardException {
		if (tokenList == null || tokenList.size() == 0) {
			return;
		}
		String query = "INSERT INTO SupportedTokenType (cardID_, tokenTypeID_) VALUES (?, ?)";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardId);
			for (int i = 0; i < tokenList.size(); i++) {
				URI tokenType = (URI) tokenList.get(i);
				int tokenTypeID = initTokenType(con, tokenType.toString());
				ps.setInt(2, tokenTypeID);
				ps.execute();
			}
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	public static void saveMCardSupportedClaimTypeList(int cardId, Connection con, List claimTypeList) throws CardException {
		if (claimTypeList == null || claimTypeList.size() == 0) {
			return;
		}
		String query = "INSERT INTO SupportedClaimType (cardID_, claimTypeID_) VALUES (?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardId);
			for (int i = 0; i < claimTypeList.size(); i++) {
				IClaimType ct = (IClaimType) claimTypeList.get(i);
				int claimTypeID = initClaimType(con, ct);
				ps.setInt(2, claimTypeID);
				ps.execute();
			}
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static int initClaimType(Connection con, IClaimType claimType) throws CardException {
		int tokenTypeID = getClaimType(con, claimType);
		if (tokenTypeID == -1)
			tokenTypeID = insertClaimType(con, claimType);
		return tokenTypeID;
	}

	private static int getClaimType(Connection con, IClaimType claimType) throws CardException {
		String query = "SELECT id_ FROM ClaimType WHERE type_ = ? AND typeLocalName_ = ? AND displayName_ = ? AND description_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, claimType.getType());
			ps.setString(2, claimType.getTypeLocalName());
			ps.setString(3, claimType.getDisplayName());
			ps.setString(4, claimType.getDescription());
			rs = ps.executeQuery();
			if (rs.next()) {
				int id = rs.getInt("id_");
				return id;
			} else
				return -1;
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static int insertClaimType(Connection con, IClaimType claimType) throws CardException {
		String query = "INSERT INTO ClaimType (type_, typeLocalName_, displayName_, description_) VALUES (?, ?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setString(1, claimType.getType());
			ps.setString(2, claimType.getTypeLocalName());
			ps.setString(3, claimType.getDisplayName());
			ps.setString(4, claimType.getDescription());
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next()) {
				int id = rs.getInt(1);
				return id;
			} else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateMCardName(int cardId, Connection con, String newName, Date lastUpdated) throws SQLException {
		String query = "UPDATE ManagedCard SET name_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, newName);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateMCardImage(int cardId, Connection con, byte[] newImage, Date lastUpdated) throws SQLException {
		String query = "UPDATE ManagedCard SET image_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setBytes(1, newImage);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateMCardImageType(int cardId, Connection con, String newImageType, Date lastUpdated) throws SQLException {
		String query = "UPDATE ManagedCard SET imageType_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, newImageType);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateMCardIssuerName(int cardId, Connection con, String newIssuerName, Date lastUpdated) throws SQLException {
		String query = "UPDATE ManagedCard SET issuerName_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, newIssuerName);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static boolean areStringsEqual(String s1, String s2) {
		return !((s1 == null && s2 != null) || (s1 != null && s2 == null) || !s1.equals(s2));
	}

	public static void updateMCard(MCardEntity card, Connection con) throws CardException, SQLException {
		updateMCardCommonFields(card, con);
		updateMCardExtensions(card, con);
	}

	private static void updateMCardCommonFields(MCardEntity card, Connection con) throws CardException {
		Integer id = (Integer) card.getId();
		Date lastUpdated = new Date();
		String query = "SELECT name_, image_, imageType_, issuerName_ FROM ManagedCard WHERE id_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id.intValue());
			rs = ps.executeQuery();
			if (rs.next()) {
				String oldCardName = rs.getString("name_");
				byte[] oldImage = rs.getBytes("image_");
				String oldImageType = rs.getString("imageType_");
				String oldIssuerName = rs.getString("issuerName_");
				String newCardName = card.getName();
				String newIssuerName = card.getIssuerName();
				byte[] newImage = card.getImage();
				String newImageType = card.getImageType();
				if (!areStringsEqual(oldCardName, newCardName)) {
					updateMCardName(id.intValue(), con, newCardName, lastUpdated);
				}
				if (!areStringsEqual(oldIssuerName, newIssuerName)) {
					updateMCardIssuerName(id.intValue(), con, newIssuerName, lastUpdated);
				}
				if (!Arrays.equals(oldImage, newImage)) {
					updateMCardImage(id.intValue(), con, newImage, lastUpdated);
				}
				if (!areStringsEqual(oldImageType, newImageType)) {
					updateMCardImageType(id.intValue(), con, newImageType, lastUpdated);
				}
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static Extension getExtensionById(Object id, List extList) {
		if (id == null) {
			return null;
		}
		for (int i = 0, s = extList.size(); i < s; i++) {
			Extension ext = (Extension) extList.get(i);
			if (id.equals(ext.getId())) {
				return ext;
			}
		}
		return null;
	}

	private static List getExtensionForInsert(List oldExt, List newExt) {
		if (newExt == null || newExt.size() == 0) {
			return Collections.EMPTY_LIST;
		}
		if (oldExt == null || oldExt.size() == 0) {
			return newExt;
		}
		ArrayList forInsert = new ArrayList();
		for (int i = 0; i < newExt.size(); i++) {
			Extension ext = (Extension) newExt.get(i);
			Object newId = ext.getId();
			if (newId == null || getExtensionById(newId, oldExt) == null) {
				forInsert.add(ext);
			}
		}
		return forInsert;
	}

	private static List getExtensionForUpdate(List oldExtList, List newExtList) {
		if (newExtList == null || newExtList.size() == 0 || oldExtList == null || oldExtList.size() == 0) {
			return Collections.EMPTY_LIST;
		}
		ArrayList forUpdate = new ArrayList();
		for (int i = 0; i < newExtList.size(); i++) {
			Extension newExt = (Extension) newExtList.get(i);
			Object newId = newExt.getId();
			if (newId != null) {
				Extension oldExt = getExtensionById(newId, oldExtList);
				if (!areStringsEqual(newExt.getExtensionData(), oldExt.getExtensionData()) || (newExt.isEnabled() != oldExt.isEnabled())) {
					forUpdate.add(newExt);
				}
			}
		}
		return forUpdate;
	}

	private static List getExtensionForDelete(List oldExtList, List newExtList) {
		if (oldExtList == null || oldExtList.size() == 0) {
			return Collections.EMPTY_LIST;
		}
		ArrayList forDelete = new ArrayList();
		for (int i = 0; i < oldExtList.size(); i++) {
			Extension oldExt = (Extension) oldExtList.get(i);
			Object oldId = oldExt.getId();
			if (getExtensionById(oldId, newExtList) == null) {
				forDelete.add(oldExt);
			}
		}
		return forDelete;
	}

	private static void updateMCardExtensions(MCardEntity card, Connection con) throws SQLException, CardException {
		Integer id = (Integer) card.getId();
		int cardRowId = id.intValue();
		List oldExtList = initMCardExtensionList(cardRowId, con);
		List newExtensionList = card.getExtensionList();
		insertMCardExtensionList(cardRowId, con, getExtensionForInsert(oldExtList, newExtensionList));
		updateMCardExtensionList(con, getExtensionForUpdate(oldExtList, newExtensionList));
		deleteMCardExtensionList(con, getExtensionForDelete(oldExtList, newExtensionList));
	}

	private static void insertMCardExtensionList(int cardRowId, Connection con, List extensions) throws SQLException, CardException {
		if (extensions == null || extensions.size() == 0) {
			return;
		}
		String query = "INSERT INTO MCardExtension (cardID_, element_, enabled_) VALUES (?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setInt(1, cardRowId);
			for (int i = 0, size = extensions.size(); i < size; i++) {
				Extension ext = (Extension) extensions.get(i);
				ps.setString(2, ext.getExtensionData());
				ps.setBoolean(3, ext.isEnabled());
				ps.execute();
				rs = ps.getGeneratedKeys();
				if (rs.next()) {
					int id = rs.getInt(1);
					ext.setId(new Integer(id));
				} else
					throw new CardException("Prepared statement did not return primary key for inserted row.");
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateMCardExtensionList(Connection con, List extensions) throws SQLException {
		if (extensions == null || extensions.size() == 0) {
			return;
		}
		String query = "UPDATE MCardExtension SET element_ = ?, enabled_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			for (int i = 0, size = extensions.size(); i < size; i++) {
				Extension ext = (Extension) extensions.get(i);
				Integer id = (Integer) ext.getId();
				ps = con.prepareStatement(query);
				ps.setString(1, ext.getExtensionData());
				ps.setBoolean(2, ext.isEnabled());
				ps.setInt(3, id.intValue());
				ps.execute();
			}
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteMCardExtensionList(Connection con, List extensions) throws SQLException {
		if (extensions == null || extensions.size() == 0) {
			return;
		}
		String query = "DELETE FROM MCardExtension WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			for (int i = 0, size = extensions.size(); i < size; i++) {
				Extension ext = (Extension) extensions.get(i);
				Integer id = (Integer) ext.getId();
				ps = con.prepareStatement(query);
				ps.setInt(1, id.intValue());
				ps.execute();
			}
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	public static void deleteMCard(MCardEntity card, Connection con) throws SQLException, CardException {
		Integer id = (Integer)card.getId();
		if (id == null) {
			throw new CardException("Card entity contains no id.");
		}
		deleteMCardRow(id.intValue(), con);
		deleteTokenService(id.intValue(), con);
		deleteExtensions(id.intValue(), con);
		deleteClaimTypes(id.intValue(), con);
		deleteTokenTypes(id.intValue(), con);
	}

	private static void deleteMCardRow(int cardRowId, Connection con) throws SQLException {
		String query = "DELETE FROM ManagedCard WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteTokenService(int cardRowId, Connection con) throws SQLException {
		String query = "DELETE FROM TokenService WHERE cardID_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteExtensions(int cardRowId, Connection con) throws SQLException {
		String query = "DELETE FROM MCardExtension WHERE cardID_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteTokenTypes(int cardRowId, Connection con) throws SQLException {
		String query = "DELETE FROM SupportedTokenType WHERE cardID_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteClaimTypes(int cardRowId, Connection con) throws SQLException {
		String query = "DELETE FROM SupportedClaimType WHERE cardID_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

}
