/**
 * Copyright (c) 2009 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */
package org.eclipse.higgins.icard.provider.cardspace.entity.mysql;

import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.CardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.Extension;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.PCardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;

public class PCardHelper {
	private static Log log = LogFactory.getLog(PCardHelper.class);

	private static final long maxMySQLDate = 253402253999000l;

	//
	private final static Hashtable claimTypeToClaimId;

	private final static Hashtable claimIdToClaimType;

	static {
		claimTypeToClaimId = SelfIssuedCardClaims.getSupportedTypesHash();
		claimIdToClaimType = new Hashtable();
		Enumeration keys = claimTypeToClaimId.keys();
		while (keys.hasMoreElements()) {
			Object key = keys.nextElement();
			Object val = claimTypeToClaimId.get(key);
			claimIdToClaimType.put(val, key);
		}
	}

	private static void initCard(CardEntity card, ResultSet rs, Connection con) throws SQLException, URISyntaxException {
		int rowId = rs.getInt("id_");
		card.setId(new Integer(rowId));
		String cardID = rs.getString("cardID_");
		card.setCardID(new URI(cardID));
		card.setLanguage(rs.getString("language_"));
		card.setName(rs.getString("name_"));
		card.setVersion(rs.getInt("version_"));
		card.setIssuerID(rs.getBytes("issuerID_"));
		card.setImage(rs.getBytes("image_"));
		card.setImageType(rs.getString("imageType_"));
		card.setTimeIssued(rs.getDate("timeIssued_"));
		card.setTimeExpires(rs.getDate("timeExpires_"));
		card.setTimeLastUpdated(rs.getDate("timeLastUpdated_"));
		card.setHashSalt(rs.getBytes("hashSalt_"));
		card.setMasterKey(rs.getBytes("masterKey_"));
	}

	public static void initPCard(PCardEntity card, ResultSet rs, Connection con) throws Exception {
		initCard(card, rs, con);
		card.setPinDigest(rs.getBytes("pinDigest_"));
		int id = ((Integer) card.getId()).intValue();
		card.setSupportedClaimTypeList(SelfIssuedCardClaims.getSupportedClaimTypeList());
		card.setRawClaims(getPCardClaims(id, con));
//		card.setSupportedTokenTypeList(initPCardSupportedTokenTypes(id, con));
//		card.setSupportedClaimTypeList(initPCardClaimTypes(id, con));
		card.setExtensionList(initPCardExtensionList(id, con));
	}

	private static HashMap getPCardClaims(int cardRowId, Connection con) throws CardException, SQLException {
		HashMap valuesMap = new HashMap();
		String query = "SELECT * FROM PersonalCardClaim WHERE cardID_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			rs = ps.executeQuery();
			while (rs.next()) {
				int typeIndex = rs.getInt("claimTypeID_");
				String value = rs.getString("claimValue_");
				String id = String.valueOf(typeIndex);
				String type;
				if (claimIdToClaimType.containsKey(id))
					type = (String) claimIdToClaimType.get(id);
				else
					throw new CardException("Can not find claim type in dictionary by id = " + id);
				valuesMap.put(type, value);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
		return valuesMap;
	}

	private static ArrayList initPCardExtensionList(int cardRowId, Connection con) throws SQLException {
		String query = "SELECT * FROM PCardExtension WHERE cardID_ = ?";
		ArrayList extensionList = new ArrayList();
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			rs = ps.executeQuery();
			while (rs.next()) {
				Extension ext = new Extension();
				ext.setId(new Integer(rs.getInt("id_")));
				ext.setExtensionData(rs.getString("element_"));
				ext.setEnabled(rs.getBoolean("enabled_"));
				extensionList.add(ext);
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
		return extensionList;
	}

	private static java.sql.Date convertDate(Date date) {
		if (date == null)
			return null;
		if (date.getTime() > maxMySQLDate) {
			return new java.sql.Date(maxMySQLDate);
		}
		if (date instanceof java.sql.Date) {
			return (java.sql.Date) date;
		}
		return new java.sql.Date(date.getTime());
	}

	public static void addPCard(PCardEntity card, Connection con, String userId) throws CardException, SQLException {
		String query = "INSERT INTO PersonalCard (cardID_, language_, name_, version_, image_, imageType_, timeIssued_, timeExpires_, timeLastUpdated_, hashSalt_, masterKey_, pinDigest_, userID_, issuerID_) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		Integer id = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setString(1, card.getCardID().toString());
			ps.setString(2, card.getLanguage());
			ps.setString(3, card.getName());
			ps.setInt(4, card.getVersion());
			ps.setBytes(5, card.getImage());
			ps.setString(6, card.getImageType());
			ps.setDate(7, convertDate(card.getTimeIssued()));
			ps.setDate(8, convertDate(card.getTimeExpires()));
			ps.setDate(9, convertDate(card.getTimeLastUpdated()));
			ps.setBytes(10, card.getHashSalt());
			ps.setBytes(11, card.getMasterKey());
			ps.setBytes(12, card.getPinDigest());
			ps.setString(13, userId);
			ps.setBytes(14, card.getIssuerID());
			ps.execute();
			rs = ps.getGeneratedKeys();
			if (rs.next())
				id = new Integer(rs.getInt(1));
			else
				throw new CardException("Prepared statement did not return primary key for inserted row.");
		} catch (SQLException e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
		insertClaims(id.intValue(), con, card.getRawClaims());
		insertExtensions(id.intValue(), con, card.getExtensionList());
		card.setId(id);
	}

	private static boolean areStringsEqual(String s1, String s2) {
		if (s1 == null && s2 == null) {
			return true;
		}
		return !((s1 == null && s2 != null) || (s1 != null && s2 == null) || !s1.equals(s2));
	}

	private static Extension getExtensionById(Object id, List extList) {
		if (id == null) {
			return null;
		}
		for (int i = 0, s = extList.size(); i < s; i++) {
			Extension ext = (Extension) extList.get(i);
			if (id.equals(ext.getId())) {
				return ext;
			}
		}
		return null;
	}

	private static List getExtensionForInsert(List oldExt, List newExt) {
		if (newExt == null || newExt.size() == 0) {
			return Collections.EMPTY_LIST;
		}
		if (oldExt == null || oldExt.size() == 0) {
			return newExt;
		}
		ArrayList forInsert = new ArrayList();
		for (int i = 0; i < newExt.size(); i++) {
			Extension ext = (Extension) newExt.get(i);
			Object newId = ext.getId();
			if (newId == null || getExtensionById(newId, oldExt) == null) {
				forInsert.add(ext);
			}
		}
		return forInsert;
	}

	private static List getExtensionForUpdate(List oldExtList, List newExtList) {
		if (newExtList == null || newExtList.size() == 0 || oldExtList == null || oldExtList.size() == 0) {
			return Collections.EMPTY_LIST;
		}
		ArrayList forUpdate = new ArrayList();
		for (int i = 0; i < newExtList.size(); i++) {
			Extension newExt = (Extension) newExtList.get(i);
			Object newId = newExt.getId();
			if (newId != null) {
				Extension oldExt = getExtensionById(newId, oldExtList);
				if (!areStringsEqual(newExt.getExtensionData(), oldExt.getExtensionData()) || (newExt.isEnabled() != oldExt.isEnabled())) {
					forUpdate.add(newExt);
				}
			}
		}
		return forUpdate;
	}

	private static List getExtensionForDelete(List oldExtList, List newExtList) {
		if (oldExtList == null || oldExtList.size() == 0) {
			return Collections.EMPTY_LIST;
		}
		ArrayList forDelete = new ArrayList();
		for (int i = 0; i < oldExtList.size(); i++) {
			Extension oldExt = (Extension) oldExtList.get(i);
			Object oldId = oldExt.getId();
			if (getExtensionById(oldId, newExtList) != null) {
				forDelete.add(oldExt);
			}
		}
		return forDelete;
	}

	private static void updatePCardExtensions(PCardEntity card, Connection con) throws SQLException, CardException {
		Integer id = (Integer) card.getId();
		int cardRowId = id.intValue();
		List oldExtList = initPCardExtensionList(cardRowId, con);
		List newExtensionList = card.getExtensionList();
		insertExtensions(cardRowId, con, getExtensionForInsert(oldExtList, newExtensionList));
		updateExtensions(con, getExtensionForUpdate(oldExtList, newExtensionList));
		deleteExtensions(con, getExtensionForDelete(oldExtList, newExtensionList));
	}

	private static void insertExtensions(int cardRowId, Connection con, List extensions) throws SQLException, CardException {
		if (extensions == null || extensions.size() == 0) {
			return;
		}
		String query = "INSERT INTO PCardExtension (cardID_, element_, enabled_) VALUES (?, ?, ?)";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
			ps.setInt(1, cardRowId);
			for (int i = 0, size = extensions.size(); i < size; i++) {
				Extension ext = (Extension) extensions.get(i);
				ps.setString(2, ext.getExtensionData());
				ps.setBoolean(3, ext.isEnabled());
				ps.execute();
				rs = ps.getGeneratedKeys();
				if (rs.next()) {
					int id = rs.getInt(1);
					ext.setId(new Integer(id));
				} else
					throw new CardException("Prepared statement did not return primary key for inserted row.");
			}
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateExtensions(Connection con, List extensions) throws SQLException {
		if (extensions == null || extensions.size() == 0) {
			return;
		}
		String query = "UPDATE PCardExtension SET element_ = ?, enabled_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			for (int i = 0, size = extensions.size(); i < size; i++) {
				Extension ext = (Extension) extensions.get(i);
				Integer id = (Integer) ext.getId();
				ps = con.prepareStatement(query);
				ps.setString(1, ext.getExtensionData());
				ps.setBoolean(2, ext.isEnabled());
				ps.setInt(3, id.intValue());
				ps.execute();
			}
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteExtensions(Connection con, List extensions) throws SQLException {
		if (extensions == null || extensions.size() == 0) {
			return;
		}
		String query = "DELETE FROM PCardExtension WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			for (int i = 0, size = extensions.size(); i < size; i++) {
				Extension ext = (Extension) extensions.get(i);
				Integer id = (Integer) ext.getId();
				ps = con.prepareStatement(query);
				ps.setInt(1, id.intValue());
				ps.execute();
			}
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	public static void updatePCard(PCardEntity card, Connection con) throws CardException, SQLException {
		updatePCardCommonFields(card, con);
		updatePCardExtensions(card, con);
		updatePCardClaims(card, con);
	}

	private static void updatePCardCommonFields(PCardEntity card, Connection con) throws CardException {
		Integer id = (Integer) card.getId();
		Date lastUpdated = new Date();
		String query = "SELECT name_, image_, imageType_, masterKey_, pinDigest_ FROM PersonalCard WHERE id_ = ?";
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, id.intValue());
			rs = ps.executeQuery();
			if (rs.next()) {
				String oldCardName = rs.getString("name_");
				byte[] oldImage = rs.getBytes("image_");
				String oldImageType = rs.getString("imageType_");
				byte[] oldMasterKey = rs.getBytes("masterKey_");
				byte[] oldPinDigest = rs.getBytes("pinDigest_");
				
				String newCardName = card.getName();
				byte[] newImage = card.getImage();
				String newImageType = card.getImageType();
				byte[] newMasterKey = card.getMasterKey();
				byte[] newPinDigest = card.getPinDigest();
				if (!areStringsEqual(oldCardName, newCardName)) {
					updateName(id.intValue(), con, newCardName, lastUpdated);
				}
				if (!Arrays.equals(oldImage, newImage)) {
					updateImage(id.intValue(), con, newImage, lastUpdated);
				}
				if (!areStringsEqual(oldImageType, newImageType)) {
					updateImageType(id.intValue(), con, newImageType, lastUpdated);
				}
				if (!Arrays.equals(oldMasterKey, newMasterKey)) {
					updateMasterKey(id.intValue(), con, newMasterKey, lastUpdated);
				}
				if (!Arrays.equals(oldPinDigest, newPinDigest)) {
					updatePinDigest(id.intValue(), con, newPinDigest, lastUpdated);
				}
			}
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e);
		} finally {
			try {
				if (rs != null) {
					rs.close();
					rs = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateMasterKey(int cardId, Connection con, byte[] newMasterKey, Date lastUpdated) throws SQLException {
		String query = "UPDATE PersonalCard SET masterKey_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setBytes(1, newMasterKey);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}
	
	private static void updatePinDigest(int cardId, Connection con, byte[] newPinDigest, Date lastUpdated) throws SQLException {
		String query = "UPDATE PersonalCard SET pinDigest_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setBytes(1, newPinDigest);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}
	
	private static void updateName(int cardId, Connection con, String newName, Date lastUpdated) throws SQLException {
		String query = "UPDATE PersonalCard SET name_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, newName);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateImage(int cardId, Connection con, byte[] newImage, Date lastUpdated) throws SQLException {
		String query = "UPDATE PersonalCard SET image_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setBytes(1, newImage);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateImageType(int cardId, Connection con, String newImageType, Date lastUpdated) throws SQLException {
		String query = "UPDATE PersonalCard SET imageType_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setString(1, newImageType);
			ps.setDate(2, convertDate(lastUpdated));
			ps.setInt(3, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

//	private static void updatePCardIssuerName(int cardId, Connection con, String newIssuerName, Date lastUpdated) throws SQLException {
//		String query = "UPDATE PersonalCard SET issuerName_ = ?, timeLastUpdated_ = ? WHERE id_ = ?";
//		PreparedStatement ps = null;
//		try {
//			ps = con.prepareStatement(query);
//			ps.setString(1, newIssuerName);
//			ps.setDate(2, convertDate(lastUpdated));
//			ps.setInt(3, cardId);
//			ps.execute();
//		} finally {
//			try {
//				if (ps != null) {
//					ps.close();
//					ps = null;
//				}
//			} catch (SQLException e) {
//				log.error(e, e);
//			}
//		}
//	}


	private static Map getClaimsForInsert(Map oldClaims, Map newClaims) {
		if (newClaims == null || newClaims.size() == 0) {
			return Collections.EMPTY_MAP;
		}
		HashMap forInsert = new HashMap();
		Iterator newClaimsItr = newClaims.keySet().iterator();
		while (newClaimsItr.hasNext()) {
			Object newClaimType = newClaimsItr.next();
			if (!oldClaims.containsKey(newClaimType)) {
				Object newClaimValue = newClaims.get(newClaimType);
				forInsert.put(newClaimType, newClaimValue);
			}
		}
		return forInsert;
	}

	private static Map getClaimsForUpdate(Map oldClaims, Map newClaims) {
		if (newClaims == null || newClaims.size() == 0 || oldClaims == null || oldClaims.size() == 0) {
			return Collections.EMPTY_MAP;
		}
		HashMap forUpdate = new HashMap();
		Iterator newClaimsItr = newClaims.keySet().iterator();
		while (newClaimsItr.hasNext()) {
			Object claimType = newClaimsItr.next();
			if (oldClaims.containsKey(claimType)) {
				String oldClaimValue = (String) oldClaims.get(claimType);
				String newClaimValue = (String) newClaims.get(claimType);
				if (!areStringsEqual(oldClaimValue, newClaimValue)) {
					forUpdate.put(claimType, newClaimValue);
				}
			}
		}
		return forUpdate;
	}

	private static List getClaimsForDelete(Map oldClaims, Map newClaims) {
		if (oldClaims == null || oldClaims.size() == 0) {
			return Collections.EMPTY_LIST;
		}
		ArrayList forDelete = new ArrayList();
		Iterator oldClaimsItr = oldClaims.keySet().iterator();
		while (oldClaimsItr.hasNext()) {
			Object claimType = oldClaimsItr.next();
			if (!newClaims.containsKey(claimType)) {
				forDelete.add(claimType);
			}
		}
		return forDelete;
	}

	private static void updatePCardClaims(PCardEntity card, Connection con) throws SQLException, CardException {
		Integer id = (Integer) card.getId();
		int cardRowId = id.intValue();
		Map oldClaimsMap = getPCardClaims(cardRowId, con);
		Map newClaimsMap = card.getRawClaims();
		insertClaims(cardRowId, con, getClaimsForInsert(oldClaimsMap, newClaimsMap));
		updateClaims(cardRowId, con, getClaimsForUpdate(oldClaimsMap, newClaimsMap));
		deleteClaims(cardRowId, con, getClaimsForDelete(oldClaimsMap, newClaimsMap));
	}
	
	private static String getClaimTypeId(String claimType) throws CardException {
		if (claimTypeToClaimId.containsKey(claimType)) {
			return (String) claimTypeToClaimId.get(claimType);
		} else
			throw new CardException("No claim id ");
	}

	private static void insertClaims(int cardRowId, Connection con, Map newClaims) throws SQLException, CardException {
		if (newClaims == null || newClaims.size() == 0) {
			return;
		}
		String query = "INSERT INTO PersonalCardClaim (cardID_, claimTypeID_, claimValue_) VALUES (?, ?, ?)";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			Iterator claimTypes = newClaims.keySet().iterator();
			while (claimTypes.hasNext()) {
				String claimType = (String) claimTypes.next();
				String claimTypeId = getClaimTypeId(claimType);
				String value = (String) newClaims.get(claimType);
				ps.setString(2, claimTypeId);
				ps.setString(3, value);
				ps.execute();
			}
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void updateClaims(int cardRowId, Connection con, Map updateClaims) throws SQLException, CardException {
		if (updateClaims == null || updateClaims.size() == 0) {
			return;
		}
		String query = "UPDATE PersonalCardClaim SET claimValue_ = ? WHERE claimTypeID_ = ? AND cardID_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(3, cardRowId);
			Iterator claimTypes = updateClaims.keySet().iterator();
			while (claimTypes.hasNext()) {
				String claimType = (String) claimTypes.next();
				String claimTypeId = getClaimTypeId(claimType);
				String value = (String) updateClaims.get(claimType);
				ps.setString(2, claimTypeId);
				ps.setString(1, value);
				ps.execute();
			}
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteClaims(int cardRowId, Connection con, List deleteClaimTypes) throws SQLException, CardException {
		if (deleteClaimTypes == null || deleteClaimTypes.size() == 0) {
			return;
		}
		String query = "DELETE FROM PersonalCardClaim WHERE cardID_ = ? AND claimTypeID_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			for (int i = 0, size = deleteClaimTypes.size(); i < size; i++) {
				String claimType = (String) deleteClaimTypes.get(i);
				String claimTypeId = getClaimTypeId(claimType);
				ps.setString(2, claimTypeId);
				ps.execute();
			}
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	public static void deletePCard(PCardEntity card, Connection con) throws CardException, SQLException {
		Integer id = (Integer) card.getId();
		if (id == null) {
			throw new CardException("Card entity contains no id.");
		}
		deletePCardRow(id.intValue(), con);
		deleteExtensions(id.intValue(), con);
		deleteClaimValues(id.intValue(), con);
	}

	private static void deletePCardRow(int cardRowId, Connection con) throws SQLException {
		String query = "DELETE FROM PersonalCard WHERE id_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteExtensions(int cardRowId, Connection con) throws SQLException {
		String query = "DELETE FROM PCardExtension WHERE cardID_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardRowId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private static void deleteClaimValues(int cardId, Connection con) throws SQLException {
		String query = "DELETE FROM PersonalCardClaim WHERE cardId_ = ?";
		PreparedStatement ps = null;
		try {
			ps = con.prepareStatement(query);
			ps.setInt(1, cardId);
			ps.execute();
		} finally {
			try {
				if (ps != null) {
					ps.close();
					ps = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

}
