/**
 * Copyright (c) 2009 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.entity.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.cache.CacheProviderFactory;
import org.eclipse.higgins.cache.api.ICache;
import org.eclipse.higgins.cache.api.key.CacheKey;
import org.eclipse.higgins.configuration.api.IConfigurableComponent;
import org.eclipse.higgins.configuration.api.ISettingDescriptor;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.IPCardManagedBean;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.IUserCredentials;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.PCardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.UsernamePasswordCredentials;

public class PCardManagedBean implements IPCardManagedBean, IConfigurableComponent { 

	private Log log = LogFactory.getLog(PCardManagedBean.class);

	private boolean isConfigured = false;

	private static ICache cardCache = CacheProviderFactory.getCacheProvider().getCache("pcard.jdbc");

	private static boolean isNoCache = true;

	private ConnectionFactory connectionFactory;

	public PCardManagedBean() {
	}

	public synchronized void configure(Map mapGlobalSettings, String strComponentName, Map mapComponentSettings,
			ISettingDescriptor componentDescriptor, ISettingDescriptor globalDescriptor) throws CardException {
		init(mapComponentSettings);
	}

	private void addCard(PCardEntity card, IUserCredentials credentials) throws CardException {
		CacheKey cacheKey = buildCacheKey(credentials);
		addCardToDB(card, credentials);
		addCardToCache(card, cacheKey);
	}

	private void addCardToDB(PCardEntity card, IUserCredentials credentials) throws CardException {
		String userId = buildUserId(credentials);
		Connection con = null;
		try {
			con = getConnection();
			PCardHelper.addPCard(card, con, userId);
			if (con.getAutoCommit() == false)
				con.commit();

		} catch (Exception e) {
			log.error(e, e);
			try {
				con.rollback();
			} catch (SQLException ex) {
				log.error(ex, ex);
				throw new CardException(ex);
			}
			throw new CardException(e);
		} finally {
			try {
				if (con != null) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private void addCardToCache(PCardEntity card, CacheKey cacheKey) throws CardException {
		if (!isNoCache) {
			HashMap cacheCards = getPCardsFromCache(cacheKey);
			if (cacheCards != null) {
				cacheCards.put(card.getCardID().toString(), card);
			}
		}
	}

	public void deletePCard(String cardID, IUserCredentials credentials) throws CardException {
		if (cardID == null)
			throw new CardException("Parameter \"cardID\" is null");
		if (credentials == null)
			throw new CardException("Parameter \"credentials\" is null");
		PCardEntity card = getPCard(cardID, credentials);
		removeCardFromDB(card, credentials);
		removeCardFromCache(card, credentials);
	}

	private void removeCardFromDB(PCardEntity card, IUserCredentials credentials) throws CardException {
		Connection con = null;
		try {
			con = getConnection();
			PCardHelper.deletePCard(card, con);
			if (con.getAutoCommit() == false)
				con.commit();

		} catch (Exception e) {
			log.error(e, e);
			try {
				con.rollback();
			} catch (SQLException ex) {
				log.error(ex, ex);
				throw new CardException(ex);
			}
			throw new CardException(e);
		} finally {
			try {
				if (con != null) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private void removeCardFromCache(PCardEntity card, IUserCredentials credentials) throws CardException {
		if (!isNoCache) {
			CacheKey cacheKey = buildCacheKey(credentials);
			HashMap cacheCards = getPCardsFromCache(cacheKey);
			String cardId = card.getCardID().toString();
			if (cacheCards != null && cacheCards.containsKey(cardId)) {
				cacheCards.remove(cardId);
			}
		}
	}

	public PCardEntity getPCard(String cardID, IUserCredentials credentials) throws CardException {
		if (cardID == null)
			throw new CardException("Parameter \"cardID\" is null");
		if (credentials == null)
			throw new CardException("Parameter \"credentials\" is null");
		HashMap userCards = getCards(credentials);
		if (userCards.containsKey(cardID))
			return (PCardEntity)userCards.get(cardID);
		else
			return null;
	}

	public List getPCards(IUserCredentials credentials) throws CardException {
		if (credentials == null)
			throw new CardException("Parameter \"credentials\" is null");
		HashMap userCards = getCards(credentials);
		ArrayList res = new ArrayList();
		Iterator itr = userCards.values().iterator();
		while (itr.hasNext()) {
			PCardEntity mc = (PCardEntity)itr.next();
			res.add(mc);
		}
		return res;
	}

	private HashMap getCards(IUserCredentials credentials) throws CardException {
		CacheKey cacheKey = buildCacheKey(credentials);
		HashMap cards = getPCardsFromCache(cacheKey);
		if (cards == null) {
			String userId = buildUserId(credentials);
			cards = getPCardsFromDB(userId);
			if (!isNoCache) {
				cardCache.put(cacheKey, cards);
			}
		}
		return cards;
	}

	private void updateCard(PCardEntity card, IUserCredentials credentials) throws CardException {
		Connection con = null;
		try {
			con = getConnection();
			PCardHelper.updatePCard(card, con);
			if (con.getAutoCommit() == false)
				con.commit();
		} catch (Exception e) {
			log.error(e, e);
			try {
				con.rollback();
			} catch (SQLException ex) {
				log.error(ex, ex);
				throw new CardException(ex);
			}
			throw new CardException(e);
		} finally {
			try {
				if (con != null) {
					con.close();
					con = null;
				}
			} catch (SQLException e) {
				log.error(e, e);
			}
		}
	}

	private HashMap getPCardsFromCache(CacheKey cacheKey) throws CardException {
		if (isNoCache) {
			if (!cardCache.isKeyInCache(cacheKey))
				return null;
			else
				return (HashMap) cardCache.get(cacheKey);
		}
		return null;
	}

	private HashMap getPCardsFromDB(String userId) throws CardException {
			String query = "SELECT * FROM PersonalCard WHERE userID_ = ? ORDER BY id_";
			HashMap cards = new HashMap();
			Connection con = null;
			PreparedStatement ps = null;
			ResultSet rs = null;
			try {
				con = getConnection();
				ps = con.prepareStatement(query);
				ps.setString(1, userId);
				rs = ps.executeQuery();
				while (rs.next()) {
					PCardEntity me = new PCardEntity();
					PCardHelper.initPCard(me, rs, con);
					cards.put(me.getCardID().toString(), me);
				}
				return cards;
			} catch (Exception e) {
				log.error(e, e);
				throw new CardException(e);
			} finally {
				try {
					if (rs != null) {
						rs.close();
						rs = null;
					}
				} catch (SQLException e) {
					log.error(e, e);
				}
				try {
					if (ps != null) {
						ps.close();
						ps = null;
					}
				} catch (SQLException e) {
					log.error(e, e);
				}
				try {
					if (con != null && !con.isClosed()) {
						con.close();
						con = null;
					}
				} catch (SQLException e) {
					log.error(e, e);
				}
			}
		}

		
		

	protected Connection getConnection() throws CardException, SQLException {
		if (!isConfigured) {
			throw new CardException("Managed bean was not configured");
		}
		return connectionFactory.getConnection();
	}

	private String buildUserId(IUserCredentials credentials) throws CardException {
		if (credentials instanceof UsernamePasswordCredentials) {
			UsernamePasswordCredentials upc = (UsernamePasswordCredentials)credentials;
			String userName = upc.getUsername();
			if (userName == null)
				throw new CardException("UsernamePasswordCredentials object contains null user name.");
			return userName.trim();
		}
		else
			throw new CardException();
	}

	private CacheKey buildCacheKey(IUserCredentials credentials) throws CardException {
		String cacheKey = null;
		if (credentials instanceof UsernamePasswordCredentials) {
			UsernamePasswordCredentials upc = (UsernamePasswordCredentials)credentials;
			String userName = upc.getUsername();
			cacheKey = userName.trim();
			return new CacheKey(cacheKey);
		}
		else
			throw new CardException();
	}

	public void init(Map config) throws CardException {
		isConfigured = false;
		connectionFactory = ConnectionFactory.buildConnectionFactory(config); 
		isConfigured = true;
	}
	
	public ISettingDescriptor getComponentDescriptor() {
		// TODO Auto-generated method stub
		return null;
	}

	public void persistPCard(PCardEntity card, IUserCredentials credentials) throws CardException {
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		if (credentials == null)
			throw new CardException("Parameter \"credentials\" is null");
		if (card.getId() == null) {
			addCard(card, credentials);
		}
		else {
			updateCard(card, credentials);
		}
	}

}
