/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.managed.db;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;

import javax.xml.transform.TransformerException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardExtensionEvent;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IClaimType;
import org.eclipse.higgins.icard.IEndpointReference;
import org.eclipse.higgins.icard.IExtensionListener;
import org.eclipse.higgins.icard.IInformationCardExtension;
import org.eclipse.higgins.icard.ITokenService;
import org.eclipse.higgins.icard.InvalidClaimException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.ReadOnlyObjectException;
import org.eclipse.higgins.icard.auth.ICredentialDescriptor;
import org.eclipse.higgins.icard.common.ClaimType;
import org.eclipse.higgins.icard.common.io.IOElement;
import org.eclipse.higgins.icard.common.utils.XMLHelper;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.CredentialDescriptor;
import org.eclipse.higgins.icard.provider.cardspace.common.EndpointReference;
import org.eclipse.higgins.icard.provider.cardspace.common.InformationCardExtension;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCard;
import org.eclipse.higgins.icard.provider.cardspace.common.STSPrivacyPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.TokenService;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.SelfIssuedCardClaims;
import org.eclipse.higgins.icard.provider.cardspace.db.ICardContext;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoInformationCardExtension;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoMCard;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoSupportedClaimType;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoSupportedTokenType;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoTokenService;
import org.w3c.dom.Element;

/**
 * Implementation of managed CardSpace-interoperable ICard stored within IdAS
 * context
 */
public class JDBCBasedManagedCard extends ManagedCard implements IExtensionListener {
	private Log log = LogFactory.getLog(JDBCBasedManagedCard.class);

	private IDaoMCard card_ = null;

	private boolean editMode = false;

	public JDBCBasedManagedCard() {
	}

	/**
	 * @param provider
	 * @param card
	 * @throws Exception
	 */
	public JDBCBasedManagedCard(JDBCBasedManagedCardProvider provider, IDaoMCard card) throws CardException {
		if (provider == null)
			throw new IllegalArgumentException("Parameter \"provider\" is null");
		if (card == null)
			throw new IllegalArgumentException("Parameter \"card\" is null");
		description_ = "IdAS-based CardSpace-interoperable managed I-Card";
		provider_ = provider;
		card_ = card;
		init();
	}

	/**
	 * @param provider
	 * @param account
	 * @param card
	 * @throws Exception
	 */
	public JDBCBasedManagedCard(JDBCBasedManagedCardProvider provider, ICardContext context, String userID, Element card) throws CardException {
		if (provider == null)
			throw new IllegalArgumentException("Parameter \"provider\" is null");
		if (card == null)
			throw new IllegalArgumentException("Parameter \"card\" is null");
		description_ = "IdAS-based CardSpace-interoperable managed I-Card";
		provider_ = provider;
		initFromXML(card);
		IDaoMCard oldCard = context.getMCard(id_.toString(), userID);
		if (oldCard != null)
			oldCard.setDeleteState();
		card_ = context.createMCard(userID);
		try {
			saveCard();

		} catch (IOException e) {
			log.error(e);
			throw new CardException(e);
		}
		if (oldCard != null)
			oldCard.store();
		card_.store();
	}

	/**
	 * @throws Exception
	 */
	public void init() throws CardException {
		initIsSelfIssued();
		initLanguage();
		initCardId();
		initCardVersion();
		initCardName();
		initCardImage();
		initCardImageType();
		initIssuer();
		initIssuerID();
		initIssuerName();
		initTimeIssued();
		initTimeExpires();
		initSupportedTokenTypes();
		initClaimTypes();
		initTokenServiceList();
		initRequireAppliesTo();
		initPrivacyNotice();
		initHashSalt();
		initTimeLastUpdated();
		initMasterKey();
		initExtensions();
		initStandardExtensionFields();
	}

	private void initIsSelfIssued() {
		isSelfIssued_ = false;
	}

	private void initLanguage() {
		language_ = card_.getLanguage();
	}

	private void initCardId() throws CardException {
		String id = card_.getCardID();
		try {
			id_ = new URI(id);
		} catch (URISyntaxException e) {
			log.error(e);
			throw new CardException(e);
		}
	}

	private void initCardVersion() {
		version_ = String.valueOf(card_.getVersion());
	}

	private void initCardName() {
		name_ = card_.getName();
	}

	private void initCardImage() {
		image_ = card_.getImage();
	}

	private void initCardImageType() {
		imageMimeType_ = card_.getImageType();
	}

	private void initIssuer() {
		issuer_ = card_.getIssuer();
	}

	private void initIssuerID() {
		issuerID_ = card_.getIssuerID();
	}

	private void initIssuerName() {
		issuerName_ = card_.getIssuerName();
	}

	private void initTimeIssued() {
		timeIssued_ = card_.getTimeIssued();
	}

	private void initTimeExpires() {
		timeExpires_ = card_.getTimeExpires();
	}

	private void initSupportedTokenTypes() throws CardException {
		supportedTokenTypes_ = new ArrayList();
		ArrayList lst = card_.getSupportedTokenTypeList();
		for (int i = 0, j = lst.size(); i < j; i++) {
			IDaoSupportedTokenType tt = (IDaoSupportedTokenType) lst.get(i);
			try {
				supportedTokenTypes_.add(new URI(tt.getType()));
			} catch (URISyntaxException e) {
				log.error(e);
				throw new CardException(e);
			}
		}
	}

	private void initClaimTypes() throws CardException {
		claimTypes_ = new ArrayList();
		ArrayList lst = card_.getSupportedClaimTypeList();
		for (int i = 0, j = lst.size(); i < j; i++) {
			IDaoSupportedClaimType ct = (IDaoSupportedClaimType) lst.get(i);
			String type = ct.getType();
			String name = ct.getDisplayName();
			String description = ct.getDescription();
			boolean isPPID = SelfIssuedCardClaims.PPID_TYPE.equals(type);
			ClaimType claimType = new ClaimType(type, name, description, !isPPID, !isPPID);
			claimTypes_.add(claimType);
		}
	}

	private void initTokenServiceList() throws CardException {
		tokenServices_ = new ArrayList();
		ArrayList lst = card_.getTokenServiceList();
		for (int i = 0, j = lst.size(); i < j; i++) {
			IDaoTokenService ts = (IDaoTokenService) lst.get(i);
			URI epAddress;
			try {
				epAddress = new URI(ts.getEndpointAddress());
			} catch (URISyntaxException e) {
				log.error(e);
				throw new CardException(e);
			}
			String epMetadata = ts.getMetadata();
			String epIdentity = ts.getIdentity();
			String ucHint = ts.getHint();
			String ucCredential = ts.getCredential();
			EndpointReference endpoint = new EndpointReference(epAddress, epMetadata, epIdentity);
			CredentialDescriptor credential;
			try {
				credential = new CredentialDescriptor(ucHint, ucCredential);
			} catch (Exception e) {
				log.error(e);
				throw new CardException(e);
			}
			TokenService tc = new TokenService(endpoint, credential);
			tokenServices_.add(tc);
		}
	}

	private void initRequireAppliesTo() {
		requireAppliesTo_ = card_.getRequireAppliesTo();
	}

	private void initPrivacyNotice() throws CardException {
		privacyNotice_ = null;
		String obj = card_.getPrivacyNotice();
		if (obj != null)
			privacyNotice_ = new STSPrivacyPolicy(obj);
	}

	private void initHashSalt() {
		hashSalt_ = card_.getHashSalt();
	}

	private void initTimeLastUpdated() {
		timeLastUpdated_ = card_.getTimeLastUpdated();
	}

	private void initMasterKey() {
		rawMasterKey_ = card_.getMasterKey();
	}

	private void initExtensions() throws CardException {
		extensions_ = new ArrayList();
		ArrayList lst = card_.getExtensions();
		for (int i = 0, j = lst.size(); i < j; i++) {
			IDaoInformationCardExtension daoExt = (IDaoInformationCardExtension) lst.get(i);
			String elmStr = daoExt.getElement();
			IOElement elm = new IOElement();
			try {
				elm.set(elmStr);
			} catch (Exception e) {
				throw new CardException(e);
			}
			boolean enabled = daoExt.isEnabled();
			InformationCardExtension icExt = new InformationCardExtension(elm, enabled, daoExt);
			icExt.addListener(this);
			extensions_.add(icExt);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setName(java.lang.String)
	 */
	public void setName(String name) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		String oldName = name_;
		try {
			name_ = (name != null) ? name.trim() : "";
			saveCardName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			name_ = oldName;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setImage(byte[], java.lang.String)
	 */
	public void setImage(byte[] image, String imageMimeType) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		byte[] oldImage = image_;
		String oldImageType = imageMimeType_;
		image_ = image;
		imageMimeType_ = imageMimeType;
		try {
			saveCardImage();
			saveCardImageType();
			saveTimeLastUpdated();
		} catch (Exception e) {
			image_ = oldImage;
			imageMimeType_ = oldImageType;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setIssuerName(java.lang.String)
	 */
	public void setIssuerName(String name) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (name == null || name.trim().length() == 0)
			throw new CardException("Couldn't set empty card issuer name.");
		String oldName = issuerName_;
		try {
			issuerName_ = name.trim();
			saveIssuerName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			issuerName_ = oldName;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setExpiredTime(java.util.Date)
	 */
	public void setTimeExpires(Date date) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		Date oldDate_ = timeExpires_;
		try {
			timeExpires_ = date;
			saveTimeExpires();
			saveTimeLastUpdated();
		} catch (Exception e) {
			timeExpires_ = oldDate_;
			log.error(e);
			throw new CardException(e);
		}
	}

	/**
	 * @throws TransformerException
	 * @throws Exception
	 */
	private void saveCard() throws CardException, IOException {
		saveIsSelfIssued();
		saveLanguage();
		saveCardId();
		saveCardVersion();
		saveCardName();
		saveCardImage();
		saveCardImageType();
		saveIssuer();
		saveIssuerID();
		saveIssuerName();
		saveTimeIssued();
		saveTimeExpires();
		saveSupportedTokenTypes();
		saveClaimTypes();
		saveTokenServiceList();
		saveRequireAppliesTo();
		savePrivacyNotice();
		saveHashSalt();
		saveTimeLastUpdated();
		saveMasterKey();
		saveExtensions();
	}

	private void saveIsSelfIssued() {
	}

	private void saveLanguage() throws CardException {
		card_.setLanguage(language_);
	}

	private void saveCardId() throws CardException {
		card_.setCardID(id_.toString());
	}

	private void saveCardVersion() throws CardException {
		card_.setVersion(Integer.parseInt(version_));
	}

	private void saveCardName() throws CardException {
		card_.setName(name_);
	}

	private void saveCardImage() throws CardException {
		card_.setImage(image_);
	}

	private void saveCardImageType() throws CardException {
		card_.setImageType(imageMimeType_);
	}

	private void saveIssuer() throws CardException {
		card_.setIssuer(issuer_);
	}

	private void saveIssuerID() throws CardException {
		card_.setIssuerID(issuerID_);
	}

	private void saveIssuerName() throws CardException {
		card_.setIssuerName(issuerName_);
	}

	private void saveTimeIssued() throws CardException {
		card_.setTimeIssued(new java.sql.Date(timeIssued_.getTime()));
	}

	private void saveTimeExpires() throws CardException {
		java.sql.Date dt = null;
		if (timeExpires_ != null) {
			dt = new java.sql.Date(timeExpires_.getTime());
		}
		card_.setTimeExpires(dt);
	}

	private void saveSupportedTokenTypes() throws CardException {
		Iterator it = supportedTokenTypes_.iterator();
		while (it.hasNext()) {
			IDaoSupportedTokenType tt = card_.createSupportedTokenType();
			URI uri = (URI) it.next();
			tt.setType(uri.toString());
			card_.addSupportedTokenType(tt);
		}
	}

	private void saveClaimTypes() throws CardException {
		Iterator it = claimTypes_.iterator();
		while (it.hasNext()) {
			IClaimType claimType = (IClaimType) it.next();
			IDaoSupportedClaimType ct = card_.createSupportedClaimType();
			ct.setType(claimType.getType());
			ct.setDescription(claimType.getDescription());
			ct.setDisplayName(claimType.getDisplayName());
			ct.setTypeLocalName(claimType.getTypeLocalName());
			card_.addSupportedClaimType(ct);
		}
	}

	private void saveTokenServiceList() throws CardException, IOException {
		Iterator it = tokenServices_.iterator();
		while (it.hasNext()) {
			ITokenService tokenService = (ITokenService) it.next();
			IDaoTokenService ts = card_.createTokenService();
			IEndpointReference er = tokenService.getEndpointReference();
			if (er != null) {
				// String metadata = XMLUtils.elementToString(er.getMetadata());
				String metadata = XMLHelper.toString(er.getMetadata());
				// String identity = XMLUtils.elementToString(er.getIdentity());
				String identity = XMLHelper.toString(er.getIdentity());
				ts.setEndpointAddress(er.getAddress().toString());
				ts.setIdentity(identity);
				ts.setMetadata(metadata);
			} else
				throw new CardException("TokenService contains null EndpointReference.");
			ICredentialDescriptor uc = tokenService.getUserCredential();
			if (uc != null) {
				// String credential = XMLUtils.elementToString(uc.asXML());
				String credential = XMLHelper.toString(uc.asXML());
				ts.setCredential(credential);
				ts.setHint(uc.getDisplayCredentialHint());
			} else
				throw new CardException("TokenService contains null UserCredential.");
			card_.addTokenService(ts);
		}
	}

	/**
	 * @throws IdASException
	 */
	private void saveRequireAppliesTo() throws CardException {
		card_.setRequireAppliesTo(requireAppliesTo_);
	}

	/**
	 * @throws IdASException
	 * @throws TransformerException
	 */
	private void savePrivacyNotice() throws CardException, IOException {
		String val = null;
		if (privacyNotice_ != null) {
			IElement elm = privacyNotice_.getPrivacyElement();
			if (elm != null) {
				try {
					val = (String) elm.getAs(String.class);
				} catch (Exception e) {
					throw new CardException(e);
				}
			}
		}
		card_.setPrivacyNotice(val);
	}

	/**
	 * @throws IdASException
	 */
	private void saveHashSalt() throws CardException {
		card_.setHashSalt(hashSalt_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeLastUpdated() throws CardException {
		timeLastUpdated_ = new Date();
		card_.setTimeLastUpdated(new java.sql.Date(timeLastUpdated_.getTime()));
	}

	/**
	 * @throws IdASException
	 */
	private void saveMasterKey() throws CardException {
		card_.setMasterKey(rawMasterKey_);
	}

	/**
	 * @throws CardException
	 */
	private void saveExtensions() throws CardException {
		for (int i = 0, j = extensions_.size(); i < j; i++) {
			IInformationCardExtension extension = (IInformationCardExtension) extensions_.get(i);
			Object id = extension.getId();
			if (id == null) { // new extension
				IDaoInformationCardExtension icExt = card_.createInformationCardExtension();
				IElement elm = extension.getElement();
				boolean enabled = extension.isEnabled();
				String data = null;
				try {
					data = (String) elm.getAs(String.class);
				} catch (Exception e) {
					throw new CardException(e);
				}
				icExt.setEnabled(enabled);
				icExt.setElement(data);
				card_.addInformationCardExtension(icExt);
				extension.setId(icExt);
				extension.addListener(this);
			}
		}
	}

	public void applyUpdates() throws InvalidStateException, CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		else {
			card_.store();
			editMode = false;
		}
	}

	public void beginUpdates() throws InvalidStateException {
		if (editMode) {
			throw new InvalidStateException("Already in edit mode.");
		} else {
			editMode = true;
		}
	}

	public void cancelUpdates() throws InvalidStateException {
		if (!editMode) {
			throw new InvalidStateException("Not in edit mode.");
		} else {
			editMode = false;
		}
	}

	public boolean isEditMode() {
		return editMode;
	}

	public boolean isEditable() {
		return false;
	}

	public IClaim createClaim(String type) throws InvalidTypeException, ReadOnlyObjectException {
		throw new ReadOnlyObjectException();
	}

	public IClaim setClaim(IClaim copyFrom) throws InvalidClaimException, InvalidTypeException, ReadOnlyObjectException {
		throw new ReadOnlyObjectException();
	}

	public boolean isClaimsRetrieved() {
		// TODO Need to implement!!!!
		return false;
	}

	public IInformationCardExtension addExtension(IElement extension) throws CardException {
		if (extension == null)
			throw new CardException("Parameter \"extension\" is null");
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		IDaoInformationCardExtension icExt = card_.createInformationCardExtension();
		boolean enabled = true;
		String data = null;
		try {
			data = (String) extension.getAs(String.class);
		} catch (Exception e) {
			throw new CardException(e);
		}
		icExt.setElement(data);
		icExt.setEnabled(enabled);
		card_.addInformationCardExtension(icExt);
		InformationCardExtension ext = new InformationCardExtension(extension, enabled, icExt);
		ext.addListener(this);
		extensions_.add(ext);
		return ext;
	}

	public void removeExtension(IInformationCardExtension extension) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		if (extensions_.contains(extension)) {
			Object id = extension.getId();
			if (id instanceof IDaoInformationCardExtension) {
				IDaoInformationCardExtension icExt = (IDaoInformationCardExtension) id;
				icExt.setDeleteState();
			}
			extensions_.remove(extension);
		} else
			throw new CardException("Could not remove extension because the card does not contain this extension");
	}

	public void extensionElementChanged(ICardExtensionEvent event) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		IInformationCardExtension icExt = event.getExtension();
		IElement iElement = icExt.getElement();
		String newElement;
		try {
			newElement = (String) iElement.getAs(String.class);
		} catch (Exception e) {
			throw new CardException(e);
		}
		Object id = icExt.getId();
		if (id instanceof IDaoInformationCardExtension) {
			IDaoInformationCardExtension daoExt = (IDaoInformationCardExtension) id;
			String oldElement = daoExt.getElement();
			if (oldElement == null || !oldElement.equals(newElement))
				daoExt.setElement(newElement);
		} else
			throw new CardException("IInformationCardExtension.getId() should return an instnace of IDaoInformationCardExtension");
	}

	public void extensionEnabledChanged(ICardExtensionEvent event) throws CardException {
		if (!editMode)
			throw new InvalidStateException("Not in edit mode.");
		IInformationCardExtension icExt = event.getExtension();
		boolean newEnabled = icExt.isEnabled();
		Object id = icExt.getId();
		if (id instanceof IDaoInformationCardExtension) {
			IDaoInformationCardExtension daoExt = (IDaoInformationCardExtension) id;
			boolean oldEnabled = daoExt.isEnabled();
			if (newEnabled != oldEnabled)
				daoExt.setEnabled(newEnabled);
		} else
			throw new CardException("IInformationCardExtension.getId() should return an instnace of IDaoInformationCardExtension");
	}

	public byte[] getMasterKey() throws CardException {
		// TODO implement pin protection logic
		return rawMasterKey_;
	}

}
