/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.managed.db;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.api.IConfigurableComponent;
import org.eclipse.higgins.configuration.api.ISettingDescriptor;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.common.utils.XMLHelper;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.CRDData;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCardProvider;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.provider.cardspace.db.ICardContext;
import org.eclipse.higgins.icard.provider.cardspace.db.IDaoMCard;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.registry.IConfiguration;
import org.eclipse.higgins.registry.MapConfiguraton;
import org.eclipse.higgins.registry.RegistryConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Implementation of managed CardSpace-interoperable ICard provider based on
 * IdAS context
 */
public class JDBCBasedManagedCardProvider extends ManagedCardProvider implements ICardProvider, IConfigurableComponent {
	private Log log = LogFactory.getLog(JDBCBasedManagedCardProvider.class);

	private String id_ = "org.eclipse.higgins.icard.provider.cardspace.managed.db";

	private String name_ = "CardSpace-compatable JDBC-based managed I-Card Provider Plug-in";

	private IConfiguration config = null;

	private ICardContext cc_ = null;

	private class UserCredentials {
		private String name_;

		public UserCredentials(String name) {
			name_ = name;
		}

		public String getName() {
			return name_;
		}
	}

	/**
	 * @return
	 */
	private HashMap convertConfigurationToMap(IConfiguration config) {
		HashMap ht = new HashMap();
		Iterator itr = config.getPropertyNames();
		while (itr.hasNext()) {
			String key = itr.next().toString();
			Object value = config.getProperty(key);
			ht.put(key, value);
		}
		return ht;
	}

	/**
	 * @param config
	 * @throws Exception
	 */
	protected void init(Map config) throws Exception {
		String cardContextClassName = (String) config.get("cardcontext.classname");
		if (cardContextClassName == null)
			throw new CardException("Can not find \"cardcontext.classname\" configuration property");
		Class cls = Class.forName(cardContextClassName);
		ICardContext cc = (ICardContext) cls.newInstance();
		super.init(config);
		cc.setPCardSupport(false);
		cc.init(config);
		cc_ = cc;
	}

	/**
	 * Initialize provider from registry configuration
	 * 
	 * @throws CardException
	 */
	private synchronized void initFromRegistryConfig() throws CardException {
		if (cc_ == null) {
			try {
				log.info("Initializing provider " + getID() + " from registry configuration.");
				HashMap configMap = convertConfigurationToMap(getConfiguration());
				init(configMap);
				log.info("Provider " + getID() + " was initialized from registry configuration.");
			} catch (Exception e) {
				log.error(e, e);
				throw new CardException(e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.configuration.api.IConfigurableComponent#configure
	 * (java.util.Map, java.lang.String, java.util.Map,
	 * org.eclipse.higgins.configuration.api.ISettingDescriptor,
	 * org.eclipse.higgins.configuration.api.ISettingDescriptor)
	 */
	public void configure(Map mapGlobalSettings, String strComponentName, Map mapComponentSettings,
			ISettingDescriptor componentDescriptor, ISettingDescriptor globalDescriptor) throws Exception {
		log.info("Initializing provider " + getID() + " from higgins configuration as IConfigurableComponent.");
		config = new MapConfiguraton(mapComponentSettings);
		init(mapComponentSettings);
		log.info("Provider " + getID() + " from higgins configuration as IConfigurableComponent.");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.higgins.configuration.api.IConfigurableComponent#
	 * getComponentDescriptor()
	 */
	public ISettingDescriptor getComponentDescriptor() {
		return null;
	}

	private ICardContext getCardContext() throws CardException {
		if (cc_ == null) {

			initFromRegistryConfig();
		}
		return cc_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.registry.IServiceProvider#setID(java.lang.String)
	 */
	public void setID(String id) throws Exception {
		id_ = id;
	}

	// public Iterator getICards(CallbackHandler authHandler, IPolicy policy)
	// throws CardException {
	public Iterator getICards(CallbackHandler authHandler) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		List cards = new ArrayList();
		UserCredentials userCrds = authenticate(authHandler);
		ArrayList lst = getCardContext().getMCards(userCrds.getName());
		for (int i = 0, j = lst.size(); i < j; i++) {
			IDaoMCard crd = (IDaoMCard) lst.get(i);
			JDBCBasedManagedCard card = new JDBCBasedManagedCard(this, crd);
			// if (policy == null) {
			cards.add(card);
			// } else if (policy.isSatisfiedBy(card)) {
			// cards.add(card);
			// }
		}
		return cards.iterator();
	}

	public ICard importCard(CallbackHandler authHandler, String filename) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (filename == null)
			throw new CardException("Parameter \"filename\" is null");
		FileInputStream fis;
		try {
			fis = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		// TODO need to separate import of different file types
		if (filename.toUpperCase().endsWith(".CRDS")) {
			importFromCrdsFile(authHandler, fis);
			return null;
		} else if (filename.toUpperCase().endsWith(".CRD"))
			return importFromCrdFile(authHandler, fis);
		else
			throw new CardException("Unsupported file extension.");
	}

	private ICard importFromCrdFile(CallbackHandler authHandler, InputStream is) throws CardException {
		try {
			Element elm = XMLHelper.domFromStream(is);
			CRDData crdData = CardCryptography.getCardFromSignedEnvelop(elm);
			IElement iElm = crdData.getCard();
			Element card = (Element)iElm.getAs(Element.class);
			UserCredentials userCrds = authenticate(authHandler);
			String userID = userCrds.getName();
			ICardContext cc = getCardContext();
			JDBCBasedManagedCard crd = new JDBCBasedManagedCard(this, cc, userID, card);
			return crd;
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
	}

	private void importFromCrdsFile(CallbackHandler authHandler, InputStream is) throws CardException {
		Document cardBackup;
		PasswordCallback pc = new PasswordCallback("Enter password for imported backup file: ", false);
		Callback[] callbacks = new Callback[] { pc };
		try {
			authHandler.handle(callbacks);
			String password = new String(pc.getPassword());
			pc.clearPassword();
			cardBackup = CardCryptography.decrypt(is, password);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		List cardsList = new ArrayList();
		NodeList newCards = cardBackup.getDocumentElement().getChildNodes();
		UserCredentials userCrds = authenticate(authHandler);
		String userID = userCrds.getName();
		ICardContext cc = getCardContext();
		for (int i = 0, j = newCards.getLength(); i < j; i++) {
			Node nd = newCards.item(i);
			if (nd.getNodeType() == Node.ELEMENT_NODE) {
				Element elm = (Element) nd;
				try {
					JDBCBasedManagedCard crd = new JDBCBasedManagedCard(this, cc, userID, elm);
					cardsList.add(crd);
				} catch (Exception e) {
					log.error(e);
				}
			}
		}
	}

	public void deleteCard(CallbackHandler authHandler, ICard card) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		UserCredentials userCrds = authenticate(authHandler);
		String userID = userCrds.getName();
		ICardContext cc = getCardContext();
		IDaoMCard crd = cc.getMCard(card.getID(), userID);
		if (crd != null) {
			crd.setDeleteState();
			crd.store();
		} else
			throw new CardException("Card not found.");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getID()
	 */
	public String getID() {
		return id_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getName()
	 */
	public String getName() {
		return name_;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.icard.ICardProvider#canCreateCard(javax.security.
	 * auth.callback.CallbackHandler, java.lang.String, java.util.Properties)
	 */
	public boolean canCreateCard(CallbackHandler authHandler, String id, Properties props) {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.icard.ICardProvider#createCard(javax.security.auth
	 * .callback.CallbackHandler, java.lang.String, java.util.Properties)
	 */
	public ICard createCard(CallbackHandler authHandler, String id, Properties props) throws CardException {
		throw new CardException("This provider can't create a card.");
	}

	public String getDescription() {
		return "IdAS-based CardSpace-interoperable managed I-Card provider";
	}

	/*
	 * public boolean canImportICard(CallbackHandler authHandler, IElement
	 * element) { if (element == null) return false; try { Element card =
	 * (Element) element.getAs(Element.class); if (card == null) { return false;
	 * } if (card == null) return false; Element crd = null; if
	 * ("Signature".equals(card.getLocalName())) crd =
	 * CardCryptography.getCardFromSignedEnvelop(card); else crd = card;
	 * JDBCBasedManagedCard c = new JDBCBasedManagedCard(); c.initFromXML(crd);
	 * return true; } catch (Exception e) { e.printStackTrace(); log.trace(e); }
	 * return false; }
	 */

	// public IElement[] exportICards(CallbackHandler authHandler,
	// IElementFormat format, CUID[] cards) throws CardException {
	// Document doc = cards.getOwnerDocument();
	// if (format instanceof IInformationCard) {
	// IInformationCard icard = (IInformationCard) format;
	// Element roamingCard = icard.toXML(doc);
	// cards.appendChild(roamingCard);
	// } else
	// throw new
	// CardException("Can not export card. This card is not an instance of IInformationCard.");
	// }
	public ICard importICard(CallbackHandler authHandler, IElement element) throws CardException {
		try {
			Element crd = (Element) element.getAs(Element.class);
//			Element crd = null;
//			if ("Signature".equals(card.getLocalName()))
//				crd = CardCryptography.getCardFromSignedEnvelop(card, true, true);
//			else
//				crd = card;
			UserCredentials userCrds = authenticate(authHandler);
			String userID = userCrds.getName();
			ICardContext cc = getCardContext();
			JDBCBasedManagedCard mc = new JDBCBasedManagedCard(this, cc, userID, crd);
			return mc;
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e);
		}
	}

	protected ICard getICardByID(CallbackHandler authHandler, String cardID) throws CardException {
		UserCredentials userCrds = authenticate(authHandler);
		String userID = userCrds.getName();
		ICardContext cc = getCardContext();
		IDaoMCard crd = cc.getMCard(cardID, userID);
		if (crd != null)
			return new JDBCBasedManagedCard(this, crd);
		else
			return null;
	}

	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException {
		if (this.getID().equals(cuid.getProviderID()) == false) {
			log.debug("ProviderID of requested CUID (" + cuid.getProviderID() + ") is not equal to this providerID ("
					+ this.getID() + ").");
			return null;
		}
		ICard crd = getICardByID(authHandler, cuid.getCardID());
		if (crd == null) {
			log.debug("Can not find any card with ID = " + cuid.getCardID());
			return null;
		} else {
			String issuer = cuid.getIssuer();
			if (issuer != null && issuer.equals(crd.getIssuer()))
				return crd;
			else {
				log.debug("Wrong issuer for card with ID = " + cuid.getCardID() + ". Requested issuer = " + issuer
						+ ", real issuer = " + crd.getIssuer());
				return null;
			}
		}
	}

	private UserCredentials authenticate(CallbackHandler authHandler) throws CardException {
		NameCallback nc = new NameCallback("User id: ");
		Callback[] callbacks = new Callback[] { nc };
		try {
			authHandler.handle(callbacks);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		String userName = nc.getName();
		if (userName == null || userName.trim().length() == 0)
			throw new CardException("Empty user name.");
		return new UserCredentials(userName);
	}

	public ICard createCard(CallbackHandler authHandler, ICardTemplate template) throws CardException {
		throw new CardException("ICard provider " + name_ + " can not create a card.");
	}

	public IConfiguration getConfiguration() {
		if (config == null) {
			initConfiguration();
		}
		return config;
	}

	public synchronized void initConfiguration() {
		if (config == null) {
			try {
				config = ICardRegistry.getInstance().getConfiguration(this);
			} catch (RegistryConfigurationException e) {
				log.error(e, e);
			}
		}
	}

}