/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.managed;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.configuration.api.IConfigurableComponent;
import org.eclipse.higgins.configuration.api.ISettingDescriptor;
import org.eclipse.higgins.icard.CUID;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICard;
import org.eclipse.higgins.icard.ICardProvider;
import org.eclipse.higgins.icard.ICardTemplate;
import org.eclipse.higgins.icard.common.utils.XMLHelper;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.CRDData;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCardProvider;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.IMCardManagedBean;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.IUserCredentials;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.MCardEntity;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.UsernamePasswordCredentials;
import org.eclipse.higgins.icard.provider.cardspace.common.utils.CardCryptography;
import org.eclipse.higgins.icard.registry.ICardRegistry;
import org.eclipse.higgins.registry.IConfiguration;
import org.eclipse.higgins.registry.MapConfiguraton;
import org.eclipse.higgins.registry.RegistryConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Implementation of managed CardSpace-interoperable ICard provider based on
 * IdAS context
 */
public class CardProvider extends ManagedCardProvider implements ICardProvider, IConfigurableComponent {
	private static Log log = LogFactory.getLog(CardProvider.class);

	private String id = "org.eclipse.higgins.icard.provider.cardspace.managed.db";

	private String name = "CardSpace-compatable managed I-Card Provider Plug-in";

	private IConfiguration config = null;

	private IMCardManagedBean mbean = null;

	/**
	 * @return
	 */
	private HashMap convertConfigurationToMap(IConfiguration config) {
		HashMap ht = new HashMap();
		Iterator itr = config.getPropertyNames();
		while (itr.hasNext()) {
			String key = itr.next().toString();
			Object value = config.getProperty(key);
			ht.put(key, value);
		}
		return ht;
	}

	/**
	 * @param config
	 * @throws Exception
	 */
	protected void init(Map config) throws Exception {
		String managedBeanClassName = (String) config.get("managedbean.classname");
		if (managedBeanClassName == null)
			throw new CardException("Can not find \"managedbean.classname\" configuration property");
		Class cls = Class.forName(managedBeanClassName);
		super.init(config);
		mbean = (IMCardManagedBean)cls.newInstance();
		mbean.init(config);
	}

	/**
	 * Initialize provider from registry configuration
	 * 
	 * @throws CardException
	 */
	private synchronized void initFromRegistryConfig() throws CardException {
		if (mbean == null) {
			try {
				HashMap configMap = convertConfigurationToMap(getConfiguration());
				init(configMap);
				log.info("Provider " + getID() + " was initialized from registry configuration.");
			} catch (Exception e) {
				log.error(e, e);
				throw new CardException(e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.configuration.api.IConfigurableComponent#configure
	 * (java.util.Map, java.lang.String, java.util.Map,
	 * org.eclipse.higgins.configuration.api.ISettingDescriptor,
	 * org.eclipse.higgins.configuration.api.ISettingDescriptor)
	 */
	public void configure(Map mapGlobalSettings, String strComponentName, Map mapComponentSettings,
			ISettingDescriptor componentDescriptor, ISettingDescriptor globalDescriptor) throws Exception {
		log.info("Initializing provider " + getID() + " from higgins configuration as IConfigurableComponent.");
		config = new MapConfiguraton(mapComponentSettings);
		init(mapComponentSettings);
		log.info("Provider " + getID() + " from higgins configuration as IConfigurableComponent.");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.eclipse.higgins.configuration.api.IConfigurableComponent#
	 * getComponentDescriptor()
	 */
	public ISettingDescriptor getComponentDescriptor() {
		return null;
	}


	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.registry.IServiceProvider#setID(java.lang.String)
	 */
	public void setID(String id) throws Exception {
		this.id = id;
	}

	public IMCardManagedBean getManagedBean() throws CardException {
		if (mbean == null) {
			initFromRegistryConfig();
		}
		return mbean;
	}
	
	public Iterator getICards(CallbackHandler authHandler) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		List cards = new ArrayList();
		IUserCredentials userCrds = authenticate(authHandler);
		List lst = getManagedBean().getMCards(userCrds);
		for (int i = 0, j = lst.size(); i < j; i++) {
			MCardEntity crd = (MCardEntity) lst.get(i);
			MCard card = new MCard(this, userCrds, crd);
			cards.add(card);
		}
		return cards.iterator();
	}

	public ICard importCard(CallbackHandler authHandler, String filename) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (filename == null)
			throw new CardException("Parameter \"filename\" is null");
		FileInputStream fis;
		try {
			fis = new FileInputStream(filename);
		} catch (FileNotFoundException e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		// TODO need to separate import of different file types
		if (filename.toUpperCase().endsWith(".CRDS")) {
			importFromCrdsFile(authHandler, fis);
			return null;
		} else if (filename.toUpperCase().endsWith(".CRD"))
			return importFromCrdFile(authHandler, fis);
		else
			throw new CardException("Unsupported file extension.");
	}

	private ICard importFromCrdFile(CallbackHandler authHandler, InputStream is) throws CardException {
		try {
			Element elm = XMLHelper.domFromStream(is);
			CRDData crdData = CardCryptography.getCardFromSignedEnvelop(elm);
			IElement iElm = crdData.getCard();
			Element crd = (Element)iElm.getAs(Element.class);
			IUserCredentials userCrds = authenticate(authHandler);
			MCard mc = new MCard(this, userCrds, crd);
			return mc;
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e);
		}
	}

	private void importFromCrdsFile(CallbackHandler authHandler, InputStream is) throws CardException {
		Document cardBackup;
		PasswordCallback pc = new PasswordCallback("Enter password for imported backup file: ", false);
		Callback[] callbacks = new Callback[] { pc };
		try {
			authHandler.handle(callbacks);
			String password = new String(pc.getPassword());
			pc.clearPassword();
			cardBackup = CardCryptography.decrypt(is, password);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e.getMessage());
		}
		List cardsList = new ArrayList();
		NodeList newCards = cardBackup.getDocumentElement().getChildNodes();
		IUserCredentials userCrds = authenticate(authHandler);

		for (int i = 0, j = newCards.getLength(); i < j; i++) {
			Node nd = newCards.item(i);
			if (nd.getNodeType() == Node.ELEMENT_NODE) {
				Element elm = (Element) nd;
				try {
					MCard mc = new MCard(this, userCrds, elm);
					cardsList.add(mc);
				} catch (Exception e) {
					log.error(e);
				}
			}
		}
	}


	public void deleteCard(CallbackHandler authHandler, ICard card) throws CardException {
		if (authHandler == null)
			throw new CardException("Parameter \"authHandler\" is null");
		if (card == null)
			throw new CardException("Parameter \"card\" is null");
		IUserCredentials userCreds = authenticate(authHandler);
		getManagedBean().deleteMCard(card.getID(), userCreds);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getID()
	 */
	public String getID() {
		return id;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICardProvider#getName()
	 */
	public String getName() {
		return name;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.icard.ICardProvider#canCreateCard(javax.security.
	 * auth.callback.CallbackHandler, java.lang.String, java.util.Properties)
	 */
	public boolean canCreateCard(CallbackHandler authHandler, String id, Properties props) {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.higgins.icard.ICardProvider#createCard(javax.security.auth
	 * .callback.CallbackHandler, java.lang.String, java.util.Properties)
	 */
	public ICard createCard(CallbackHandler authHandler, String id, Properties props) throws CardException {
		throw new CardException("This provider can't create a card.");
	}

	public String getDescription() {
		return "IdAS-based CardSpace-interoperable managed I-Card provider";
	}

	public ICard importICard(CallbackHandler authHandler, IElement element) throws CardException {
		try {
			Element crd = (Element) element.getAs(Element.class);
			IUserCredentials userCrds = authenticate(authHandler);
			MCard mc = new MCard(this, userCrds, crd);
			return mc;
		} catch (CardException e) {
			throw e;
		} catch (Exception e) {
			log.error(e, e);
			throw new CardException(e);
		}
	}

	protected ICard getICardByID(CallbackHandler authHandler, String cardID) throws CardException {
		IUserCredentials userCreds = authenticate(authHandler);
		MCardEntity crd = getManagedBean().getMCard(cardID, userCreds);
		if (crd != null)
			return new MCard(this, userCreds, crd);
		else
			return null;
	}

	public ICard getICardByCUID(CallbackHandler authHandler, CUID cuid) throws CardException {
		if (this.getID().equals(cuid.getProviderID()) == false) {
			log.debug("ProviderID of requested CUID (" + cuid.getProviderID() + ") is not equal to this providerID ("
					+ this.getID() + ").");
			return null;
		}
		ICard crd = getICardByID(authHandler, cuid.getCardID());
		if (crd == null) {
			log.debug("Can not find any card with ID = " + cuid.getCardID());
			return null;
		} else {
			String issuer = cuid.getIssuer();
			if (issuer != null && issuer.equals(crd.getIssuer()))
				return crd;
			else {
				log.debug("Wrong issuer for card with ID = " + cuid.getCardID() + ". Requested issuer = " + issuer
						+ ", real issuer = " + crd.getIssuer());
				return null;
			}
		}
	}

	private IUserCredentials authenticate(CallbackHandler authHandler) throws CardException {
		NameCallback nc = new NameCallback("User id: ");
		PasswordCallback pc = new PasswordCallback("Password: ", false);
		Callback[] callbacks = new Callback[] { nc, pc };
		try {
			authHandler.handle(callbacks);
		} catch (Exception e) {
			log.error(e);
			throw new CardException(e);
		}
		String userName = nc.getName();
		if (userName == null || userName.trim().length() == 0)
			throw new CardException("Empty user name.");
		String pass = (pc != null && pc.getPassword() != null) ? new String(pc.getPassword()) : "";
		return new UsernamePasswordCredentials(userName, pass);
	}

	public ICard createCard(CallbackHandler authHandler, ICardTemplate template) throws CardException {
		throw new CardException("ICard provider " + name + " can not create a card.");
	}

	public IConfiguration getConfiguration() {
		if (config == null) {
			initConfiguration();
		}
		return config;
	}

	public synchronized void initConfiguration() {
		if (config == null) {
			try {
				config = ICardRegistry.getInstance().getConfiguration(this);
			} catch (RegistryConfigurationException e) {
				log.error(e, e);
			}
		}
	}

}