/**
 * Copyright (c) 2007 Parity Communications, Inc. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Sergey Lyakhov - initial API and implementation
 */

package org.eclipse.higgins.icard.provider.cardspace.managed;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.xml.transform.TransformerException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.higgins.icard.CardException;
import org.eclipse.higgins.icard.ICardExtensionEvent;
import org.eclipse.higgins.icard.IClaim;
import org.eclipse.higgins.icard.IExtensionListener;
import org.eclipse.higgins.icard.IInformationCardExtension;
import org.eclipse.higgins.icard.InvalidClaimException;
import org.eclipse.higgins.icard.InvalidStateException;
import org.eclipse.higgins.icard.InvalidTypeException;
import org.eclipse.higgins.icard.ReadOnlyObjectException;
import org.eclipse.higgins.icard.common.io.IOElement;
import org.eclipse.higgins.icard.io.IElement;
import org.eclipse.higgins.icard.provider.cardspace.common.InformationCardExtension;
import org.eclipse.higgins.icard.provider.cardspace.common.ManagedCard;
import org.eclipse.higgins.icard.provider.cardspace.common.STSPrivacyPolicy;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.Extension;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.IMCardManagedBean;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.IUserCredentials;
import org.eclipse.higgins.icard.provider.cardspace.common.entity.MCardEntity;
import org.w3c.dom.Element;

public class MCard extends ManagedCard implements IExtensionListener {
	private Log log = LogFactory.getLog(MCard.class);

	private MCardEntity card_ = null;

	private boolean editMode_ = false;
	
	IUserCredentials credentials_;

	public MCard() {
	}

	/**
	 * @param provider
	 * @param card
	 * @throws Exception
	 */
	public MCard(CardProvider provider, IUserCredentials credentials, MCardEntity card) throws CardException {
		if (provider == null)
			throw new IllegalArgumentException("Parameter \"provider\" is null");
		if (card == null)
			throw new IllegalArgumentException("Parameter \"card\" is null");
		description_ = "IdAS-based CardSpace-interoperable managed I-Card";
		provider_ = provider;
		credentials_ = credentials;
		card_ = card;
		init();
	}

	/**
	 * @param provider
	 * @param account
	 * @param card
	 * @throws Exception
	 */
	public MCard(CardProvider provider, IUserCredentials credentials, Element card) throws CardException {
		if (provider == null)
			throw new IllegalArgumentException("Parameter \"provider\" is null");
		if (card == null)
			throw new IllegalArgumentException("Parameter \"card\" is null");
		description_ = "IdAS-based CardSpace-interoperable managed I-Card";
		provider_ = provider;
		credentials_ = credentials;
		initFromXML(card);
		card_ = new MCardEntity();
		try {
			saveCard();
		} catch (IOException e) {
			log.error(e);
			throw new CardException(e);
		}
		IMCardManagedBean mbean = provider.getManagedBean();
		MCardEntity oldCard = mbean.getMCard(id_.toString(), credentials);
		if (oldCard != null) {
			mbean.deleteMCard(oldCard.getCardID().toString(), credentials);
		}
		mbean.persistMCard(card_, credentials);
	}

	/**
	 * @throws Exception
	 */
	public void init() throws CardException {
		initIsSelfIssued();
		initLanguage();
		initCardId();
		initCardVersion();
		initCardName();
		initCardImage();
		initCardImageType();
		initIssuer();
		initIssuerID();
		initIssuerName();
		initTimeIssued();
		initTimeExpires();
		initSupportedTokenTypes();
		initClaimTypes();
		initTokenServiceList();
		initRequireAppliesTo();
		initPrivacyNotice();
		initHashSalt();
		initTimeLastUpdated();
		initMasterKey();
		initExtensions();
		initStandardExtensionFields();
	}

	private void initIsSelfIssued() {
		isSelfIssued_ = false;
	}

	private void initLanguage() {
		language_ = card_.getLanguage();
	}

	private void initCardId() throws CardException {
		id_  = card_.getCardID();
	}

	private void initCardVersion() {
		version_ = String.valueOf(card_.getVersion());
	}

	private void initCardName() {
		name_ = card_.getName();
	}

	private void initCardImage() {
		image_ = card_.getImage();
	}

	private void initCardImageType() {
		imageMimeType_ = card_.getImageType();
	}

	private void initIssuer() {
		issuer_ = (card_.getIssuer() != null) ? card_.getIssuer().toString() : null;
	}

	private void initIssuerID() {
		issuerID_ = card_.getIssuerID();
	}

	private void initIssuerName() {
		issuerName_ = card_.getIssuerName();
	}

	private void initTimeIssued() {
		timeIssued_ = card_.getTimeIssued();
	}

	private void initTimeExpires() {
		timeExpires_ = card_.getTimeExpires();
	}

	private void initSupportedTokenTypes() throws CardException {
		supportedTokenTypes_ = card_.getSupportedTokenTypeList();
	}

	private void initClaimTypes() throws CardException {
		claimTypes_ = card_.getSupportedClaimTypeList();
	}

	private void initTokenServiceList() throws CardException {
		tokenServices_ = card_.getTokenServiceList();
	}

	private void initRequireAppliesTo() {
		requireAppliesTo_ = card_.getRequireAppliesTo();
	}

	private void initPrivacyNotice() throws CardException {
		privacyNotice_ = null;
		URI uri = card_.getPolicyURL();
		String version = (card_.getPolicyVersion() != null) ? card_.getPolicyVersion().toString() : null;
		if (uri != null) {
			privacyNotice_ = new STSPrivacyPolicy(uri.toString(), version);
		}
	}

	private void initHashSalt() {
		hashSalt_ = card_.getHashSalt();
	}

	private void initTimeLastUpdated() {
		timeLastUpdated_ = card_.getTimeLastUpdated();
	}

	private void initMasterKey() {
		rawMasterKey_ = card_.getMasterKey();
	}

	private void initExtensions() throws CardException {
		extensions_ = new ArrayList();
		List lst = card_.getExtensionList();
		for (int i = 0, j = lst.size(); i < j; i++) {
			Extension ext = (Extension)lst.get(i);
			String elmStr = ext.getExtensionData();
			IOElement elm = new IOElement();
			try {
				elm.set(elmStr);
			} catch (Exception e) {
				throw new CardException(e);
			}
			boolean enabled = ext.isEnabled();
			InformationCardExtension icExt = new InformationCardExtension(elm, enabled, ext);
			icExt.addListener(this);
			extensions_.add(icExt);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setName(java.lang.String)
	 */
	public void setName(String name) throws CardException {
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		String oldName = name_;
		try {
			name_ = (name != null) ? name.trim() : "";
			saveCardName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			name_ = oldName;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setImage(byte[], java.lang.String)
	 */
	public void setImage(byte[] image, String imageMimeType) throws CardException {
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		byte[] oldImage = image_;
		String oldImageType = imageMimeType_;
		image_ = image;
		imageMimeType_ = imageMimeType;
		try {
			saveCardImage();
			saveCardImageType();
			saveTimeLastUpdated();
		} catch (Exception e) {
			image_ = oldImage;
			imageMimeType_ = oldImageType;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setIssuerName(java.lang.String)
	 */
	public void setIssuerName(String name) throws CardException {
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		if (name == null || name.trim().length() == 0)
			throw new CardException("Couldn't set empty card issuer name.");
		String oldName = issuerName_;
		try {
			issuerName_ = name.trim();
			saveIssuerName();
			saveTimeLastUpdated();
		} catch (Exception e) {
			issuerName_ = oldName;
			log.error(e);
			throw new CardException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.higgins.icard.ICard#setExpiredTime(java.util.Date)
	 */
	public void setTimeExpires(Date date) throws CardException {
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		Date oldDate_ = timeExpires_;
		try {
			timeExpires_ = date;
			saveTimeExpires();
			saveTimeLastUpdated();
		} catch (Exception e) {
			timeExpires_ = oldDate_;
			log.error(e);
			throw new CardException(e);
		}
	}

	/**
	 * @throws TransformerException
	 * @throws Exception
	 */
	private void saveCard() throws CardException, IOException {
		saveIsSelfIssued();
		saveLanguage();
		saveCardId();
		saveCardVersion();
		saveCardName();
		saveCardImage();
		saveCardImageType();
		saveIssuer();
		saveIssuerID();
		saveIssuerName();
		saveTimeIssued();
		saveTimeExpires();
		saveSupportedTokenTypes();
		saveClaimTypes();
		saveTokenServiceList();
		saveRequireAppliesTo();
		savePrivacyNotice();
		saveHashSalt();
		saveTimeLastUpdated();
		saveMasterKey();
		saveExtensions();
	}

	private void saveIsSelfIssued() {
	}

	private void saveLanguage() throws CardException {
		card_.setLanguage(language_);
	}

	private void saveCardId() throws CardException {
		card_.setCardID(id_);
	}

	private void saveCardVersion() throws CardException {
		card_.setVersion(Integer.parseInt(version_));
	}

	private void saveCardName() throws CardException {
		card_.setName(name_);
	}

	private void saveCardImage() throws CardException {
		card_.setImage(image_);
	}

	private void saveCardImageType() throws CardException {
		card_.setImageType(imageMimeType_);
	}

	private void saveIssuer() throws CardException {
		URI uri;
		try {
			uri = (issuer_ != null) ? new URI (issuer_) : null;
		} catch (URISyntaxException e) {
			log.error(e, e);
			throw new CardException(e);
		}
		card_.setIssuer(uri);
	}

	private void saveIssuerID() throws CardException {
		card_.setIssuerID(issuerID_);
	}

	private void saveIssuerName() throws CardException {
		card_.setIssuerName(issuerName_);
	}

	private void saveTimeIssued() throws CardException {
		card_.setTimeIssued(new java.sql.Date(timeIssued_.getTime()));
	}

	private void saveTimeExpires() throws CardException {
		java.sql.Date dt = null;
		if (timeExpires_ != null) {
			dt = new java.sql.Date(timeExpires_.getTime());
		}
		card_.setTimeExpires(dt);
	}

	private void saveSupportedTokenTypes() throws CardException {
		card_.setSupportedTokenTypeList(supportedTokenTypes_);
	}

	private void saveClaimTypes() throws CardException {
		card_.setSupportedClaimTypeList(claimTypes_);
	}

	private void saveTokenServiceList() throws CardException, IOException {
		card_.setTokenServiceList(tokenServices_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveRequireAppliesTo() throws CardException {
		card_.setRequireAppliesTo(requireAppliesTo_);
	}

	/**
	 * @throws IdASException
	 * @throws TransformerException
	 */
	private void savePrivacyNotice() throws CardException, IOException {
		String val = null;
		if (privacyNotice_ != null) {
			String url = privacyNotice_.getPrivacyUrl();
			String versionStr = privacyNotice_.getPrivacyVersion();
			if (url != null) {
				try {
					URI uri = new URI(url);
					card_.setPolicyURL(uri);
					if (versionStr != null) {
						Integer version = Integer.decode(versionStr);
						card_.setPolicyVersion(version);
					}
				} catch (Exception e) {
					throw new CardException(e);
				}
			}
		}
	}

	/**
	 * @throws IdASException
	 */
	private void saveHashSalt() throws CardException {
		card_.setHashSalt(hashSalt_);
	}

	/**
	 * @throws IdASException
	 */
	private void saveTimeLastUpdated() throws CardException {
		timeLastUpdated_ = new Date();
		card_.setTimeLastUpdated(new java.sql.Date(timeLastUpdated_.getTime()));
	}

	/**
	 * @throws IdASException
	 */
	private void saveMasterKey() throws CardException {
		card_.setMasterKey(rawMasterKey_);
	}

	/**
	 * @throws CardException
	 */
	private void saveExtensions() throws CardException {
		ArrayList extList = new ArrayList();
		for (int i = 0, j = extensions_.size(); i < j; i++) {
			IInformationCardExtension extension = (IInformationCardExtension) extensions_.get(i);
			Object id = extension.getId();
			if (id == null) { // new extension
				Extension icExt = new Extension();
				IElement elm = extension.getElement();
				boolean enabled = extension.isEnabled();
				String data = null;
				try {
					data = (String) elm.getAs(String.class);
				} catch (Exception e) {
					throw new CardException(e);
				}
				icExt.setEnabled(enabled);
				icExt.setExtensionData(data);
				extList.add(icExt);
				extension.setId(icExt);
				extension.addListener(this);
			}
		}
		card_.setExtensionList(extList);
	}

	public void applyUpdates() throws InvalidStateException, CardException {
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		else {
			IMCardManagedBean mbean = ((CardProvider)provider_).getManagedBean();
			mbean.persistMCard(card_, credentials_);
			editMode_ = false;
		}
	}

	public void beginUpdates() throws InvalidStateException {
		if (editMode_) {
			throw new InvalidStateException("Already in edit mode.");
		} else {
			editMode_ = true;
		}
	}

	public void cancelUpdates() throws InvalidStateException {
		if (!editMode_) {
			throw new InvalidStateException("Not in edit mode.");
		} else {
			editMode_ = false;
		}
	}

	public boolean isEditMode() {
		return editMode_;
	}

	public boolean isEditable() {
		return false;
	}

	public IClaim createClaim(String type) throws InvalidTypeException, ReadOnlyObjectException {
		throw new ReadOnlyObjectException();
	}

	public IClaim setClaim(IClaim copyFrom) throws InvalidClaimException, InvalidTypeException, ReadOnlyObjectException {
		throw new ReadOnlyObjectException();
	}

	public boolean isClaimsRetrieved() {
		// TODO Need to implement!!!!
		return false;
	}

	public IInformationCardExtension addExtension(IElement extension) throws CardException {
		if (extension == null)
			throw new CardException("Parameter \"extension\" is null");
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		Extension icExt = new Extension();
		boolean enabled = true;
		String data = null;
		try {
			data = (String) extension.getAs(String.class);
		} catch (Exception e) {
			throw new CardException(e);
		}
		icExt.setExtensionData(data);
		icExt.setEnabled(enabled);
		card_.getExtensionList().add(icExt);
		InformationCardExtension ext = new InformationCardExtension(extension, enabled, icExt);
		ext.addListener(this);
		extensions_.add(ext);
		return ext;
	}

	public void removeExtension(IInformationCardExtension extension) throws CardException {
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		if (extensions_.contains(extension)) {
			Object id = extension.getId();
			if (id instanceof Extension) {
				card_.getExtensionList().remove(id);
				extensions_.remove(extension);
			}
			else
				throw new CardException("Could not remove extension.");
		} else
			throw new CardException("Could not remove extension because the card does not contain this extension.");
	}

	public void extensionElementChanged(ICardExtensionEvent event) throws CardException {
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		IInformationCardExtension icExt = event.getExtension();
		IElement iElement = icExt.getElement();
		String newElement;
		try {
			newElement = (String) iElement.getAs(String.class);
		} catch (Exception e) {
			throw new CardException(e);
		}
		Object id = icExt.getId();
		if (id instanceof Extension) {
			Extension ext = (Extension) id;
			String oldElement = ext.getExtensionData();
			if (oldElement == null || !oldElement.equals(newElement))
				ext.setExtensionData(newElement);
		} else
			throw new CardException("IInformationCardExtension.getId() should return an instnace of IDaoInformationCardExtension");
	}

	public void extensionEnabledChanged(ICardExtensionEvent event) throws CardException {
		if (!editMode_)
			throw new InvalidStateException("Not in edit mode.");
		IInformationCardExtension icExt = event.getExtension();
		boolean newEnabled = icExt.isEnabled();
		Object id = icExt.getId();
		if (id instanceof Extension) {
			Extension ext = (Extension) id;
			boolean oldEnabled = ext.isEnabled();
			if (newEnabled != oldEnabled)
				ext.setEnabled(newEnabled);
		} else
			throw new CardException("IInformationCardExtension.getId() should return an instnace of IDaoInformationCardExtension");
	}

	public byte[] getMasterKey() throws CardException {
		// TODO implement pin protection logic
		return rawMasterKey_;
	}

}
